package com.xforceplus.ultraman.oqsengine.sdk.service.export.impl;

import cn.hutool.core.collection.CollectionUtil;
import com.xforceplus.tech.base.core.context.ContextKeys;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.impl.ColumnField;
import com.xforceplus.ultraman.oqsengine.sdk.command.ImportCmd;
import com.xforceplus.ultraman.oqsengine.sdk.event.EntityErrorImported;
import com.xforceplus.ultraman.oqsengine.sdk.event.EntityImported;
import com.xforceplus.ultraman.oqsengine.sdk.facade.EntityFacade;
import com.xforceplus.ultraman.oqsengine.sdk.facade.result.CreateOneResult;
import com.xforceplus.ultraman.oqsengine.sdk.facade.result.ResultStatus;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.ImportService;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.PostImportAware;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.entity.ImportResult;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.enums.ImportModeEnum;
import com.xforceplus.ultraman.oqsengine.sdk.store.engine.IEntityClassGroup;
import com.xforceplus.ultraman.oqsengine.sdk.transactional.DefaultTransactionManager;
import com.xforceplus.ultraman.oqsengine.sdk.transactional.OqsTransaction;
import com.xforceplus.ultraman.oqsengine.sdk.transactional.OqsTransactionManager;
import com.xforceplus.ultraman.oqsengine.sdk.util.CompletableFutureUtils;
import io.vavr.control.Either;
import io.vavr.control.Validation;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Comment;
import org.apache.poi.ss.usermodel.RichTextString;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.xssf.usermodel.XSSFRow;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationEventPublisher;

import java.io.IOException;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.stream.Collectors;

/**
 *
 */
public class ImportDefaultExcelServiceImpl implements ImportService {
    private final Logger logger = LoggerFactory.getLogger(ImportService.class);

    protected EntityFacade entityFacade;

    protected ContextService contextService;

    protected ExecutorService importThreadPool;

    protected OqsTransactionManager manager;

    protected ApplicationEventPublisher publisher;

    @Autowired(required = false)
    List<PostImportAware> postImportAwareList = new ArrayList<>();

    private int defaultStep = 1000;

    public ImportDefaultExcelServiceImpl(
            EntityFacade entityFacade
            , ContextService contextService
            , ExecutorService importThreadPool
            , OqsTransactionManager manager
            , ApplicationEventPublisher publisher
    ) {
        this.entityFacade = entityFacade;
        this.contextService = contextService;
        this.importThreadPool = importThreadPool;
        this.manager = manager;
        this.publisher = publisher;
    }

    @Override
    public ImportModeEnum getImportMode() {
        return ImportModeEnum.BASE;
    }

    @Override
    public ImportResult doImport(IEntityClassGroup entityClassGroup, ImportCmd cmd) {
        int step = cmd.getStep();
        if (step <= 0) {
            step = defaultStep;
        }

        XSSFWorkbook xssfwb = null;
        try {
            xssfwb = new XSSFWorkbook(cmd.getInputStream());
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        IEntityClass entityClass = entityClassGroup.getEntityClass();
        int numberOfSheets = xssfwb.getNumberOfSheets();
        try {
            Map<String, Object> contextRelatedHeaderMap = Optional.ofNullable(contextService)
                    .map(ContextService::getAll)
                    .orElseGet(Collections::emptyMap);

            contextRelatedHeaderMap.put("index", new HashMap());


            //do 1 to n insert
            XSSFSheet mainSheet = xssfwb.getSheetAt(0);

            if (mainSheet == null) {
                throw new RuntimeException("main sheet is null");
            }
            List<String> relatedCode = new ArrayList<>();
            if (numberOfSheets > 1) {
                for (int i = 1; i < numberOfSheets; i++) {
                    XSSFSheet sheetAt = xssfwb.getSheetAt(i);
                    prepareRelatedHeader(sheetAt, contextRelatedHeaderMap, entityClassGroup);
                    relatedCode.add(sheetAt.getSheetName());
                }
            }

            /**
             * future one
             */
            List<CompletableFuture<Either<CreateOneResult, Long>>> futures = new ArrayList<>();

            Iterator<Row> rowIterator = mainSheet.rowIterator();

            Map<String, Object> rowContentMap = new HashMap<>();
            Row headerRow = null;
            boolean isFirstRow = true;
            short lastCellNum = 0;
            List<ImportCmd.Sheet.FieldMapping> headerFiledMapping = new ArrayList<>();
            if (CollectionUtil.isNotEmpty(cmd.getSheets())) {
                Optional<ImportCmd.Sheet> sheetHeaderOptional = cmd.getSheets().stream().filter(sheetItem -> mainSheet.getSheetName().equals(sheetItem.getSheet())).findFirst();
                headerFiledMapping = sheetHeaderOptional.isPresent() ? sheetHeaderOptional.get().getFieldMapping() : new ArrayList();
            }
            List<String> headerFiledCodeList = new ArrayList();
            int currentBatchCount = 0;

            //start transaction
            OqsTransaction newTransaction = ((DefaultTransactionManager) manager).createNewTransaction(cmd.getTimeout(), "");
            contextService.set(ContextKeys.StringKeys.TRANSACTION_KEY, newTransaction.getId());

            while (rowIterator.hasNext()) {
                Row row = rowIterator.next();
                if (isFirstRow) {
                    headerRow = row;
                    isFirstRow = false;
                    lastCellNum = headerRow.getLastCellNum();
                    for (int j = 0; j < lastCellNum; j++) {
                        if (headerFiledMapping.size() > 0 && j < headerFiledMapping.size()) {
                            headerFiledCodeList.add(getBoFieldCodeFromExcelHeader(headerRow.getCell(j).getStringCellValue(), j, headerFiledMapping, entityClass.code()));
                        } else {
                            headerFiledCodeList.add(getBoFieldCodeFromExcelHeader(headerRow.getCell(j).getStringCellValue(), entityClass));
                        }
                    }
                    continue;
                }

                rowContentMap.clear();
                for (int i = 0; i < lastCellNum; i++) {
                    Cell cell = row.getCell(i);
                    rowContentMap.put(headerFiledCodeList.get(i), cell == null ? null : readCellRawValue(cell));
                }
                //insert one by one
                Optional<Validation<String, Map<String, Object>>> first = postImportAwareList.stream().map(
                                x -> x.doPostFilter(entityClass, rowContentMap, contextRelatedHeaderMap))
                        .filter(Validation::isInvalid).findFirst();
                CompletableFuture<Either<CreateOneResult, Long>> insertResult;
                if (!first.isPresent()) {
                    insertResult = entityFacade.create(entityClass, rowContentMap, contextRelatedHeaderMap).toCompletableFuture()
                            .thenApplyAsync(x -> x, importThreadPool);
                    currentBatchCount++;
                } else {
                    Validation<String, Map<String, Object>> validation = first.get();
                    insertResult = CompletableFuture.completedFuture(Either.left(CreateOneResult.from(new RuntimeException(validation.getError()))));
                }

                Either<CreateOneResult, Long> join = insertResult.join();

                if (currentBatchCount != 0 && currentBatchCount % step == 0) {
                    String transKey = contextService.get(ContextKeys.StringKeys.TRANSACTION_KEY);
                    if (transKey != null) {
                        ((DefaultTransactionManager) manager).commit(transKey);
                    }
                    //then start new
                    newTransaction = ((DefaultTransactionManager) manager).createNewTransaction(cmd.getTimeout(), "");
                    contextService.set(ContextKeys.StringKeys.TRANSACTION_KEY, newTransaction.getId());
                    currentBatchCount = 0;
                }

//            insertResult.thenAccept(result -> {
                List<String> finalRelatedCodeList = relatedCode;
//                result.peek(x -> {
                if (join.isRight()) {
                    for (String related : finalRelatedCodeList) {
                        buildRelatedIndex(related, contextRelatedHeaderMap, rowContentMap, join.get());
                    }
                } else {
//                }).peekLeft(cor -> {
                    if (join.getLeft().getOriginCause() == ResultStatus.OriginStatus.HALF_SUCCESS) {
                        logger.warn("Half success detect {}", join.getLeft().getErrorMap());
                        for (String related : finalRelatedCodeList) {
                            buildRelatedIndex(related, contextRelatedHeaderMap, rowContentMap, join.getLeft().getId());
                        }
                    } else {
                        if (!cmd.isSilent()) {
                            Map<String, Object> notifyContext = new HashMap<>();
                            notifyContext.put("appId", cmd.getAppId());
                            publisher.publishEvent(new EntityErrorImported(cmd.isAsync() ? "async" : "sync"
                                    , entityClass.code(), entityClass.code(), join.getLeft().getMessage()
                                    , notifyContext, contextRelatedHeaderMap));
                            throw new RuntimeException(join.getLeft().getMessage());
                        }
                    }
                }

                futures.add(insertResult);
            }

            String transKey = contextService.get(ContextKeys.StringKeys.TRANSACTION_KEY);
            if (transKey != null && currentBatchCount > 0) {
                ((DefaultTransactionManager) manager).commit(transKey);
            }

            CompletableFuture<List<Either<CreateOneResult, Long>>> sequence = CompletableFutureUtils.sequence(futures);
            XSSFWorkbook finalXssfwb = xssfwb;
            sequence.thenAccept(result -> {
                for (int i = 1; i < numberOfSheets; i++) {
                    XSSFSheet sheetAt = finalXssfwb.getSheetAt(i);
                    String code = sheetAt.getSheetName();
                    Optional<IEntityClass> iEntityClass = entityClassGroup.relatedEntityClass(code);
                    insertRelated(iEntityClass.get(), sheetAt, contextRelatedHeaderMap, cmd.getSheets(), cmd.getToManyRelations());
                }
            });

            if (!cmd.isAsync()) {
                sequence.join();
            }

            sequence.thenRun(() -> {
                Map<String, Object> notifyContext = new HashMap<>();
                notifyContext.put("appId", cmd.getAppId());
                publisher.publishEvent(new EntityImported(
                        cmd.isAsync() ? "async" : "sync"
                        , entityClass.code(), entityClass.code()
                        , notifyContext, contextRelatedHeaderMap));
            });

        } finally {
            try {
                xssfwb.close();
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }

        return null;
    }

    /**
     * insert related
     *
     * @param entityClass
     * @param sheet
     * @param contextMap
     */
    private List<CompletableFuture<Either<CreateOneResult, Long>>> insertRelated(IEntityClass entityClass
            , XSSFSheet sheet
            , Map<String, Object> contextMap
            , List<ImportCmd.Sheet> sheets
            , List<ImportCmd.ToManyRelation> toManyRelations) {

        /**
         * get related sheetname
         */
        String sheetName = sheet.getSheetName();
        List<CompletableFuture<Either<CreateOneResult, Long>>> futures = new ArrayList<>();
        List<ImportCmd.Sheet.FieldMapping> headerFiledMapping = new ArrayList<>();
        if (CollectionUtil.isNotEmpty(sheets)) {
            Optional<ImportCmd.Sheet> sheetHeaderOptional = sheets.stream().filter(sheetItem -> sheet.getSheetName().equals(sheetItem.getSheet())).findFirst();
            headerFiledMapping = sheetHeaderOptional.isPresent() ? sheetHeaderOptional.get().getFieldMapping() : new ArrayList();
        }
        List<String> headerFiledCodeList = new ArrayList();
        List<Integer> relationCodeColIndexList = new ArrayList<>();

        Row headerRow = null;
        short size = 0;
        Iterator<Row> rowIterator = sheet.rowIterator();
        while (rowIterator.hasNext()) {
            Map<String, Object> map = new HashMap<>();
            Row row = rowIterator.next();
            if (headerRow == null) {
                headerRow = row;
                size = headerRow.getLastCellNum();
                for (int j = 0; j < size; j++) {
                    Cell headerCell = headerRow.getCell(j);
                    if (headerCell.getCellComment() != null
                            && ((List) contextMap.get(sheetName)).contains(headerCell.getCellComment().getString().getString())) {
                        relationCodeColIndexList.add(j);
                    }

                    if (headerFiledMapping.size() > 0 && j < headerFiledMapping.size()) {
                        headerFiledCodeList.add(getBoFieldCodeFromExcelHeader(headerRow.getCell(j).getStringCellValue(), j, headerFiledMapping, entityClass.code()));
                    } else {
                        headerFiledCodeList.add(getBoFieldCodeFromExcelHeader(headerRow.getCell(j).getStringCellValue(), entityClass));
                    }
                }
                continue;
            }

            StringBuilder sb = new StringBuilder();
            for (int i = 0; i < size; i++) {
                Cell cell = row.getCell(i);
                if (cell != null) {
                    Object rawValue = readCellRawValue(cell);
                    if (relationCodeColIndexList.contains(i)) {
                        if (sb.length() > 0) {
                            sb.append("%^%");
                        }
                        sb.append(rawValue);
                    }

                    map.put(headerFiledCodeList.get(i), cell == null ? null : rawValue);
                }
            }

            String s = sb.toString();
            Map<String, Long> indexMapping = getRelatedIndexMapping(contextMap, sheetName);

            Long related = indexMapping.get(s);

            if (related == null) {
                logger.warn("{} has no related value, body: {}", sheetName, map);
            } else {
                map.put(sheetName.concat(".id"), related);
            }

            Optional<Validation<String, Map<String, Object>>> first = postImportAwareList.stream().map(x -> x.doPostFilter(entityClass, map, contextMap))
                    .filter(Validation::isInvalid).findFirst();
            if (!first.isPresent()) {
                //insert one by one
                CompletableFuture<Either<CreateOneResult, Long>> insertResult = entityFacade.create(entityClass, map, contextMap)
                        .toCompletableFuture()
                        .thenApplyAsync(x -> x, importThreadPool);
                futures.add(insertResult);
            } else {
                //do nothing
                logger.error("{}, {}, {} is not execute", entityClass.code(), map, contextMap);
                futures.add(CompletableFuture.completedFuture(Either.left(CreateOneResult.from(new RuntimeException("Skip")))));
            }
        }

        return futures;
    }

    protected Map<String, Long> getRelatedIndexMapping(Map<String, Object> context, String code) {
        Object index = context.get("index");

        if (index != null) {
            Map<String, Map<String, Long>> indexMapping = (Map<String, Map<String, Long>>) index;
            return Optional.ofNullable(indexMapping.get(code)).orElseGet(Collections::emptyMap);
        } else {
            return Collections.emptyMap();
        }
    }

    /**
     * build index
     *
     * @param code
     * @param contextMap
     * @param id
     */
    protected void buildRelatedIndex(String code, Map<String, Object> contextMap, Map<String, Object> body, long id) {
        Object index = contextMap.get("index");
        Object codeMapping = contextMap.get(code);
        if (index != null && codeMapping != null) {
            Map<String, Map<String, Long>> indexMapping = (Map<String, Map<String, Long>>) index;
            List<String> relatedCodeList = (List<String>) codeMapping;
            Map<String, Long> indexCodeRelatedMapping = indexMapping.get(code);

            /**
             * initlization
             */
            if (indexCodeRelatedMapping == null) {
                indexCodeRelatedMapping = new HashMap<>();
            }

            indexMapping.put(code, indexCodeRelatedMapping);

            String collect = relatedCodeList.stream().map(x -> body.get(x))
                    .filter(Objects::nonNull)
                    .map(Object::toString).collect(Collectors.joining("%^%"));
            Long prev = indexCodeRelatedMapping.put(collect, id);
            if (prev != null) {
                throw new RuntimeException("has same related value " + relatedCodeList + ":" + collect);
            }
        }
    }

    /**
     * this ugly
     *
     * @param code
     */
    private boolean isValid(String code, IEntityClassGroup group) {
        Optional<ColumnField> column = group.column(code);
        return column.isPresent();
    }

    /**
     * get header remark
     *
     * @param sheet
     */
    protected void prepareRelatedHeader(XSSFSheet sheet, Map<String, Object> contextRelatedHeaderMap, IEntityClassGroup entityClassGroup) {
        int firstRowNum = sheet.getFirstRowNum();
        //TODO sheetName is relatedCode
        String sheetName = sheet.getSheetName();

        if (firstRowNum >= 0) {
            XSSFRow row = sheet.getRow(firstRowNum);
            Short size = row.getLastCellNum();
            for (int i = 0; i < size; i++) {
                Cell cell = row.getCell(i);
                Comment cellComment = cell.getCellComment();
                if (cellComment != null) {
                    Optional<String> codeOp = Optional.ofNullable(cellComment.getString()).map(RichTextString::getString);
                    codeOp.filter(code -> isValid(code, entityClassGroup)).ifPresent(code -> {
                        contextRelatedHeaderMap.compute(sheetName, (k, v) -> {
                            /**
                             * add related key
                             */
                            if (v == null) {
                                v = new ArrayList<>();
                            }

                            ((List<String>) v).add(code);
                            return v;
                        });
                    });
                }
            }
        } else {
            throw new RuntimeException("do not has row");
        }
    }

}