package com.xforceplus.ultraman.oqsengine.sdk.rebuild;

import akka.NotUsed;
import akka.stream.javadsl.Source;
import com.xforceplus.ultraman.oqsengine.pojo.cdc.constant.CDCConstant;
import com.xforceplus.ultraman.oqsengine.pojo.converter.IEntityClassHelper;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.sdk.*;
import com.xforceplus.ultraman.oqsengine.sdk.rebuild.dto.*;
import com.xforceplus.ultraman.oqsengine.sdk.service.EntityService;
import com.xforceplus.ultraman.oqsengine.sdk.vo.dto.ConditionQueryRequest;
import com.xforceplus.ultraman.oqsengine.sdk.vo.dto.RebuildRequestDTO;
import com.xforceplus.ultraman.oqsengine.sdk.vo.dto.Response;
import com.xforceplus.ultraman.oqsengine.sdk.vo.dto.TaskInfoDTO;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;

/**
 * rebuild controller
 */
@RequestMapping
public class EntityRebuildController {

    @Autowired
    private EntityRebuildService rebuildService;

    @Autowired
    private EntityService entityService;

    private static TaskInfoDTO MISSING = new TaskInfoDTO("-1", "对象不存在");

    @GetMapping(value = "/rebuild/{taskId}/taskinfo", produces = MediaType.APPLICATION_STREAM_JSON_VALUE)
    public ResponseEntity<Source<TaskInfoDTO, NotUsed>> getTaskStream(@PathVariable Long taskId) {
        return ResponseEntity.ok(rebuildService.showProgress(ShowTask.newBuilder().setId(taskId).build()).map(this::toTaskInfoDTO));
    }

    @GetMapping(value = "/rebuild/list/active", produces = MediaType.APPLICATION_STREAM_JSON_VALUE)
    public ResponseEntity<Source<TaskInfoDTO, NotUsed>> listActiveTask(ConditionQueryRequest queryRequest) {
        return ResponseEntity.ok(rebuildService
                .listActiveTasks(QueryPage.newBuilder().setNumber(queryRequest.getPageNo()).setSize(queryRequest.getPageSize()).build())
                .map(this::toTaskInfoDTO));
    }

    @GetMapping(value = "/rebuild/entity/id/{id}/active")
    public ResponseEntity<CompletionStage<TaskInfoDTO>> showCurrentTaskbyId(@PathVariable String id) {
        return ResponseEntity.ok(entityService.load(id).map(this::getActiveTask)
                .orElse(CompletableFuture.completedFuture(MISSING)));
    }

    @GetMapping(value = "/rebuild/entity/code/{code}/active")
    public ResponseEntity<CompletionStage<TaskInfoDTO>> showCurrentTaskByCode(@PathVariable String code) {
        return ResponseEntity.ok(entityService.loadByCode(code).map(this::getActiveTask)
                .orElse(CompletableFuture.completedFuture(MISSING)));
    }

    private CompletionStage<TaskInfoDTO> getActiveTask(IEntityClass entityClass) {
        return rebuildService.getActiveTask(IEntityClassHelper.toEntityUp(entityClass)).thenApply(this::toTaskInfoDTO);
    }

    @GetMapping(value = "/rebuild/list/all", produces = MediaType.APPLICATION_STREAM_JSON_VALUE)
    public ResponseEntity<Source<TaskInfoDTO, NotUsed>> listAllTask(ConditionQueryRequest queryRequest) {
        return ResponseEntity.ok(rebuildService.listAllTasks(QueryPage
                .newBuilder().setNumber(queryRequest.getPageNo()).setSize(queryRequest.getPageSize()).build())
                .map(this::toTaskInfoDTO));
    }

    @PostMapping("/rebuild/{taskId}/cancel")
    public ResponseEntity<CompletionStage<TaskInfoDTO>> cancel(@PathVariable String taskId) {
        return ResponseEntity.ok(rebuildService.cancelTask(ShowTask.newBuilder().setId(Long.parseLong(taskId)).build()).thenApply(this::toTaskInfoDTO));
    }

    @PostMapping(value = "/rebuild/")
    public ResponseEntity<CompletionStage<TaskInfoDTO>> requestTask(@RequestBody RebuildRequestDTO rebuildRequest) {

        String code = rebuildRequest.getCode();
        String id = rebuildRequest.getId();

        /**
         * yyyyMMddHHmmSS
         */
        String startDate = rebuildRequest.getStartDate();
        String endDate = rebuildRequest.getEndDate();

        /**
         *
         */
        Optional<IEntityClass> entityClass = Optional.empty();

        if (StringUtils.isEmpty(id)) {
            if (StringUtils.isNoneEmpty(code)) {
                entityClass = entityService.loadByCode(code);
            }
        } else {
            entityClass = entityService.load(id);
        }

        return ResponseEntity.ok(entityClass.map(x -> {

            EntityUp entityUp = IEntityClassHelper.toEntityUp(x);

            return rebuildService.rebuildIndex(RebuildRequest.newBuilder()
                    .setEntity(entityUp)
                    .setStart(startDate)
                    .setEnd(endDate)
                    .build()).thenApply(this::toTaskInfoDTO);
        }).orElse(CompletableFuture.completedFuture(MISSING)));
    }

    @PostMapping("/cdc/send-error-recover")
    public ResponseEntity<CompletionStage<Response>> cdcSendErrorRecover(@RequestBody CdcRecoverSubmitDTO dto){
        CompletionStage<OperationResult> result = rebuildService.cdcSendErrorRecover(CdcRecoverSubmit
                .newBuilder()
                .setSeqNo(dto.getSeqNo())
                .setRecoverObjectString(dto.getRecoverObjectString())
                .build());

        return ResponseEntity.ok(result.toCompletableFuture().thenApply(this::toResponse));
    }

    @PostMapping("/cdc/cdc-recover-ok")
    public ResponseEntity<CompletionStage<Response>> cdcRecoverOk(@RequestBody CdcRecoverDTO cdcRecoverDTO) {
        return ResponseEntity.ok(rebuildService.cdcRecoverOk(CdcRecover.newBuilder()
                .setSeqNo(cdcRecoverDTO.getSeqNo())
                .build()).thenApply(this::toResponse));
    }

    @PostMapping(value ="/cdc/query-cdc-error", produces = MediaType.APPLICATION_STREAM_JSON_VALUE)
    public ResponseEntity<Source<CdcErrorTaskInfoDTO, NotUsed>> queryCdcError(@RequestBody CdcErrorCondDTO cond){
        return ResponseEntity.ok(rebuildService.queryCdcError(buildCdcCond(cond)).map(this::toCdcErrorTaskInfoDTO));
    }

    @PostMapping("/cdc/range-of-commits")
    public ResponseEntity<CompletionStage<CommitIdMaxMinDTO>> rangeOfCommits(@RequestBody CommitIdUpDTO upDTO){
        return ResponseEntity.ok(rebuildService.rangeOfCommits(CommitIdUp.newBuilder().setCommitId(upDTO.getCommitId()).build()).thenApply(x -> {
            CommitIdMaxMinDTO dto = new CommitIdMaxMinDTO();
            dto.setMax(x.getMax());
            dto.setMin(x.getMin());
            return dto;
        }));
    }

    @PostMapping("/cdc/clean-less-than")
    public ResponseEntity<CompletionStage<Response>> cleanLessThan(@RequestBody CommitIdUpDTO upDTO){

        return ResponseEntity.ok(rebuildService.cleanLessThan(CommitIdUp.newBuilder().setCommitId(upDTO.getCommitId()).build())
                .thenApply(this::toResponse));
    }

    private CdcErrorTaskInfoDTO toCdcErrorTaskInfoDTO(CdcErrorTaskInfo info){
        CdcErrorTaskInfoDTO dto = new CdcErrorTaskInfoDTO();
        dto.setBatchId(info.getBatchId());
        dto.setCommitId(info.getCommitId());
        dto.setEntity(info.getEntity());
        dto.setErrorType(info.getErrorType());
        dto.setExecuteTime(info.getExecuteTime());
        dto.setFixedTime(info.getFixedTime());
        dto.setMessage(info.getMessage());
        dto.setOp(info.getOp());
        dto.setSeqNo(info.getSeqNo());
        dto.setOperationObject(info.getOperationObject());
        dto.setVersion(info.getVersion());
        dto.setStatus(info.getStatus());
        return dto;
    }

    private CdcErrorCond buildCdcCond(CdcErrorCondDTO cond) {

        CdcErrorCond.Builder cdcErrorQueryCondition = CdcErrorCond.newBuilder();

        if (cond.getSeqNo() > CDCConstant.ZERO) {
            cdcErrorQueryCondition.setSeqNo(cond.getSeqNo());
        }

        if (cond.getBatchId() > CDCConstant.ZERO) {
            cdcErrorQueryCondition.setBatchId(cond.getBatchId());
        }

        if (cond.getId() > CDCConstant.ZERO) {
            cdcErrorQueryCondition.setId(cond.getId());
        }

        if (cond.getEntity() > CDCConstant.ZERO) {
            cdcErrorQueryCondition.setEntity(cond.getEntity());
        }

        if (cond.getType() > CDCConstant.ZERO) {
            cdcErrorQueryCondition.setType(cond.getType());
        }

        if (cond.getStatus() > CDCConstant.ZERO) {
            cdcErrorQueryCondition.setStatus(cond.getStatus());
            cdcErrorQueryCondition.setEqualStatus(cond.isEqualStatus());
        }

        if (cond.getRangeLEExecuteTime() > CDCConstant.ZERO) {
            cdcErrorQueryCondition.setRangeLEExecuteTime(cond.getRangeLEExecuteTime());
        }

        if (cond.getRangeGeExecuteTime() > CDCConstant.ZERO) {
            cdcErrorQueryCondition.setRangeGeExecuteTime(cond.getRangeGeExecuteTime());
        }

        if (cond.getRangeLEFixedTime() > CDCConstant.ZERO) {
            cdcErrorQueryCondition.setRangeLEFixedTime(cond.getRangeLEFixedTime());
        }

        if (cond.getRangeGeFixedTime() > CDCConstant.ZERO) {
            cdcErrorQueryCondition.setRangeGeFixedTime(cond.getRangeGeFixedTime());
        }

        return cdcErrorQueryCondition.build();
    }


    private Response toResponse(OperationResult result){
        Response response = new Response();
        String message = result.getMessage();
        if(result.getCode() == OperationResult.Code.OK ){
            response.setCode("1");
        }else{
            response.setCode("1");
        }
        response.setMessage(message);
        return response;
    }


    private TaskInfoDTO toTaskInfoDTO(RebuildTaskInfo rebuildTaskInfo) {
        TaskInfoDTO dto = new TaskInfoDTO();
        if (StringUtils.isNoneEmpty(rebuildTaskInfo.getErrCode()) && Integer.parseInt(rebuildTaskInfo.getErrCode()) < 0) {
            //error condition
            dto.setErrCode(rebuildTaskInfo.getErrCode());
            dto.setMessage(rebuildTaskInfo.getMessage());
            return dto;
        } else {
            dto.setBatchSize(rebuildTaskInfo.getBatchSize());
            dto.setCancel(rebuildTaskInfo.getIsCancel());
            dto.setDone(rebuildTaskInfo.getIsDone());
            dto.setEnds(Long.toString(rebuildTaskInfo.getEnds()));
            dto.setStarts(Long.toString(rebuildTaskInfo.getStarts()));
            dto.setTid(Long.toString(rebuildTaskInfo.getTid()));
            dto.setPercentage(rebuildTaskInfo.getPercentage());
            dto.setFinishSize(rebuildTaskInfo.getFinishSize());
            dto.setEntityId(Long.toString(rebuildTaskInfo.getEntityId()));
            dto.setStatus(rebuildTaskInfo.getStatus());
        }
        return dto;
    }
}
