package com.xforceplus.oqsengine.sdk.reexploit.spring;

import com.xforceplus.oqsengine.sdk.reexploit.spring.annotation.OqsEntity;
import com.xforceplus.oqsengine.sdk.reexploit.spring.model.Domain;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.sdk.facade.EntityFacade;
import com.xforceplus.ultraman.oqsengine.sdk.service.core.ExecutionConfig;
import com.xforceplus.ultraman.oqsengine.sdk.facade.ProfileFetcher;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.ApplicationEventPublisherAware;
import org.springframework.data.mapping.MappingException;
import org.springframework.data.mapping.PersistentPropertyPath;
import org.springframework.data.mapping.PersistentPropertyPaths;
import org.springframework.data.mapping.PropertyPath;
import org.springframework.data.mapping.context.InvalidPersistentPropertyPath;
import org.springframework.data.mapping.context.MappingContext;
import org.springframework.data.mapping.context.MappingContextEvent;
import org.springframework.data.mapping.model.SimpleTypeHolder;
import org.springframework.data.util.DynamicEntityTypeInformation;
import org.springframework.data.util.Optionals;
import org.springframework.data.util.TypeInformation;
import org.springframework.util.Assert;

import java.util.*;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Predicate;
import java.util.stream.Collectors;

public class OqsMappingContext
        implements MappingContext<OqsPersistentEntity<?>, OqsPersistentProperty>, ApplicationEventPublisherAware, ApplicationContextAware {

    private EntityFacade entityFacade;

    private ContextService contextService;

    private ProfileFetcher profileFetcher;

    private Map<Class<?>, Domain> domains = new HashMap<>();

    private Set<? extends Class<?>> initialEntitySet = new HashSet<>();

    private final Map<TypeInformation<?>, Optional<OqsPersistentEntity<?>>> persistentEntities = new HashMap<>();

    private SimpleTypeHolder simpleTypeHolder = SimpleTypeHolder.DEFAULT;

    private ApplicationEventPublisher applicationEventPublisher;

    private ApplicationContext applicationContext;

    private ExecutionConfig executionConfig;

    private final ReentrantReadWriteLock lock = new ReentrantReadWriteLock();
    private final Lock read = lock.readLock();
    private final Lock write = lock.writeLock();

    public OqsMappingContext(EntityFacade entityFacade, ContextService contextService, ProfileFetcher fetcher, ExecutionConfig config) {
        this.entityFacade = entityFacade;
        this.contextService = contextService;
        this.executionConfig = config;
        this.profileFetcher = fetcher;
    }


    public void initialize() {
        initialEntitySet.forEach(this::addPersistentEntity);
    }

    public void init() {

        initialize();

        this.domains = this.getManagedTypes().stream().map(TypeInformation::getType)
                .filter(clz -> clz.isAnnotationPresent(OqsEntity.class))
                .collect(Collectors.toMap(clz -> clz, clz -> new Domain(this, clz)));

    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Override
    public void setApplicationEventPublisher(ApplicationEventPublisher applicationEventPublisher) {
        this.applicationEventPublisher = applicationEventPublisher;
    }

    @Override
    public Collection<OqsPersistentEntity<?>> getPersistentEntities() {
        try {
            read.lock();
            return persistentEntities.values().stream()
                    .flatMap(Optionals::toStream)
                    .collect(Collectors.toSet());
        } finally {
            read.unlock();
        }
    }

    @Override
    public OqsPersistentEntity getPersistentEntity(Class<?> type) {

        OqsEntity annotation = type.getAnnotation(OqsEntity.class);
        if (annotation == null) {
            return null;
        }

        String code = annotation.value();
        String profile = profileFetcher.getProfile(Collections.emptyMap());
        Optional<IEntityClass> iEntityClass = entityFacade.loadByCode(code, profile);
        return iEntityClass.map(x ->
                getPersistentEntity(new DynamicEntityTypeInformation<>(entityFacade.getReader(x, contextService.getAll()), type, profile)))
                .orElse(null);
    }

    @Override
    public boolean hasPersistentEntityFor(Class<?> type) {
        Assert.notNull(type, "Type must not be null!");

        OqsEntity annotation = type.getAnnotation(OqsEntity.class);

        if (annotation == null) {
            return false;
        }

        String code = annotation.value();
        String profile = profileFetcher.getProfile(Collections.emptyMap());
        Optional<IEntityClass> iEntityClass = entityFacade.loadByCode(code, profile);

        Map<String, Object> context = contextService.getAll();

        return iEntityClass.map(x -> {

            Optional<OqsPersistentEntity<?>> oqsPersistentEntity = persistentEntities
                    .get(new DynamicEntityTypeInformation<>(entityFacade.getReader(x, context), type, profile));
            return oqsPersistentEntity != null && oqsPersistentEntity.isPresent();
        }).orElse(false);
    }

    @Override
    public OqsPersistentEntity getPersistentEntity(TypeInformation<?> type) {

        Assert.notNull(type, "Type must not be null!");

        try {
            read.lock();
            Optional<OqsPersistentEntity<?>> entity = persistentEntities.get(type);
            if (entity != null) {
                return entity.orElse(null);
            }
        } finally {
            read.unlock();
        }
        throw new RuntimeException("No Related Entity Found ");
    }

    @Override
    public OqsPersistentEntity getPersistentEntity(OqsPersistentProperty persistentProperty) {
        Assert.notNull(persistentProperty, "PersistentProperty must not be null!");

        if (!persistentProperty.isEntity()) {
            return null;
        }

        TypeInformation<?> typeInfo = persistentProperty.getTypeInformation();
        return getPersistentEntity(typeInfo.getRequiredActualType());
    }

    @Override
    public PersistentPropertyPath<OqsPersistentProperty> getPersistentPropertyPath(PropertyPath propertyPath) throws InvalidPersistentPropertyPath {
        return null;
    }

    @Override
    public PersistentPropertyPath<OqsPersistentProperty> getPersistentPropertyPath(String propertyPath, Class<?> type) throws InvalidPersistentPropertyPath {
        return null;
    }

    @Override
    public <T> PersistentPropertyPaths<T, OqsPersistentProperty> findPersistentPropertyPaths(Class<T> type, Predicate<? super OqsPersistentProperty> predicate) {
        return doFindPersistentPropertyPaths(type, predicate, it -> !it.isAssociation());
    }

    //What is this use
    //TODO
    protected final <T> PersistentPropertyPaths<T, OqsPersistentProperty> doFindPersistentPropertyPaths(Class<T> type,
                                                                                    Predicate<? super OqsPersistentProperty> predicate, Predicate<OqsPersistentProperty> traversalGuard) {
        return null;
    }

    @Override
    public Collection<TypeInformation<?>> getManagedTypes() {
        try {
            read.lock();
            return Collections.unmodifiableSet(new HashSet<>(persistentEntities.keySet()));
        } finally {
            read.unlock();
        }
    }

    protected Optional<OqsPersistentEntity<?>> addPersistentEntity(Class<?> type) {

        OqsEntity annotation = type.getAnnotation(OqsEntity.class);
        if (annotation == null) {
            return null;
        }

        String code = annotation.value();
        String profile = profileFetcher.getProfile(Collections.emptyMap());
        Optional<IEntityClass> iEntityClass = entityFacade.loadByCode(code, profile);

        return iEntityClass.flatMap(x -> this.addPersistentEntity((new DynamicEntityTypeInformation<>(entityFacade.getReader(x, contextService.getAll()), type, profile))));
    }


    /**
     * TODO
     *
     * @param typeInformation
     * @return
     */
    private OqsPersistentEntity createPersistentEntity(TypeInformation<?> typeInformation) {
        OqsPersistentEntity<?> entity = new OqsPersistentEntityImpl<>((DynamicEntityTypeInformation) typeInformation);
        return entity;
    }

    protected Optional<OqsPersistentEntity<?>> addPersistentEntity(TypeInformation<?> typeInformation) {

        Assert.notNull(typeInformation, "TypeInformation must not be null!");

        try {
            read.lock();
            Optional<OqsPersistentEntity<?>> persistentEntity = persistentEntities.get(typeInformation);

            if (persistentEntity != null) {
                return persistentEntity;
            }

        } finally {
            read.unlock();
        }

        Class<?> type = typeInformation.getType();
        OqsPersistentEntity entity = null;

        try {
            write.lock();
            entity = createPersistentEntity(typeInformation);
            // Eagerly cache the entity as we might have to find it during recursive lookups.
            persistentEntities.put(typeInformation, Optional.of(entity));

        } catch (BeansException e) {
            throw new MappingException(e.getMessage(), e);
        } finally {
            write.unlock();
        }

        // Inform listeners
        if (applicationEventPublisher != null && entity != null) {
            applicationEventPublisher.publishEvent(new MappingContextEvent<>(this, entity));
        }

        return Optional.of(entity);
    }

    public void setInitialEntitySet(Set<? extends Class<?>> initialEntitySet) {
        this.initialEntitySet = initialEntitySet;
    }

    public EntityFacade getEntityService() {
        return entityFacade;
    }

    public Domain getDomain(Class<?> entityClass) {
        return this.domains.get(entityClass);
    }

    public Domain getRequiredDomain(Class<?> entityClass) {
        Domain domain = this.getDomain(entityClass);
        if (null == domain) {
            throw new MappingException("Could not find Domain for type: " + entityClass);
        }
        return domain;
    }

    public ContextService getContextService() {
        return contextService;
    }

    public ExecutionConfig getExecutionConfig() {
        return executionConfig;
    }

    public void setExecutionConfig(ExecutionConfig executionConfig) {
        this.executionConfig = executionConfig;
    }
}
