package com.xforceplus.oqsengine.sdk.reexploit.spring.mapping;

import com.xforceplus.oqsengine.sdk.reexploit.spring.annotation.OqsEntity;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.context.annotation.ClassPathScanningCandidateComponentProvider;
import org.springframework.core.type.filter.AnnotationTypeFilter;
import org.springframework.lang.Nullable;
import org.springframework.util.ClassUtils;
import org.springframework.util.StringUtils;

import java.lang.annotation.Annotation;
import java.util.*;


/**
 * Scans packages for entities. The entity classes annotated with
 * {@link #getEntityAnnotations()} will be selected.
 *
 * @author JARVIS SONG
 * @since 2.0.2
 */
public class EntityClassScanner {

    private Set<String> entityBasePackages = new HashSet<>();

    private Set<Class<?>> entityBasePackageClasses = new HashSet<>();

    private @Nullable
    ClassLoader beanClassLoader;

    public static Set<Class<?>> scan(String... entityBasePackages) throws ClassNotFoundException {
        return new EntityClassScanner(entityBasePackages).scanForEntityClasses();
    }

    public static Set<Class<?>> scan(Class<?>... entityBasePackageClasses) throws ClassNotFoundException {
        return new EntityClassScanner(entityBasePackageClasses).scanForEntityClasses();
    }

    public static Set<Class<?>> scan(Collection<String> entityBasePackages) throws ClassNotFoundException {
        return new EntityClassScanner(entityBasePackages).scanForEntityClasses();
    }

    public static Set<Class<?>> scan(Collection<String> entityBasePackages,
                                     Collection<Class<?>> entityBasePackageClasses) throws ClassNotFoundException {
        return new EntityClassScanner(entityBasePackages, entityBasePackageClasses).scanForEntityClasses();
    }

    public EntityClassScanner() {
    }

    public EntityClassScanner(Class<?>... entityBasePackages) {
        this.setEntityBasePackageClasses(Arrays.asList(entityBasePackages));
    }

    public EntityClassScanner(String... entityBasePackages) {
        this(Arrays.asList(entityBasePackages));
    }

    public EntityClassScanner(Collection<String> entityBasePackages) {
        this.setEntityBasePackages(entityBasePackages);
    }

    public EntityClassScanner(Collection<String> entityBasePackages,
                              Collection<Class<?>> entityBasePackageClasses) {
        this.setEntityBasePackages(entityBasePackages);
        this.setEntityBasePackageClasses(entityBasePackageClasses);
    }

    public Set<Class<?>> scanForEntityClasses() throws ClassNotFoundException {
        Set<Class<?>> classes = new HashSet<>();
        for (String basePackage : this.getEntityBasePackages()) {
            classes.addAll(this.scanBasePackageForEntities(basePackage));
        }

        for (Class<?> basePackageClass : this.getEntityBasePackageClasses()) {
            classes.addAll(this.scanBasePackageForEntities(basePackageClass.getPackage().getName()));
        }

        return classes;
    }

    protected Set<Class<?>> scanBasePackageForEntities(String basePackage) throws ClassNotFoundException {
        HashSet<Class<?>> classes = new HashSet<>();
        if (StringUtils.isEmpty(basePackage)) {
            return classes;
        }

        ClassPathScanningCandidateComponentProvider componentProvider = new ClassPathScanningCandidateComponentProvider(
                false);
        for (Class<? extends Annotation> annotation : this.getEntityAnnotations()) {
            componentProvider.addIncludeFilter(new AnnotationTypeFilter(annotation));
        }

        for (BeanDefinition candidate : componentProvider.findCandidateComponents(basePackage)) {
            if (null != candidate.getBeanClassName()) {
                classes.add(ClassUtils.forName(candidate.getBeanClassName(), this.beanClassLoader));
            }
        }
        return classes;
    }

    protected Class<? extends Annotation>[] getEntityAnnotations() {
        return new Class[] { OqsEntity.class };
    }

    public Set<String> getEntityBasePackages() {
        return Collections.unmodifiableSet(this.entityBasePackages);
    }

    public void setEntityBasePackages(Collection<String> entityBasePackages) {
        this.entityBasePackages = new HashSet<>(entityBasePackages);
    }

    public Set<Class<?>> getEntityBasePackageClasses() {
        return Collections.unmodifiableSet(this.entityBasePackageClasses);
    }

    public void setEntityBasePackageClasses(Collection<Class<?>> entityBasePackageClasses) {
        this.entityBasePackageClasses = new HashSet<>(entityBasePackageClasses);
    }

    public void setBeanClassLoader(ClassLoader beanClassLoader) {
        this.beanClassLoader = beanClassLoader;
    }

}
