package com.xforceplus.oqsengine.sdk.reexploit.spring.query;

import com.xforceplus.ultraman.oqsengine.pojo.utils.ConvertHelper;
import com.xforceplus.ultraman.oqsengine.sdk.query.dsl.*;
import com.xforceplus.ultraman.oqsengine.sdk.store.engine.IEntityClassGroup;
import com.xforceplus.ultraman.oqsengine.sdk.utils.PropertyHelperEx;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.repository.query.ParameterAccessor;
import org.springframework.data.repository.query.parser.OqsAbstractQueryCreator;
import org.springframework.data.repository.query.parser.OqsPart;
import org.springframework.data.repository.query.parser.OqsPartTree;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;

import static org.springframework.data.repository.query.parser.OqsPart.Type.IN;

public class OqsQueryCreator extends OqsAbstractQueryCreator<ExpQuery, ExpCondition> {

    IEntityClassGroup reader;

    public OqsQueryCreator(OqsPartTree tree, ParameterAccessor parameters, IEntityClassGroup reader) {
        super(tree, parameters);
        this.reader = reader;
    }

    public OqsQueryCreator(OqsPartTree tree) {
        super(tree);
    }

    @Override
    protected ExpCondition create(OqsPart part, Iterator<Object> iterator) {
        return from(part, iterator);
    }

    @Override
    protected ExpCondition and(OqsPart part, ExpCondition base, Iterator<Object> iterator) {

        if (base == null) {
            return create(part, iterator);
        }

        return ExpCondition.call(ExpOperator.AND, create(part, iterator), base);
    }

    @Override
    protected ExpCondition or(ExpCondition base, ExpCondition criteria) {
        return ExpCondition.call(ExpOperator.OR, base, criteria);
    }

    @Override
    protected ExpQuery complete(ExpCondition criteria, Sort sort, Pageable pageable) {
        ExpQuery expQuery = new ExpQuery();

        if (criteria != null) {
            expQuery.filters(criteria);
        }

        if (sort != null && sort.isSorted()) {
            expQuery.sort(toSort(sort));
        }

        if (pageable != null && !pageable.isUnpaged()) {
            expQuery.range(pageable.getPageNumber(), pageable.getPageSize());
        }

        return expQuery;
    }

    private ExpSort toSort(Sort sort) {

        if (!sort.isUnsorted()) {
            Iterator<Sort.Order> iterator = sort.iterator();
            ExpSort expSort = ExpSort.init();
            while (iterator.hasNext()) {
                Sort.Order order = iterator.next();
                expSort.withSort(order.getProperty(), order.isAscending() ? "asc" : "desc");
            }
            return expSort;
        } else {
            return ExpSort.init();
        }
    }

    private ExpCondition from(OqsPart part, Iterator<?> parameters) {
        OqsPart.Type type = part.getType();

        String key = part.getProperty().toDotPath();

        Optional<String> columnName = PropertyHelperEx.getColumnName(reader, key);
        String codeName = columnName.orElseThrow(() -> new RuntimeException("no such field"));

        switch (type) {
            case SIMPLE_PROPERTY:
                return ExpCondition.call(ExpOperator.EQUALS, ExpField.field(codeName), ExpValue.fromSingle(parameters.next()));
            case BETWEEN:
                return ExpCondition.call(ExpOperator.GREATER_EQ_AND_LESS_EQ, ExpField.field(codeName), ExpValue.fromSingle(parameters.next()), ExpValue.fromSingle(parameters.next()));
            case LESS_THAN:
                return ExpCondition.call(ExpOperator.LESS_THAN, ExpField.field(codeName), ExpValue.fromSingle(parameters.next()));
            case LESS_THAN_EQUAL:
                return ExpCondition.call(ExpOperator.LESS_EQUALS, ExpField.field(codeName), ExpValue.fromSingle(parameters.next()));
            case GREATER_THAN:
                return ExpCondition.call(ExpOperator.GREATER_THAN, ExpField.field(codeName), ExpValue.fromSingle(parameters.next()));
            case GREATER_THAN_EQUAL:
                return ExpCondition.call(ExpOperator.GREATER_EQUALS, ExpField.field(codeName), ExpValue.fromSingle(parameters.next()));
            case AFTER:
                //TODO only date?
                return ExpCondition.call(ExpOperator.GREATER_THAN, ExpField.field(codeName), ExpValue.fromSingle(parameters.next()));
            case BEFORE:
                //TODO only date?
                return ExpCondition.call(ExpOperator.LESS_THAN, ExpField.field(codeName), ExpValue.fromSingle(parameters.next()));
            case LIKE:
                return ExpCondition.call(ExpOperator.LIKE, ExpField.field(codeName), ExpValue.fromSingle(parameters.next()));
            case STARTING_WITH:
                return ExpCondition.call(ExpOperator.LIKE, ExpField.field(codeName), ExpValue.literal(ConvertHelper.convert(parameters.next()) + "%"));
            case ENDING_WITH:
                return ExpCondition.call(ExpOperator.LIKE, ExpField.field(codeName), ExpValue.literal("%" + ConvertHelper.convert(parameters.next())));
            case IS_NOT_EMPTY:
                return ExpCondition.call(ExpOperator.IS_NOT_NULL, ExpField.field(codeName));
            case IS_EMPTY:
                return ExpCondition.call(ExpOperator.IS_NULL, ExpField.field(codeName));
            case CONTAINING:
                return ExpCondition.call(ExpOperator.LIKE, ExpField.field(codeName), ExpValue.literal("%" + ConvertHelper.convert(parameters.next()) + "%"));
            case NEGATING_SIMPLE_PROPERTY:
                return ExpCondition.call(ExpOperator.NOT_EQUALS, ExpField.field(codeName), ExpValue.fromSingle(parameters.next()));
            case IN:
            case NOT_IN:
                Object next = parameters.next();
                List<ExpNode> values = ExpValue.from(next);
                LinkedList<ExpNode> nodes = new LinkedList<>();
                nodes.add(ExpField.field(codeName));
                nodes.addAll(values);
                return ExpCondition.call(type == IN ? ExpOperator.IN : ExpOperator.NOT_IN, nodes);
            case TRUE:
                return ExpCondition.call(ExpOperator.EQUALS, ExpField.field(codeName), ExpValue.fromSingle(true));
            case FALSE:
                return ExpCondition.call(ExpOperator.EQUALS, ExpField.field(codeName), ExpValue.fromSingle(false));
            default:
                throw new RuntimeException("not support");
        }
    }
}
