package org.springframework.data.repository.query.parser;

import com.xforceplus.oqsengine.sdk.reexploit.spring.OqsEntityPersistentPropertyImpl;
import org.springframework.data.domain.Sort;
import org.springframework.data.mapping.PropertyPath;
import org.springframework.data.mapping.model.SnakeCaseFieldNamingStrategy;
import org.springframework.data.util.TypeInformation;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class OqsOrderBySource {

    static OqsOrderBySource EMPTY = new OqsOrderBySource("");

    private static final String BLOCK_SPLIT = "(?<=Asc|Desc)(?=\\p{Lu})";
    private static final Pattern DIRECTION_SPLIT = Pattern.compile("(.+?)(Asc|Desc)?$");
    private static final String INVALID_ORDER_SYNTAX = "Invalid order syntax for part %s!";
    private static final Set<String> DIRECTION_KEYWORDS = new HashSet<>(Arrays.asList("Asc", "Desc"));

    private final List<Sort.Order> orders;

    /**
     * Creates a new {@link OrderBySource} for the given String clause not doing any checks whether the referenced
     * property actually exists.
     *
     * @param clause must not be {@literal null}.
     */
    OqsOrderBySource(String clause) {
        this(clause, Optional.empty());
    }


    /**
     * Creates a new {@link OrderBySource} for the given clause, checking the property referenced exists on the given
     * type.
     *
     * @param clause must not be {@literal null}.
     * @param domainClass must not be {@literal null}.
     */
    OqsOrderBySource(String clause, Optional<TypeInformation<?>> domainClass) {

        this.orders = new ArrayList<>();

        if (!StringUtils.hasText(clause)) {
            return;
        }

        for (String part : clause.split(BLOCK_SPLIT)) {

            Matcher matcher = DIRECTION_SPLIT.matcher(part);

            if (!matcher.find()) {
                throw new IllegalArgumentException(String.format(INVALID_ORDER_SYNTAX, part));
            }

            String propertyString = matcher.group(1);
            String directionString = matcher.group(2);

            // No property, but only a direction keyword
            if (DIRECTION_KEYWORDS.contains(propertyString) && directionString == null) {
                throw new IllegalArgumentException(String.format(INVALID_ORDER_SYNTAX, part));
            }


            this.orders.add(createOrder(propertyString, Sort.Direction.fromOptionalString(directionString), domainClass));
        }
    }

    /**
     * Creates an {@link Sort.Order} instance from the given property source, direction and domain class. If the domain class
     * is given, we will use it for nested property traversal checks.
     *
     * @param propertySource
     * @param direction must not be {@literal null}.
     * @param domainClass must not be {@literal null}.
     * @return
     * @see PropertyPath#from(String, Class)
     */
    private Sort.Order createOrder(String propertySource, Optional<Sort.Direction> direction, Optional<TypeInformation<?>> domainClass) {

        return domainClass.map(type -> {



            PropertyPath propertyPath = PropertyPath.from(propertySource, type);

            String fieldName = new SnakeCaseFieldNamingStrategy().getFieldName(new OqsEntityPersistentPropertyImpl(propertyPath.toDotPath(), null));
            return direction.map(it -> new Sort.Order(it, fieldName))
                    .orElseGet(() -> Sort.Order.by(fieldName));

        }).orElseGet(() -> direction//
                .map(it -> new Sort.Order(it, StringUtils.uncapitalize(propertySource)))
                .orElseGet(() -> Sort.Order.by(StringUtils.uncapitalize(propertySource))));
    }

    /**
     * Returns the clause as {@link Sort}.
     *
     * @return the {@link Sort}.
     */
    Sort toSort() {
        return Sort.by(this.orders);
    }

    /*
     * (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return "Order By " + StringUtils.collectionToDelimitedString(orders, ", ");
    }
}