package org.springframework.data.repository.query.parser;

import org.springframework.data.domain.Sort;
import org.springframework.data.util.ClassTypeInformation;
import org.springframework.data.util.Streamable;
import org.springframework.data.util.TypeInformation;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

public class OqsPartTree implements Streamable<OqsPartTree.OrPart> {

    /*
     * We look for a pattern of: keyword followed by
     *
     *  an upper-case letter that has a lower-case variant \p{Lu}
     * OR
     *  any other letter NOT in the BASIC_LATIN Uni-code Block \\P{InBASIC_LATIN} (like Chinese, Korean, Japanese, etc.).
     *
     * @see <a href="https://www.regular-expressions.info/unicode.html">https://www.regular-expressions.info/unicode.html</a>
     * @see <a href="https://docs.oracle.com/javase/7/docs/api/java/util/regex/Pattern.html#ubc">Pattern</a>
     */
    private static final String KEYWORD_TEMPLATE = "(%s)(?=(\\p{Lu}|\\P{InBASIC_LATIN}))";
    private static final String QUERY_PATTERN = "find|read|get|query|stream";
    private static final String COUNT_PATTERN = "count";
    private static final String EXISTS_PATTERN = "exists";
    private static final String DELETE_PATTERN = "delete|remove";
    private static final Pattern PREFIX_TEMPLATE = Pattern.compile( //
            "^(" + QUERY_PATTERN + "|" + COUNT_PATTERN + "|" + EXISTS_PATTERN + "|" + DELETE_PATTERN + ")((\\p{Lu}.*?))??By");

    /**
     * The subject, for example "findDistinctUserByNameOrderByAge" would have the subject "DistinctUser".
     */
    private final Subject subject;

    /**
     * The subject, for example "findDistinctUserByNameOrderByAge" would have the predicate "NameOrderByAge".
     */
    private final Predicate predicate;

    /**
     * Creates a new {@link org.springframework.data.repository.query.parser.PartTree} by parsing the given {@link String}.
     *
     * @param source the {@link String} to parse
     * @param domainClass the domain class to check individual parts against to ensure they refer to a property of the
     *          class
     */
    public OqsPartTree(String source, Class<?> domainClass) {

        Assert.notNull(source, "Source must not be null");
        Assert.notNull(domainClass, "Domain class must not be null");

        Matcher matcher = PREFIX_TEMPLATE.matcher(source);

        if (!matcher.find()) {
            this.subject = new Subject(Optional.empty());
            this.predicate = new Predicate(source, domainClass);
        } else {
            this.subject = new Subject(Optional.of(matcher.group(0)));
            this.predicate = new Predicate(source.substring(matcher.group().length()), domainClass);
        }
    }

    public OqsPartTree(String source, TypeInformation<?> domainClass) {
        Assert.notNull(source, "Source must not be null");
        Assert.notNull(domainClass, "Domain class must not be null");

        Matcher matcher = PREFIX_TEMPLATE.matcher(source);

        if (!matcher.find()) {
            this.subject = new Subject(Optional.empty());
            this.predicate = new Predicate(source, domainClass);
        } else {
            this.subject = new Subject(Optional.of(matcher.group(0)));
            this.predicate = new Predicate(source.substring(matcher.group().length()), domainClass);
        }
    }

    /*
     * (non-Javadoc)
     * @see java.lang.Iterable#iterator()
     */
    public Iterator<OrPart> iterator() {
        return predicate.iterator();
    }

    /**
     * Returns the {@link Sort} specification parsed from the source.
     *
     * @return never {@literal null}.
     */
    public Sort getSort() {
        return predicate.getOrderBySource().toSort();
    }

    /**
     * Returns whether we indicate distinct lookup of entities.
     *
     * @return {@literal true} if distinct
     */
    public boolean isDistinct() {
        return subject.isDistinct();
    }

    /**
     * Returns whether a count projection shall be applied.
     *
     * @return
     */
    public boolean isCountProjection() {
        return subject.isCountProjection();
    }

    /**
     * Returns whether an exists projection shall be applied.
     *
     * @return
     * @since 1.13
     */
    public boolean isExistsProjection() {
        return subject.isExistsProjection();
    }

    /**
     * return true if the created {@link org.springframework.data.repository.query.parser.PartTree} is meant to be used for delete operation.
     *
     * @return
     * @since 1.8
     */
    public boolean isDelete() {
        return subject.isDelete();
    }

    /**
     * Return {@literal true} if the create {@link org.springframework.data.repository.query.parser.PartTree} is meant to be used for a query with limited maximal results.
     *
     * @return
     * @since 1.9
     */
    public boolean isLimiting() {
        return getMaxResults() != null;
    }

    /**
     * Return the number of maximal results to return or {@literal null} if not restricted.
     *
     * @return {@literal null} if not restricted.
     * @since 1.9
     */
    @Nullable
    public Integer getMaxResults() {
        return subject.getMaxResults().orElse(null);
    }

    /**
     * Returns an {@link Iterable} of all parts contained in the {@link org.springframework.data.repository.query.parser.PartTree}.
     *
     * @return the iterable {@link Part}s
     */
    public Streamable<OqsPart> getParts() {
        return flatMap(OrPart::stream);
    }

    /**
     * Returns all {@link Part}s of the {@link org.springframework.data.repository.query.parser.PartTree} of the given {@link Part.Type}.
     *
     * @param type
     * @return
     */
    public Streamable<OqsPart> getParts(OqsPart.Type type) {
        return getParts().filter(part -> part.getType().equals(type));
    }

    /**
     * Returns whether the {@link org.springframework.data.repository.query.parser.PartTree} contains predicate {@link Part}s.
     *
     * @return
     */
    public boolean hasPredicate() {
        return predicate.iterator().hasNext();
    }

    /*
     * (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {

        return String.format("%s %s", StringUtils.collectionToDelimitedString(predicate.nodes, " or "),
                predicate.getOrderBySource().toString()).trim();
    }

    /**
     * Splits the given text at the given keywords. Expects camel-case style to only match concrete keywords and not
     * derivatives of it.
     *
     * @param text the text to split
     * @param keyword the keyword to split around
     * @return an array of split items
     */
    private static String[] split(String text, String keyword) {

        Pattern pattern = Pattern.compile(String.format(KEYWORD_TEMPLATE, keyword));
        return pattern.split(text);
    }

    /**
     * A part of the parsed source that results from splitting up the resource around {@literal Or} keywords. Consists of
     * {@link Part}s that have to be concatenated by {@literal And}.
     */
    public static class OrPart implements Streamable<OqsPart> {

        private final List<OqsPart> children;

        /**
         * Creates a new {@link org.springframework.data.repository.query.parser.PartTree.OrPart}.
         *
         * @param source the source to split up into {@literal And} parts in turn.
         * @param domainClass the domain class to check the resulting {@link Part}s against.
         * @param alwaysIgnoreCase if always ignoring case
         */
        OrPart(String source, Class<?> domainClass, boolean alwaysIgnoreCase) {

            String[] split = split(source, "And");

            this.children = Arrays.stream(split)//
                    .filter(StringUtils::hasText)//
                    .map(part -> new OqsPart(part, domainClass, alwaysIgnoreCase))//
                    .collect(Collectors.toList());
        }

        OrPart(String source, TypeInformation<?> domainClass, boolean alwaysIgnoreCase) {

            String[] split = split(source, "And");

            this.children = Arrays.stream(split)//
                    .filter(StringUtils::hasText)//
                    .map(part -> new OqsPart(part, domainClass, alwaysIgnoreCase))//
                    .collect(Collectors.toList());
        }

        public Iterator<OqsPart> iterator() {
            return children.iterator();
        }

        @Override
        public String toString() {
            return StringUtils.collectionToDelimitedString(children, " and ");
        }
    }

    /**
     * Represents the subject part of the query. E.g. {@code findDistinctUserByNameOrderByAge} would have the subject
     * {@code DistinctUser}.
     *
     * @author Phil Webb
     * @author Oliver Gierke
     * @author Christoph Strobl
     * @author Thomas Darimont
     */
    private static class Subject {

        private static final String DISTINCT = "Distinct";
        private static final Pattern COUNT_BY_TEMPLATE = Pattern.compile("^count(\\p{Lu}.*?)??By");
        private static final Pattern EXISTS_BY_TEMPLATE = Pattern.compile("^(" + EXISTS_PATTERN + ")(\\p{Lu}.*?)??By");
        private static final Pattern DELETE_BY_TEMPLATE = Pattern.compile("^(" + DELETE_PATTERN + ")(\\p{Lu}.*?)??By");
        private static final String LIMITING_QUERY_PATTERN = "(First|Top)(\\d*)?";
        private static final Pattern LIMITED_QUERY_TEMPLATE = Pattern
                .compile("^(" + QUERY_PATTERN + ")(" + DISTINCT + ")?" + LIMITING_QUERY_PATTERN + "(\\p{Lu}.*?)??By");

        private final boolean distinct;
        private final boolean count;
        private final boolean exists;
        private final boolean delete;
        private final Optional<Integer> maxResults;

        public Subject(Optional<String> subject) {

            this.distinct = subject.map(it -> it.contains(DISTINCT)).orElse(false);
            this.count = matches(subject, COUNT_BY_TEMPLATE);
            this.exists = matches(subject, EXISTS_BY_TEMPLATE);
            this.delete = matches(subject, DELETE_BY_TEMPLATE);
            this.maxResults = returnMaxResultsIfFirstKSubjectOrNull(subject);
        }

        /**
         * @param subject
         * @return
         * @since 1.9
         */
        private Optional<Integer> returnMaxResultsIfFirstKSubjectOrNull(Optional<String> subject) {

            return subject.map(it -> {

                Matcher grp = LIMITED_QUERY_TEMPLATE.matcher(it);

                if (!grp.find()) {
                    return null;
                }

                return StringUtils.hasText(grp.group(4)) ? Integer.valueOf(grp.group(4)) : 1;
            });

        }

        /**
         *
         * @return
         * @since 1.8
         */
        public boolean isDelete() {
            return delete;
        }

        public boolean isCountProjection() {
            return count;
        }

        /**
         *
         * @return
         * @since 1.13
         */
        public boolean isExistsProjection() {
            return exists;
        }

        public boolean isDistinct() {
            return distinct;
        }

        public Optional<Integer> getMaxResults() {
            return maxResults;
        }

        private boolean matches(Optional<String> subject, Pattern pattern) {
            return subject.map(it -> pattern.matcher(it).find()).orElse(false);
        }
    }

    /**
     * Represents the predicate part of the query.
     *
     * @author Oliver Gierke
     * @author Phil Webb
     */
    private static class Predicate implements Streamable<OrPart> {

        private static final Pattern ALL_IGNORE_CASE = Pattern.compile("AllIgnor(ing|e)Case");
        private static final String ORDER_BY = "OrderBy";

        private final List<OrPart> nodes;
        private final
        OqsOrderBySource orderBySource;
        private boolean alwaysIgnoreCase;

        public Predicate(String predicate, Class<?> domainClass) {

            String[] parts = split(detectAndSetAllIgnoreCase(predicate), ORDER_BY);

            if (parts.length > 2) {
                throw new IllegalArgumentException("OrderBy must not be used more than once in a method name!");
            }

            this.nodes = Arrays.stream(split(parts[0], "Or")) //
                    .filter(StringUtils::hasText) //
                    .map(part -> new OrPart(part, domainClass, alwaysIgnoreCase)) //
                    .collect(Collectors.toList());

            this.orderBySource = parts.length == 2 ? new OqsOrderBySource(parts[1], Optional.of(ClassTypeInformation.from(domainClass)))
                    : OqsOrderBySource.EMPTY;
        }

        public Predicate(String predicate, TypeInformation<?> domainClass) {

            String[] parts = split(detectAndSetAllIgnoreCase(predicate), ORDER_BY);

            if (parts.length > 2) {
                throw new IllegalArgumentException("OrderBy must not be used more than once in a method name!");
            }

            this.nodes = Arrays.stream(split(parts[0], "Or")) //
                    .filter(StringUtils::hasText) //
                    .map(part -> new OrPart(part, domainClass, alwaysIgnoreCase)) //
                    .collect(Collectors.toList());

            this.orderBySource = parts.length == 2 ? new OqsOrderBySource(parts[1], Optional.ofNullable(domainClass))
                    : OqsOrderBySource.EMPTY;
        }


        private String detectAndSetAllIgnoreCase(String predicate) {

            Matcher matcher = ALL_IGNORE_CASE.matcher(predicate);

            if (matcher.find()) {
                alwaysIgnoreCase = true;
                predicate = predicate.substring(0, matcher.start()) + predicate.substring(matcher.end(), predicate.length());
            }

            return predicate;
        }

        /*
         * (non-Javadoc)
         * @see java.lang.Iterable#iterator()
         */
        @Override
        public Iterator<OrPart> iterator() {
            return nodes.iterator();
        }

        public OqsOrderBySource getOrderBySource() {
            return orderBySource;
        }
    }
}
