package com.xforceplus.ultraman.starter.autoconfigure;

import com.esotericsoftware.kryo.util.ObjectMap;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.ultraman.metadata.component.GlobalInited;
import com.xforceplus.ultraman.metadata.service.DictService;
import com.xforceplus.ultraman.sdk.core.bulk.exporter.config.ExportConfig;
import com.xforceplus.ultraman.sdk.core.facade.ProfileFetcher;
import com.xforceplus.ultraman.sdk.core.facade.impl.TenantContextProfile;
import com.xforceplus.ultraman.sdk.core.pipeline.TransformerPipeline;
import com.xforceplus.ultraman.sdk.core.pipeline.operation.*;
import com.xforceplus.ultraman.sdk.core.pipeline.operation.validator.*;
import com.xforceplus.ultraman.sdk.core.pipeline.query.ExpRangeValidator;
import com.xforceplus.ultraman.sdk.core.pipeline.query.ExpTreeValidatorImpl;
import com.xforceplus.ultraman.sdk.core.rel.legacy.transformer.ExpTreeTransformer;
import com.xforceplus.ultraman.sdk.core.rel.legacy.validate.ExpTreeValidator;
import com.xforceplus.ultraman.sdk.infra.base.AuthConfig;
import com.xforceplus.ultraman.sdk.infra.base.CdcConfig;
import com.xforceplus.ultraman.sdk.infra.base.ExecutionConfig;
import org.apache.calcite.util.ConversionUtil;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.actuate.autoconfigure.jdbc.DataSourceHealthContributorAutoConfiguration;
import org.springframework.boot.autoconfigure.EnableAutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

@EnableConfigurationProperties(SdkConfiguration.class)
@ConditionalOnProperty(value = "xplat.oqsengine.sdk.enabled", matchIfMissing = true)
public class UltramanAutoConfiguration {

    static {
        System.setProperty("saffron.default.charset", ConversionUtil.NATIVE_UTF16_CHARSET_NAME);
        System.setProperty("saffron.default.nationalcharset",ConversionUtil.NATIVE_UTF16_CHARSET_NAME);
        System.setProperty("saffron.default.collation.name",ConversionUtil.NATIVE_UTF16_CHARSET_NAME + "$en_US");
    }

    @Bean
    public AuthConfig authConfig(SdkConfiguration sdkConfiguration) {
        return sdkConfiguration.getAuth();
    }

    @Bean
    public ExecutionConfig execConfig(SdkConfiguration sdkConfiguration) {
        return sdkConfiguration.getExec();
    }

    @Bean
    public ExportConfig exportConfig(SdkConfiguration sdkConfiguration)  {
        return sdkConfiguration.getExport();
    }

    @Bean
    public CdcConfig cdcConfig(SdkConfiguration sdkConfiguration) {
        return sdkConfiguration.getCdc();
    }

    @Bean
    public ProfileFetcher profileFetcher(ContextService contextService){
        return new TenantContextProfile(contextService);
    }

    @ConditionalOnMissingBean(TransformerPipeline.class)
    @Bean
    public TransformerPipeline transformerPipeline(
            List<FieldOperationHandler> fieldOperationHandlers
            , List<FieldValidator<Object>> fieldValidators
            , List<ExpTreeTransformer> treeTransformers
            , List<ExpTreeValidator> treeValidators
    ) {
        return new TransformerPipeline(fieldOperationHandlers, fieldValidators, treeTransformers, treeValidators);
    }

    @Bean
    public ExpTreeValidator expTreeValidator(List<FieldValidator<Object>> fieldValidators, ProfileFetcher profileFetcher) {
        return new ExpTreeValidatorImpl(fieldValidators, profileFetcher);
    }

    @Bean
    public ExpTreeValidator rangeValidator(@Value("${xplat.oqsengine.sdk.strict.range:true}") boolean isStrict) {
        return new ExpRangeValidator(isStrict);
    }

    @Bean
    public FieldOperationHandler booleanOperationHandler(){
        return new BooleanFieldOperationHandler();
    }

    @Bean
    public FieldValidator regex(SdkConfiguration sdkConfiguration) {
        return new RegxValidator(sdkConfiguration.getValidators());
    }

    @Bean
    public FieldValidator required(SdkConfiguration sdkConfiguration) {
        return new RequiredValidator(sdkConfiguration.getValidators());
    }

    @Bean
    public FieldValidator typedCheck(SdkConfiguration sdkConfiguration) {
        return new TypeCheckValidator(sdkConfiguration.getValidators());
    }

    @Bean
    public FieldValidator enumValue(SdkConfiguration sdkConfiguration, DictService dictService) {
        return new EnumValueValidator(sdkConfiguration.getValidators(),dictService);
    }

    @Bean
    public FieldValidator editable(SdkConfiguration sdkConfiguration) {
        return new EditableValidator(sdkConfiguration.getValidators());
    }

    @Bean
    public MaxLengthValidator maxLengthValidator(SdkConfiguration sdkConfiguration) {
        return new MaxLengthValidator(sdkConfiguration.getValidators());
    }

    @Bean
    public FieldOperationHandler stringsSplitterOperationHandler() {
        return new StringsSplitterOperationHandler();
    }

    @Bean
    public FieldOperationHandler defaultField() {
        return new DefaultFieldOperationHandler();
    }

//    @Bean
//    public FieldOperationHandler defaultValueField() {
//        return new DefaultFieldValueOperationHandler();
//    }

    @Bean
    public FieldOperationHandler defaultSystemField(ContextService contextService, @Value("${xplat.oqsengine.sdk.override:false}") Boolean isOverride) {
        return new FixedDefaultSystemOperationHandler(contextService, isOverride);
    }

    @Bean
    public FieldOperationHandler removeId(){
        return new RemoveZeroOperationHandler();
    }

    @Bean
    public FieldOperationHandler typedConvertToString(DictService dictService){
        return new TypeConvertOperationHandler(dictService);
    }

    @Bean
    public FieldOperationHandler simpleExpressionFieldOperationHandler(ContextService contextService) {
        return new SimpleExpressionFieldOperationHandler(contextService);
    }

    @Bean
    public FieldOperationHandler anotherSimpleExpressionFieldOperationHandler(ContextService contextService) {
        return new AnotherSimpleExpressionFieldOperationHandler(contextService);
    }

    @Bean
    public GlobalInited globalInited(ExecutionConfig executionConfig) {
        return new GlobalInited(
                executionConfig.getInitSize()
                , executionConfig.getInitTimeout());
    }

    @Bean
    public FieldOperationHandler stringDateFormatter(ExecutionConfig executionConfig) {
        List<String> dateFormatters = executionConfig.getDateFormatters();
        if(dateFormatters == null || dateFormatters.isEmpty()) {
            dateFormatters = Collections.singletonList("yyyy-MM-dd HH:mm:ss");
        }

        return new StringToDateTimeOperationHandler(dateFormatters);
    }
}
