package com.xforceplus.ultraman.test.containers.impl;

import com.xforceplus.ultraman.test.containers.AbstractUltramanContainerExtension;
import com.xforceplus.ultraman.test.enums.ContainerSupport;
import com.xforceplus.ultraman.test.enums.constant.Global;
import com.xforceplus.ultraman.test.extend.MysqlParameters;
import com.xforceplus.ultraman.test.extend.TestParameter;
import com.xforceplus.ultraman.test.utils.SqlInitUtils;
import org.junit.jupiter.api.extension.ExtensionContext;
import org.junit.jupiter.api.extension.ParameterContext;
import org.junit.jupiter.api.extension.ParameterResolutionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.testcontainers.containers.BindMode;
import org.testcontainers.containers.GenericContainer;
import org.testcontainers.containers.wait.strategy.Wait;
import org.testcontainers.shaded.org.apache.commons.lang.StringUtils;

import java.time.Duration;

/**
 * Created by justin.xu on 09/2021.
 *
 * @since 1.8
 */
public class MysqlContainerExtension extends AbstractUltramanContainerExtension {

    private static final String MYSQL_USER_PASS = "root";
    private static final Logger LOGGER = LoggerFactory.getLogger(MysqlContainerExtension.class);

    private GenericContainer container;

    @Override
    protected GenericContainer buildContainer(TestParameter parameter) {

        String mysqlUser = parameter.getValue(MysqlParameters.MYSQL_USER);
        String mysqlPass = parameter.getValue(MysqlParameters.MYSQL_PASS);
        String resourcePath = parameter.getValue(MysqlParameters.INIT_SCRIPT);

        if(StringUtils.isEmpty(resourcePath)) {
            resourcePath = "/mysql/default";
        }

        if (StringUtils.isEmpty(mysqlUser)) {
            mysqlUser = "root";
        }

        if (StringUtils.isEmpty(mysqlPass)) {
            mysqlPass = "testing";
        }

        container = new GenericContainer("mysql:8.0.31")
                .withNetworkAliases(buildAliase("mysql"))
                .withExposedPorts(3306)
                .withEnv("MYSQL_DATABASE", "oqs")
                .withEnv("MYSQL_ROOT_USERNAME", mysqlUser)
                .withEnv("MYSQL_ROOT_PASSWORD", mysqlPass)
                .withClasspathResourceMapping(resourcePath.concat("/mysql.cnf"), "/etc/my.cnf", BindMode.READ_ONLY)
                .waitingFor(
                        Wait.forListeningPort().withStartupTimeout(Duration.ofSeconds(Global.WAIT_START_TIME_OUT)));

        return container;
    }

    @Override
    protected void init(TestParameter parameter) {

        String mysqlUser = parameter.getValue(MysqlParameters.MYSQL_USER);
        String mysqlPass = parameter.getValue(MysqlParameters.MYSQL_PASS);
        String resourcePath = parameter.getValue(MysqlParameters.INIT_SCRIPT);

        if(StringUtils.isEmpty(resourcePath)) {
            resourcePath = "/mysql/default";
        }

        if (StringUtils.isEmpty(mysqlUser)) {
            mysqlUser = "root";
        }

        if (StringUtils.isEmpty(mysqlPass)) {
            mysqlPass = "testing";
        }


        setSystemProperties(container.getHost(), container.getMappedPort(3306).toString(), mysqlUser, mysqlPass);

        try {
            SqlInitUtils.execute(resourcePath, "MYSQL_JDBC_WITH_AUTH");
        } catch (Exception e) {
            throw new RuntimeException(e.getMessage(), e);
        }
    }

    @Override
    protected void clean() {
        if (this.container != null) {

            try {
                SqlInitUtils.clean("MYSQL_JDBC_WITH_AUTH");
                SqlInitUtils.execute("/mysql/drop", "MYSQL_JDBC_WITH_AUTH");
            } catch (Exception e) {
                throw new RuntimeException(e.getMessage(), e);
            }
        }
    }

    @Override
    protected ContainerSupport containerSupport() {
        return ContainerSupport.MYSQL;
    }

    @Override
    protected GenericContainer getGenericContainer() {
        return this.container;
    }

    private void setSystemProperties(String address, String port, String usr, String pass) {
        if (null == address || null == port) {
            throw new RuntimeException(
                    String.format("container mysql init failed of null value, address[%s] or port[%s]", address, port));
        }

        System.setProperty("MYSQL_HOST", address);
        System.setProperty("MYSQL_PORT", port);

        System.setProperty(
                "MYSQL_JDBC",
                String.format(
                        "jdbc:mysql://%s:%s/oqs?"
                                + "useUnicode=true"
                                + "&serverTimezone=GMT"
                                + "&useSSL=false"
                                + "&characterEncoding=utf8"
                                + "&allowMultiQueries=true"
                                + "&allowPublicKeyRetrieval=true"
                                + "&rewriteBatchedStatements=true"
                                + "&connectTimeout=50000&socketTimeout=30000",
                        address, port)
        );

        System.setProperty(
                "MYSQL_JDBC_WITH_AUTH",
                String.format(
                        "jdbc:mysql://%s:%s/oqs?"
                                + "useUnicode=true"
                                + "&serverTimezone=GMT"
                                + "&useSSL=false"
                                + "&characterEncoding=utf8"
                                + "&allowMultiQueries=true"
                                + "&allowPublicKeyRetrieval=true"
                                + "&rewriteBatchedStatements=true"
                                + "&connectTimeout=50000&socketTimeout=30000"
                                + "&user=%s&password=%s",
                        address, port, usr, pass)
        );

        LOGGER.info("Start mysql server.({}:{})", address, port);
        LOGGER.info("JDBC URL: {}", System.getProperty("MYSQL_JDBC", "NULL"));
        LOGGER.info("JDBC AUTH URL: {}", System.getProperty("MYSQL_JDBC_WITH_AUTH", "NULL"));
    }

    @Override
    public boolean supportsParameter(ParameterContext parameterContext, ExtensionContext extensionContext) throws ParameterResolutionException {
        return false;
    }

    @Override
    public Object resolveParameter(ParameterContext parameterContext, ExtensionContext extensionContext) throws ParameterResolutionException {
        return null;
    }
}
