package com.xforceplus.ultraman.bocp.gen.autodb.solution;

import com.xforceplus.ultraman.bocp.gen.autodb.sql.create.CreateVal;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class MysqlSqlGenerator implements SqlGenerator {

    final private DdlContext ddlContext;

    private final List<String> NUMERIC_FIELD_TYPES = new ArrayList<>(Arrays.asList(CreateVal.INT, CreateVal.DECIMAL));

    private final List<String> TEXT_FIELD_TYPES = new ArrayList<>(Arrays.asList(CreateVal.TEXT, CreateVal.MEDIUM_TEXT, CreateVal.LONG_TEXT, CreateVal.JSON));

    private final String SYS_DYNAMIC_FIELD_CODE = "_sys_dynamic";
    private TableMeta tableMeta;

    private IndexMeta indexMeta;

    private FieldMeta fieldMeta;

    private DdlOpEnum ddlOp;

    private ObjectType type;

    public MysqlSqlGenerator() {
        this.ddlContext = new DdlContext();
    }

    public MysqlSqlGenerator(DdlContext ddlContext){
        this.ddlContext = ddlContext;
    }

    @Override
    public SqlGenerator table(TableMeta tableMeta, DdlOpEnum ddlOpEnum) {
        this.tableMeta = tableMeta;
        this.ddlOp = ddlOpEnum;
        this.type = ObjectType.TABLE;
        return this;
    }

    @Override
    public SqlGenerator index(IndexMeta indexMeta, DdlOpEnum ddlOpEnum) {
        this.indexMeta = indexMeta;
        this.ddlOp = ddlOpEnum;
        this.type = ObjectType.INDEX;
        return this;
    }

    @Override
    public SqlGenerator field(FieldMeta fieldMeta, DdlOpEnum ddlOpEnum) {
        this.fieldMeta = fieldMeta;
        this.ddlOp = ddlOpEnum;
        this.type = ObjectType.FIELD;
        return this;
    }

    @Override
    public SqlGenerator view(TableMeta tableMeta, DdlOpEnum ddlOpEnum) {
        this.tableMeta = tableMeta;
        this.ddlOp = ddlOpEnum;
        this.type = ObjectType.VIEW;
        return this;
    }

    @Override
    public String getSql() {
        return this.generateSql(false);
    }

    @Override
    public String getRSql() {
        return this.generateSql(true);
    }

    private String generateSql(boolean reverse){
        check();
        if(ObjectType.TABLE.equals(this.type)) {
            if(this.tableMeta == null){
                throw new RuntimeException("table meta must not be null");
            }
            switch (this.ddlOp) {
                case CREATE:
                case REMOVE:
                    return reverse ? generateCreateOrRemoveTable(ddlOp.equals(DdlOpEnum.CREATE) ? DdlOpEnum.REMOVE : DdlOpEnum.CREATE) : generateCreateOrRemoveTable(ddlOp);
                case MODIFY:
                    return genModifyTable(reverse);
                default:
            }
        } else if(ObjectType.INDEX.equals(this.type)) {
            if(this.indexMeta == null){
                throw new RuntimeException("index meta must not be null");
            }
            switch (this.ddlOp) {
                case CREATE:
                case REMOVE:
                    return reverse
                            ? generateCreateOrRemoveIndex(
                                    ddlOp.equals(DdlOpEnum.CREATE) ? DdlOpEnum.REMOVE : DdlOpEnum.CREATE
                    ) : generateCreateOrRemoveIndex(ddlOp);
                case MODIFY:
                    return genModifyIndex(reverse);
                default:
            }
        } else if(ObjectType.FIELD.equals(this.type)) {
            if(this.fieldMeta == null){
                throw new RuntimeException("field meta must not be null");
            }
            switch (this.ddlOp) {
                case CREATE:
                case REMOVE:
                    return reverse ? genCreateOrRemoveField(ddlOp.equals(DdlOpEnum.CREATE) ? DdlOpEnum.REMOVE : DdlOpEnum.CREATE) : genCreateOrRemoveField(ddlOp);
                case MODIFY:
                    return genModifyField(reverse);
                default:
            }
        } else if(ObjectType.VIEW.equals(this.type)) {
            if(this.tableMeta == null){
                throw new RuntimeException("view meta must not be null");
            }
            switch (this.ddlOp) {
                case CREATE:
                case REMOVE:
                    return reverse
                            ? generateCreateOrRemoveView(
                            ddlOp.equals(DdlOpEnum.CREATE) ? DdlOpEnum.REMOVE : DdlOpEnum.CREATE
                    ) : generateCreateOrRemoveView(ddlOp);
                case MODIFY:
                    return genModifyView();
                default:
            }
        }
        return null;
    }

    private String generateCreateOrRemoveTable(DdlOpEnum ddlOpEnum){
        StringBuilder sql = new StringBuilder();
        if(DdlOpEnum.CREATE.equals(ddlOpEnum)) {
            //创建表语句
            sql.append(String.format("CREATE TABLE `%s` ( \n", tableMeta.getCode()));
            //处理字段信息
            if(tableMeta.getFieldMetas() != null) {
                List<FieldMeta> fields = tableMeta.getFieldMetas();
                for (FieldMeta field : fields) {
                    sql.append(String.format("  `%s` ", field.getCode()));
                    sql.append(convertToColumnType(field.getType(), field.getAttr()));
                    if (field.getAttr().isDynamic()) {
                        //动态字段
                        sql.append(String.format(" GENERATED ALWAYS AS (%s->>'$.%s') VIRTUAL", SYS_DYNAMIC_FIELD_CODE, field.getCode()));
                    }
                    if (field.getAttr().isNotNull()) {
                        sql.append(CreateVal.BLANK_SPACE + CreateVal.NOT_NULL);
                    }
                    if (!field.getAttr().isDynamic()) {
                        //动态字段不支持默认值
                        sql.append(generateDefaultValue(field.getType(), field.getAttr()));
                    }

                    if (!StringUtils.isEmpty(field.getAttr().getName())) {
                        sql.append(String.format(" COMMENT '%s'", field.getAttr().getName()));
                    }
                    sql.append(CreateVal.COMMA_SYMBOL);
                    sql.append(CreateVal.CHANGE_LINE);
                }
            }
            sql.append("   PRIMARY KEY (`id`) \n");
            sql.append(String.format(") ENGINE = InnoDB DEFAULT CHARSET = %s COMMENT='%s';\n", ddlContext.getCharset(), tableMeta.getName()));
            //创建索引
            if(tableMeta.getIndexMetas() != null) {
                for (IndexMeta indexMeta : tableMeta.getIndexMetas()) {
                    this.indexMeta = indexMeta;
                    sql.append(generateCreateOrRemoveIndex(ddlOpEnum));
                }
            }
        } else {
            sql.append(String.format("DROP TABLE `%s`; \n", tableMeta.getCode()));
        }
        return sql.toString();
    }

    private String genModifyTable(boolean reverse){
        if(!reverse) {
            return String.format("ALTER TABLE `%s` COMMENT '%s'; \n", tableMeta.getCode(), tableMeta.getNewName());
        } else {
            return String.format("ALTER TABLE `%s` COMMENT '%s'; \n", tableMeta.getCode(), tableMeta.getName());
        }
    }

    private String generateCreateOrRemoveIndex(DdlOpEnum ddlOpEnum){
        StringBuilder sql = new StringBuilder();
        if(DdlOpEnum.CREATE.equals(ddlOpEnum)) {
            String fieldCodeSql = getIndexFieldCodesSql(indexMeta.getFieldCodes());
            //创建索引语句
            sql.append(String.format("CREATE %sINDEX `%s` on `%s` (%s); \n",
                    "unique".equals(indexMeta.getType()) ? "UNIQUE " : "", indexMeta.getCode(), indexMeta.getTableCode(), fieldCodeSql));
        } else {
            sql.append(String.format("DROP INDEX `%s` on `%s`; \n", indexMeta.getCode(), indexMeta.getTableCode()));
        }
        return sql.toString();
    }

    private String genModifyIndex(boolean reverse) {
        StringBuilder sql = new StringBuilder();
        sql.append(String.format("DROP INDEX `%s` on `%s`; \n", indexMeta.getCode(), indexMeta.getTableCode()));
        String fieldCodeSql = getIndexFieldCodesSql(indexMeta.getNewFieldCodes());
        sql.append(String.format("CREATE %sINDEX `%s` on `%s` (%s) \n",
                "unique".equals(indexMeta.getNewType()) ? "UNIQUE " : "",
                indexMeta.getCode(), indexMeta.getTableCode(), reverse ? indexMeta.getFieldCodes() : fieldCodeSql));
        return sql.toString();
    }

    private String genCreateOrRemoveField(DdlOpEnum ddlOpEnum){
        if(fieldMeta.getAttr().isDynamic()) {
            return genCreateOrRemoveDynamicField(ddlOpEnum);
        }

        if(DdlOpEnum.CREATE.equals(ddlOpEnum)) {
            StringBuilder sql = new StringBuilder();
            sql.append(String.format("ALTER TABLE `%s` ADD COLUMN `%s` ", fieldMeta.getTableCode(), fieldMeta.getCode()));
            sql.append(convertToColumnType(fieldMeta.getType(), fieldMeta.getAttr()));
            if (fieldMeta.getAttr().isNotNull()) {
                sql.append(CreateVal.BLANK_SPACE + CreateVal.NOT_NULL);
            }

            sql.append(generateDefaultValue(fieldMeta.getType(), fieldMeta.getAttr()));

            if (!StringUtils.isEmpty(fieldMeta.getAttr().getName())) {
                sql.append(String.format(" COMMENT '%s'", fieldMeta.getAttr().getName()));
            }
            sql.append(CreateVal.END_SYMBOL);
            sql.append(CreateVal.CHANGE_LINE);
            return sql.toString();
        } else {
            return String.format(String.format("ALTER TABLE `%s` DROP COLUMN `%s`; \n", fieldMeta.getTableCode(), fieldMeta.getCode()));
        }
    }

    private String genModifyField(boolean reverse){
        if(fieldMeta.getAttr().isDynamic()) {
            return genModifyDynamicField(reverse);
        }

        if(fieldMeta.getAttr() == null || fieldMeta.getOriginAttr() == null) {
            throw new RuntimeException("field attr or origin attr is null");
        }
        FieldAttrMeta fieldAttr = reverse ? fieldMeta.getOriginAttr() : fieldMeta.getAttr();
        StringBuilder sql = new StringBuilder();
        sql.append(String.format("ALTER TABLE `%s` MODIFY COLUMN `%s` ", fieldMeta.getTableCode(), fieldMeta.getCode()));
        sql.append(convertToColumnType(fieldMeta.getType(), fieldAttr));
        if (fieldAttr.isNotNull()) {
            sql.append(CreateVal.BLANK_SPACE + CreateVal.NOT_NULL);
        }

        sql.append(generateDefaultValue(fieldMeta.getType(), fieldAttr));

        if (!StringUtils.isEmpty(fieldAttr.getName())) {
            sql.append(String.format(" COMMENT '%s'", fieldAttr.getName()));
        }
        sql.append(CreateVal.END_SYMBOL);
        sql.append(CreateVal.CHANGE_LINE);
        return sql.toString();
    }

    private String genCreateOrRemoveDynamicField(DdlOpEnum ddlOpEnum){
        if(DdlOpEnum.CREATE.equals(ddlOpEnum)) {
            StringBuilder sql = new StringBuilder();
            sql.append(String.format("ALTER TABLE `%s` ADD COLUMN `%s` ", fieldMeta.getTableCode(), fieldMeta.getCode()));
            sql.append(convertToColumnType(fieldMeta.getType(), fieldMeta.getAttr()));
            //动态字段
            sql.append(String.format(" GENERATED ALWAYS AS (%s->>'$.%s') VIRTUAL", SYS_DYNAMIC_FIELD_CODE, fieldMeta.getCode()));
            if (fieldMeta.getAttr().isNotNull()) {
                sql.append(CreateVal.BLANK_SPACE + CreateVal.NOT_NULL);
            }
            sql.append(generateDefaultValue(fieldMeta.getType(), fieldMeta.getAttr()));

            if (!StringUtils.isEmpty(fieldMeta.getAttr().getName())) {
                sql.append(String.format(" COMMENT '%s'", fieldMeta.getAttr().getName()));
            }

            sql.append(CreateVal.END_SYMBOL);
            sql.append(CreateVal.CHANGE_LINE);
            return sql.toString();
        } else {
            return String.format(String.format("ALTER TABLE `%s` DROP COLUMN `%s`; \n", fieldMeta.getTableCode(), fieldMeta.getCode()));
        }
    }

    private String genModifyDynamicField(boolean reverse){
        if(fieldMeta.getAttr() == null || fieldMeta.getOriginAttr() == null) {
            throw new RuntimeException("field attr or origin attr is null");
        }
        FieldAttrMeta fieldAttr = reverse ? fieldMeta.getOriginAttr() : fieldMeta.getAttr();
        fieldMeta.setAttr(fieldAttr);

        StringBuilder sql = new StringBuilder();
        sql.append(genCreateOrRemoveDynamicField(DdlOpEnum.REMOVE));
        sql.append(genCreateOrRemoveDynamicField(DdlOpEnum.CREATE));

        return sql.toString();
    }

    private String generateCreateOrRemoveView(DdlOpEnum ddlOpEnum){
        if(null == tableMeta.getParentTableMeta()) {
            return "";
        }
        if(DdlOpEnum.CREATE.equals(ddlOpEnum)) {
            return createViewSql();
        } else {
            return String.format("DROP VIEW `%s_%s`; \n", tableMeta.getCode(), ddlContext.getViewAffix());
        }
    }

    private String genModifyView() {
        return generateViewSql(DdlOpEnum.MODIFY);
    }

    private String createViewSql() {
        return generateViewSql(DdlOpEnum.CREATE);
    }

    private String generateViewSql(DdlOpEnum ddlOpEnum) {
        if (tableMeta.isTenantBo()) {
            return generateTenantBoViewSql(ddlOpEnum);
        }

        Stack<String> tableAliasStack = getTableAliasStack();
        if(null == tableMeta.getParentTableMeta()) {
            return "";
        }
        StringBuilder sql = new StringBuilder();
        Map<String, String> fieldCodeMap = tableMeta.getFieldMetas().stream()
                .filter(fm -> !SYS_DYNAMIC_FIELD_CODE.equals(fm.getCode()))
                .collect(Collectors.toMap(FieldMeta::getCode, f -> String.format("`a`.`%s` AS `%s`", f.getCode(), f.getCode())));
        //创建视图语句
        sql.append(String.format("%s VIEW `%s_%s` AS \n", ddlOpEnum.ddlOp(), tableMeta.getCode(), ddlContext.getViewAffix()));
        sql.append("SELECT ");
        sql.append(String.join(",", fieldCodeMap.values()));

        List<String> childFieldCodeList = new ArrayList<>(fieldCodeMap.keySet());

        StringBuilder leftJoinSql = new StringBuilder();
        List<String> parentTableAliasUsed = new ArrayList<>();
        TableMeta parentTableMeta = tableMeta.getParentTableMeta();
        // 处理父类字段
        while (null != parentTableMeta) {
            dealParentTableMeta(sql, leftJoinSql, parentTableMeta, tableAliasStack, parentTableAliasUsed, childFieldCodeList);
            parentTableMeta = parentTableMeta.getParentTableMeta();
        }
        // 处理父类租户字段
        if (null != tableMeta.getParentTenantTableMeta()) {
            dealParentTableMeta(sql, leftJoinSql, tableMeta.getParentTenantTableMeta(), tableAliasStack, parentTableAliasUsed, childFieldCodeList);
        }

        //根据几层父类生成
        if (!parentTableAliasUsed.isEmpty()) {
            sql.append(String.format(",\nJSON_MERGE_PRESERVE(`a`.`%s`, %s) AS `%s`",
                    SYS_DYNAMIC_FIELD_CODE,
                    parentTableAliasUsed.stream()
                            .map(c -> String.format("`%s`.`%s`", c, SYS_DYNAMIC_FIELD_CODE))
                            .collect(Collectors.joining(", ")),
                    SYS_DYNAMIC_FIELD_CODE));
        }
        sql.append(CreateVal.CHANGE_LINE);
        sql.append(String.format("FROM `%s` `a`", tableMeta.getCode()));
        sql.append(leftJoinSql);
        sql.append("; \n");
        return sql.toString();
    }

    private String generateTenantBoViewSql(DdlOpEnum ddlOpEnum) {
        Stack<String> tableAliasStack = getTableAliasStack();
        if(null == tableMeta.getParentTableMeta()) {
            return "";
        }

        TableMeta mainTableMeta = tableMeta.getParentTableMeta();

        StringBuilder sql = new StringBuilder();
        Map<String, String> fieldCodeMap = mainTableMeta.getFieldMetas().stream()
                .filter(fm -> !SYS_DYNAMIC_FIELD_CODE.equals(fm.getCode()))
                .collect(Collectors.toMap(FieldMeta::getCode, f -> String.format("`a`.`%s` AS `%s`", f.getCode(), f.getCode())));
        //创建视图语句
        sql.append(String.format("%s VIEW `%s_%s` AS \n", ddlOpEnum.ddlOp(), tableMeta.getCode(), ddlContext.getViewAffix()));
        sql.append("SELECT ");
        sql.append(String.join(",", fieldCodeMap.values()));

        List<String> childFieldCodeList = new ArrayList<>(fieldCodeMap.keySet());

        StringBuilder leftJoinSql = new StringBuilder();
        List<String> parentTableAliasUsed = new ArrayList<>();
        TableMeta parentTableMeta = mainTableMeta.getParentTableMeta();
        // 处理父类字段
        while (null != parentTableMeta) {
            dealParentTableMeta(sql, leftJoinSql, parentTableMeta, tableAliasStack, parentTableAliasUsed, childFieldCodeList);
            parentTableMeta = parentTableMeta.getParentTableMeta();
        }
        // 处理父类租户字段
        if (null != tableMeta.getParentTenantTableMeta()) {
            dealParentTableMeta(sql, leftJoinSql, tableMeta.getParentTenantTableMeta(), tableAliasStack, parentTableAliasUsed, childFieldCodeList);
        }

        dealParentTableMeta(sql, leftJoinSql, tableMeta, tableAliasStack, parentTableAliasUsed, childFieldCodeList);

        //根据几层父类生成
        if (!parentTableAliasUsed.isEmpty()) {
            sql.append(String.format(",\nJSON_MERGE_PRESERVE(`a`.`%s`, %s) AS `%s`",
                    SYS_DYNAMIC_FIELD_CODE,
                    parentTableAliasUsed.stream()
                            .map(c -> String.format("`%s`.`%s`", c, SYS_DYNAMIC_FIELD_CODE))
                            .collect(Collectors.joining(", ")),
                    SYS_DYNAMIC_FIELD_CODE));
        }
        sql.append(CreateVal.CHANGE_LINE);
        sql.append(String.format("FROM `%s` `a`", mainTableMeta.getCode()));
        sql.append(leftJoinSql);
        sql.append("; \n");
        return sql.toString();
    }

    private void check() {
        if(ddlContext.getCharset() == null) {
            throw new RuntimeException("charset must not be null");
        }
        if(this.ddlOp == null) {
            throw new RuntimeException("ddl op must not be null");
        }
    }

    private void dealParentTableMeta(
            StringBuilder sql, StringBuilder leftJoinSql, TableMeta parentTableMeta,
            Stack<String> tableAliasStack, List<String> parentTableAliasUsed, List<String> childFieldCodeList) {
        String tableAlias = tableAliasStack.pop();
        parentTableAliasUsed.add(tableAlias);
        Map<String, String> parentFieldCodeMap = parentTableMeta.getFieldMetas().stream()
                .filter(fm -> !SYS_DYNAMIC_FIELD_CODE.equals(fm.getCode()) && !childFieldCodeList.contains(fm.getCode()))
                .collect(Collectors.toMap(FieldMeta::getCode, f -> String.format("`%s`.`%s` AS `%s`", tableAlias, f.getCode(), f.getCode())));
        if (!parentFieldCodeMap.isEmpty()) {
            sql.append(", \n");
            sql.append(parentFieldCodeMap.entrySet()
                    .stream().map(Map.Entry::getValue).collect(Collectors.joining(",")));
        }
        //提前生成left join语句
        leftJoinSql.append(String.format("\nLEFT JOIN `%s` `%s` ON (`a`.`id` = `%s`.`id`) ", parentTableMeta.getCode(), tableAlias, tableAlias));

        parentFieldCodeMap.keySet().forEach(c -> {
            childFieldCodeList.add(c);
        });
    }

    private String convertToColumnType(String fieldType, FieldAttrMeta fieldAttr){
        String mysqlFieldType;
        switch (fieldType) {
            case CreateVal.INT:
                mysqlFieldType = convertToIntType(fieldAttr.getMaxLength());
                break;
            case CreateVal.TINY_INT:
                mysqlFieldType = convertToTinyIntType(fieldAttr.getMaxLength());
                break;
            case CreateVal.DATETIME:
                mysqlFieldType = convertToDateTimeType(fieldAttr.getMaxLength());
                break;
            case CreateVal.DECIMAL:
                mysqlFieldType = convertToDecimalType(fieldAttr.getMaxLength(), fieldAttr.getDecimalPoint());
                break;
            case CreateVal.BOOLEAN:
                mysqlFieldType = CreateVal.TYPE_BOOL;
                break;
            case CreateVal.TEXT:
                mysqlFieldType = CreateVal.TYPE_TEXT;
                break;
            case CreateVal.MEDIUM_TEXT:
                mysqlFieldType = CreateVal.TYPE_MTEXT;
                break;
            case CreateVal.LONG_TEXT:
                mysqlFieldType = CreateVal.TYPE_LTEXT;
                break;
            case CreateVal.JSON:
                mysqlFieldType = CreateVal.TYPE_JSON;
                break;
            case CreateVal.STRING:
            default:
                mysqlFieldType = convertToStringType(fieldAttr.getMaxLength());
        }
        return mysqlFieldType;
    }

    private String convertToIntType(Integer length) {
        return String.format("bigint(%d)", (null != length ? length : 255));
    }

    private String convertToTinyIntType(Integer length) {
        return String.format("tinyint(%d)", (null != length ? length : 255));
    }

    private String convertToStringType(Integer length) {
        return ObjectType.TABLE.equals(type) ?
                String.format("varchar(%d)", (null != length ? length : 255)) :
                String.format("varchar(%d) CHARACTER SET %s", (null != length ? length : 255), ddlContext.getCharset());
    }

    private String convertToDecimalType(Integer length, Integer decimalPoint) {
        return String.format("decimal(%d, %d)", length, decimalPoint);
    }

    private String convertToDateTimeType(Integer digit) {
//        return String.format("datetime(%d)", null != digit ? digit : 30);
        return "datetime";
    }

    private String generateDefaultValue(String fieldType, FieldAttrMeta fieldAttr) {
        String rs = "";
        if(!StringUtils.isEmpty(fieldAttr.getDefaultValue())) {
            if (NUMERIC_FIELD_TYPES.contains(fieldType)) {
                rs = String.format(" DEFAULT %s", fieldAttr.getDefaultValue());
            } else if (CreateVal.TYPE_DATE.equals(fieldType)) {
                if ("CURRENT_TIMESTAMP".equals(fieldAttr.getDefaultValue())) {
                    rs = " DEFAULT CURRENT_TIMESTAMP";
                }
            } else if (CreateVal.TYPE_JSON.equals(fieldType)) {
                // TODO 多值类型如何填充默认值
            } else if (TEXT_FIELD_TYPES.contains(fieldType)) {
                // json,blob,text等类型不支持默认值
            } else {
                rs = String.format(" DEFAULT '%s'", fieldAttr.getDefaultValue());
            }
        }
        return rs;
    }

    private Stack<String> getTableAliasStack() {
        Stack<String> tableAliasStack = new Stack<>();
        tableAliasStack.push("e");
        tableAliasStack.push("d");
        tableAliasStack.push("c");
        tableAliasStack.push("b");
        return tableAliasStack;
    }

    private String getIndexFieldCodesSql(String fieldCodes) {
        return Stream.of(fieldCodes.split(","))
                .map(c -> String.format("`%s`", c.trim()))
                .collect(Collectors.joining(", "));
    }

    private enum ObjectType {
        TABLE,
        INDEX,
        FIELD,
        VIEW;
    }

}
