package com.xforceplus.ultraman.datarule.action.executor;

import com.alibaba.fastjson2.JSON;
import com.esotericsoftware.reflectasm.MethodAccess;
import com.google.common.base.Preconditions;
import com.xforceplus.ultraman.action.constant.ActionType;
import com.xforceplus.ultraman.action.entity.ActionDefinition;
import com.xforceplus.ultraman.datarule.action.DataRuleActionConstant;
import com.xforceplus.ultraman.datarule.action.DataRuleActionScanManager;
import com.xforceplus.ultraman.datarule.action.exception.DataRuleActionException;
import com.xforceplus.ultraman.datarule.domain.dto.DataRuleActionDTO;
import com.xforceplus.ultraman.datarule.domain.dto.DataRuleActionParamDTO;
import com.xforceplus.ultraman.datarule.util.ReflectUtil;
import org.apache.commons.lang3.StringUtils;
import org.springframework.aop.framework.Advised;
import org.springframework.aop.support.AopUtils;
import org.springframework.context.ApplicationContext;

import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 自定义action执行器.
 *
 * @author leo
 * @version 0.1 2022/2/28 2:01 下午
 * @since 1.8
 */
public class DataRuleActionExecutor {

    private ApplicationContext applicationContext;

    public DataRuleActionExecutor(ApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
    }

    public Object execute(DataRuleActionDTO action) {
        checkConfiguration(action);

        List<DataRuleActionParamDTO> params = action.getParams().stream()
                .sorted(Comparator.comparingInt(DataRuleActionParamDTO::getParamIndex))
                .collect(Collectors.toList());
        Object result = innerExecute(action, params);
        return result;
    }

    private boolean checkConfiguration(DataRuleActionDTO action) {
        Preconditions.checkArgument(!StringUtils.isBlank(action.getActionCode()), "action的actionCode不能为空！");
        Preconditions.checkArgument(Optional.ofNullable(action.getParams()).isPresent(), "参数配置不能为空！");
        Preconditions.checkArgument(!Optional.ofNullable(action.getParams()).get().isEmpty(), "action参数配置不能为空！");
        return true;
    }

    private Object innerExecute(DataRuleActionDTO action, List<DataRuleActionParamDTO> actionParams) {
        ActionType actionType = Optional.ofNullable(action.getActionType()).isPresent() ?
                ActionType.fromValue(action.getActionType()) : ActionType.BEANMETHOD;
        Object ret = null;
        if (actionType.equals(ActionType.BEANMETHOD)) {
            Optional<ActionDefinition> actionDefinition = DataRuleActionScanManager.getAction(action.getActionCode());
            if (!actionDefinition.isPresent()) {
                throw new DataRuleActionException(
                        String.format("action setting not found!", action.getActionCode()));
            }
            String beanName = actionDefinition.get().getActionContent().split("\\.")[0];
            String methodName = actionDefinition.get().getActionContent().split("\\.")[1];
            Object bean = applicationContext.getBean(beanName);
            if (AopUtils.isCglibProxy(bean)) {
                try {
                    bean = ((Advised) bean).getTargetSource().getTarget();
                } catch (Exception e) {
                    throw new DataRuleActionException("action bean not found!", e);
                }
            }
            MethodAccess toMethodAccess = ReflectUtil.getMethodAccess(bean.getClass());
            Method method = ReflectUtil.getMethod(bean.getClass(), methodName);
            if (method == null) {
                throw new DataRuleActionException(
                        String.format(" method %s in %s not found", methodName, bean.getClass().getName()));
            }
            List<DataRuleActionParamDTO> input = getInputParams(actionParams);
            if(null != input && !input.isEmpty()) {
                throw new DataRuleActionException("action requires zero input params");
            }

            ret = toMethodAccess.invoke(bean, methodName);
        }
        if (!Optional.ofNullable(ret).isPresent()) {
            throw new DataRuleActionException("action return null value!");
        }

        return getReturn(actionParams, ret);
    }

    private List<DataRuleActionParamDTO> getInputParams(List<DataRuleActionParamDTO> actionParams) {
        List<DataRuleActionParamDTO> input =
            actionParams.stream().filter(param -> param.getParamType() == 1).collect(
                Collectors.toList());
        return input;
    }

    private Object getReturn(List<DataRuleActionParamDTO> actionParams, Object ret) {
        Optional<DataRuleActionParamDTO> outputOptl =
                actionParams.stream().filter(param -> param.getParamType() == 2).findAny();
        if(!outputOptl.isPresent()) {
            throw new DataRuleActionException("action output params setting not found");
        }
        Map<String, Object> outputParam = JSON.parseObject(outputOptl.get().getParamSchema(), Map.class);
        Boolean isArray = (Boolean)outputParam.get("array");
        if(ReflectUtil.isArrayType(ret.getClass()) && isArray == false) {
            throw new DataRuleActionException("action 实际返回值是数组类型，而配置要求返回值不是数组类型");
        }
        if(!ReflectUtil.isArrayType(ret.getClass()) && isArray == true) {
            throw new DataRuleActionException("action 实际返回值不是数组类型，而配置要求返回值是数组类型");
        }
        String type = String.valueOf(outputParam.get("type"));
        String actualType = null;
        if(isArray) {
            if(((List)ret).size() > 0) {
                actualType = ((List)ret).get(0).getClass().getSimpleName().toUpperCase();
            }
        } else {
            actualType = ret.getClass().getSimpleName().toUpperCase();
        }

        if (!DataRuleActionConstant.RETURN_BASIC_TYPES
                .contains(actualType) && type.equals(actualType)) {
            throw new DataRuleActionException(
                    String.format("action要求返回值类型必须为%s或者数组 ",
                            DataRuleActionConstant.RETURN_BASIC_TYPES_STR));
        }

        return parseReturn(isArray, ret);
    }

    private Object parseReturn(Boolean isArray, Object ret){
        if(isArray) {
            return ((List)ret).stream()
                    .map(v -> convertValueToString(v))
                    .collect(Collectors.toList());
        } else {
            return convertValueToString(ret);
        }
    }

    private String convertValueToString(Object v) {
        if(null == v) {
            return "";
        }
        if(v instanceof BigDecimal) {
            return ((BigDecimal)v).toString();
        } else if(v instanceof Boolean) {
            return ((Boolean)v).toString();
        } else if(v instanceof Long) {
            return ((Long)v).toString();
        } else if(v instanceof String) {
            return (String)v;
        } else {
            throw new DataRuleActionException(
                    String.format("action返回类型不属于%s",
                            DataRuleActionConstant.RETURN_BASIC_TYPES_STR));
        }
    }
}
