package com.xforceplus.ultraman.datarule.api.executor;

import com.google.common.collect.Lists;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.tenant.security.core.utils.CompressionUtils;
import com.xforceplus.tenant.security.token.domain.UserType;
import com.xforceplus.tenant.security.token.domain.view.TokenView;
import com.xforceplus.ultraman.datarule.api.convertor.DataRuleApiConvertor;
import com.xforceplus.ultraman.datarule.domain.dto.DataRuleApiAuthTemplateDTO;
import com.xforceplus.ultraman.datarule.domain.dto.DataRuleApiDTO;
import com.xplat.ultraman.api.management.pojo.auth.AuthContentPlaceType;
import com.xplat.ultraman.api.management.pojo.auth.AuthTemplateType;
import com.xplat.ultraman.api.management.restclient.agent.executor.AgentExecutor;
import com.xplat.ultraman.api.management.restclient.dto.AgentClient;
import com.xplat.ultraman.api.management.restclient.dto.AuthTemplate;
import com.xplat.ultraman.api.management.restclient.rest.ParameterTypeReference;
import io.geewit.web.utils.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * CopyRight: 上海云砺信息科技有限公司
 * User: youyifan
 * DateTime: 2023/2/20 14:26
 * Description:
 * History:
 */
@Slf4j
@Component
public class DataRuleApiExecutor {

    private AgentExecutor restAgentExecutor;

    private DataRuleApiConvertor dataRuleApiConvertor;

    private Map<String, String> authTplMap;

    public DataRuleApiExecutor(AgentExecutor restAgentExecutor, List<String> authTpls) {
        this.restAgentExecutor = restAgentExecutor;
        this.dataRuleApiConvertor = new DataRuleApiConvertor();
        this.authTplMap = authTpls.stream().filter(s -> s.contains(":"))
                .collect(Collectors.toMap(s -> s.split(":")[0], s -> s.split(":")[1]));
    }

    public List<String> execute(DataRuleApiDTO api, DataRuleApiAuthTemplateDTO authTpl) {
        AuthTemplate authTemplate = getAuthTemplate(authTpl);
        if(api.getResponse().getBody().isArray()) {
            return doExecuteWithReturnList(api, authTemplate);
        } else {
            return doExecuteWithReturnBasicType(api, authTemplate);
        }
    }

    private List<String> doExecuteWithReturnList(DataRuleApiDTO api, AuthTemplate authTemplate) {
        AgentClient.AgentClientBuilder agentClientBuilder = new AgentClient.AgentClientBuilder<List>(authTemplate, api.getUrl())
                .method(api.getMethod())
                .headers(getUserHeaders())
                .parameterTypeReference(new ParameterTypeReference<List>() {});

        List<Object> resp = (List) restAgentExecutor.execute(agentClientBuilder.builder());

        return null == resp
                ? Lists.newArrayList() : resp.stream().map(i -> dataRuleApiConvertor.convertObjToString(i)).collect(Collectors.toList());
    }

    private List<String> doExecuteWithReturnBasicType(DataRuleApiDTO api, AuthTemplate authTemplate) {
        AgentClient.AgentClientBuilder agentClientBuilder = new AgentClient.AgentClientBuilder<>(authTemplate, api.getUrl())
                .method(api.getMethod())
                .headers(getUserHeaders())
                .parameterTypeReference(new ParameterTypeReference<Object>() {});

        Object resp = restAgentExecutor.execute(agentClientBuilder.builder());

        return null == resp ? Lists.newArrayList() : Arrays.asList(dataRuleApiConvertor.convertObjToString(resp));
    }

    private AuthTemplate getAuthTemplate(DataRuleApiAuthTemplateDTO authTpl) {
        if(!authTplMap.containsKey(authTpl.getAuthTemplateCode())) {
            throw new RuntimeException(String.format("env code not found in ultraman.datarule.api.authTpls: %s", authTpl.getAuthTemplateCode()));
        }
        String envCode = authTplMap.get(authTpl.getAuthTemplateCode());

        return new AuthTemplate(envCode, authTpl.getAuthTemplateCode(), AuthTemplateType.JWT_AUTH, AuthContentPlaceType.CUSTOM);
    }

    /**
     * 组织放置到请求头的用户信息
     * @return
     */
    private Map<String, Object> getUserHeaders(){
        Map<String, Object> headers = new HashMap();

        IAuthorizedUser user = UserInfoHolder.get();
        if(null != user) {
            try {
                String value = CompressionUtils.encode(JsonUtils.toJson(user, TokenView.class));
                headers.put(UserType.USER.userinfoKey(), value);
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
        }
        return headers;
    }

}
