package com.xforceplus.ultraman.datarule.api.usercenter.uc;

import com.google.common.collect.Maps;
import com.xforceplus.ultraman.datarule.api.usercenter.api.IUserCenterEnvApi;
import com.xforceplus.ultraman.datarule.api.usercenter.dto.ApiSmartResult;
import com.xforceplus.ultraman.datarule.api.usercenter.dto.AuthTplDTO;
import com.xforceplus.ultraman.datarule.api.usercenter.enums.UserInfoType;
import com.xforceplus.ultraman.datarule.domain.enums.AppEnvType;
import com.xplat.ultraman.api.management.restclient.exception.RestCallException;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * CopyRight: 上海云砺信息科技有限公司
 * User: youyifan
 * DateTime: 2022/9/14 16:59
 * Description:
 * History:
 */
@Slf4j
public class DefaultUserInfoApiImpl implements IUserInfoApi {

    public static final String TEMPLATE_CODE = "usercenter";

    private final int PAGE_NO = 0;

    private final int PAGE_ROW_NUM = 1;

    private int inListComputeLimit = 100;

    private int notInListComputeLimit = 10;

    private IUserCenterEnvApi userCenterEnvApi;

    private AuthTplDTO authTplDTO;

    public DefaultUserInfoApiImpl(IUserCenterEnvApi userCenterEnvApi, String env, Integer inListComputeLimit, Integer notInListComputeLimit) {
        this.userCenterEnvApi = userCenterEnvApi;
        this.inListComputeLimit = null == inListComputeLimit ? this.inListComputeLimit : inListComputeLimit;
        this.notInListComputeLimit = null == notInListComputeLimit ? this.notInListComputeLimit : notInListComputeLimit;
        this.authTplDTO = AuthTplDTO.builder()
                .templateCode(TEMPLATE_CODE)
                .env(AppEnvType.fromCode(env).desc())
                .build();
    }

    @Override
    public Map<String, Object> getUserInfo(Long tenantId, Long userId) {
        return userCenterEnvApi.getUserInfo(authTplDTO, tenantId, userId);
    }

    @Override
    public Map<String, Object> getUserInfoByUserCode(Long tenantId, String userCode) {
        Map<String, Object> params = Maps.newHashMap();
        params.put("userCode", userCode);
        params.put("page", PAGE_NO);
        params.put("row", PAGE_ROW_NUM);
        List<Map> users = userCenterEnvApi.getUserList(authTplDTO, tenantId, params);
        if (users.isEmpty()) {
            return null;
        }
        return getUserInfo(tenantId, Long.valueOf((String) users.get(0).get("id")));
    }

    @Override
    public List<Long> getUserOrgIdsWithFullFlag(Long tenantId, Long userId) {
        List<String> orgIdStrs = getUserData(tenantId, userId, "orgId", this::getUserOrgsWithFullFlag);
        return null == orgIdStrs ? null : orgIdStrs.stream().map(Long::valueOf).collect(Collectors.toList());
    }

    @Override
    public List<String> getUserOrgCodesWithFullFlag(Long tenantId, Long userId) {
        return getUserData(tenantId, userId, "orgCode", this::getUserOrgsWithFullFlag);
    }

    @Override
    public List<Long> getUserOrgIds(Long tenantId, Long userId) {
        return getUserData(tenantId, userId, "orgId", this::getUserOrgs)
                .stream()
                .map(Long::valueOf)
                .collect(Collectors.toList());
    }

    @Override
    public List<String> getUserOrgCodes(Long tenantId, Long userId) {
        return getUserData(tenantId, userId, "orgCode", this::getUserOrgs);
    }

    @Override
    public List<String> getUserCpyCodes(Long tenantId, Long userId) {
        return getUserData(tenantId, userId, "companyCode", this::getUserCpys);
    }

    @Override
    public List<String> getUserCpyTaxNums(Long tenantId, Long userId) {
        return getUserData(tenantId, userId, "taxNum", this::getUserCpys);
    }

    @Override
    public ApiSmartResult<Long> getUserOrgIdsUsingResult(Long tenantId, Long userId) {
        return getSmartResult(tenantId, userId, this::getUserOrgIdsWithFullFlag, this::getTenantOrgIds, this::getTenantOrgTotalCount);
    }

    @Override
    public ApiSmartResult<String> getUserOrgCodesUsingResult(Long tenantId, Long userId) {
        return getSmartResult(tenantId, userId, this::getUserOrgCodesWithFullFlag, this::getTenantOrgCodes, this::getTenantOrgTotalCount);
    }

    @Override
    public ApiSmartResult<String> getUserCpyCodesUsingResult(Long tenantId, Long userId) {
        return getSmartResult(tenantId, userId, this::getUserCpyCodes, this::getTenantCpyCodes, this::getTenantCpyTotalCount);
    }

    @Override
    public ApiSmartResult<String> getUserCpyTaxNumsUsingResult(Long tenantId, Long userId) {
        return getSmartResult(tenantId, userId, this::getUserCpyTaxNums, this::getTenantCpyTaxNums, this::getTenantCpyTotalCount);
    }

    @Override
    public List getUserData(Long tenantId, Long userId, UserInfoType userInfoType) {
        if (null == userInfoType) {
            return new ArrayList();
        }
        switch (userInfoType) {
            case USER_ORG_ID:
                return getUserOrgIds(tenantId, userId);
            case USER_ORG_CODE:
                return getUserOrgCodes(tenantId, userId);
            case USER_CPY_CODE:
                return getUserCpyCodes(tenantId, userId);
            case USER_CPY_TAX_NUM:
                return getUserCpyTaxNums(tenantId, userId);
        }
        return new ArrayList();
    }

    @Override
    public ApiSmartResult getUserDataUsingResult(Long tenantId, Long userId, UserInfoType userInfoType) {
        if (null == userInfoType) {
            return new ApiSmartResult();
        }
        switch (userInfoType) {
            case USER_ORG_ID:
                ApiSmartResult apiSmartResult = getUserOrgIdsUsingResult(tenantId, userId);
                apiSmartResult.setNumeric(true);
                return apiSmartResult;
            case USER_ORG_CODE:
                return getUserOrgCodesUsingResult(tenantId, userId);
            case USER_CPY_CODE:
                return getUserCpyCodesUsingResult(tenantId, userId);
            case USER_CPY_TAX_NUM:
                return getUserCpyTaxNumsUsingResult(tenantId, userId);
        }
        return new ApiSmartResult();
    }

    @Override
    public List<Map> getTenantCpys(Long tenantId) {
        try {
            return userCenterEnvApi.getTenantCpys(authTplDTO,
                    tenantId, null);
        } catch (RestCallException e) {
            log.error("failed to request user center api", e);
            return new ArrayList<>();
        }
    }

    private List<String> getTenantData(Long tenantId, String key, Function<Long, List<Map>> getTenantData) {
        return getTenantData.apply(tenantId).stream()
                .map(m -> (String) m.get(key))
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    private List<String> getUserData(Long tenantId, Long userId, String key, BiFunction<Long, Long, List<Map>> getUserData) {
        List<Map> userData = getUserData.apply(tenantId, userId);
        return null == userData ? null : userData.stream()
                .map(m -> (String) m.get(key))
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    private int getTenantOrgTotalCount(Long tenantId) {
        Integer total = userCenterEnvApi.getTenantOrgTotalCount(authTplDTO, tenantId);
        return null != total ? total.intValue() : 0;
    }

    private int getTenantCpyTotalCount(Long tenantId) {
        Integer total = userCenterEnvApi.getTenantCpyTotalCount(authTplDTO, tenantId);
        return null != total ? total.intValue() : 0;
    }

    private List<Long> getTenantOrgIds(Long tenantId) {
        return getTenantData(tenantId, "orgId", this::getTenantOrgs)
                .stream()
                .map(Long::valueOf)
                .collect(Collectors.toList());
    }

    private List<String> getTenantOrgCodes(Long tenantId) {
        return getTenantData(tenantId, "orgCode", this::getTenantOrgs);
    }

    private List<String> getTenantCpyCodes(Long tenantId) {
        return getTenantData(tenantId, "companyCode", this::getTenantCpys);
    }

    private List<String> getTenantCpyTaxNums(Long tenantId) {
        return getTenantData(tenantId, "taxNum", this::getTenantCpys);
    }

    private List<Map> getTenantOrgs(Long tenantId) {
        try {
            return userCenterEnvApi.getTenantOrgs(authTplDTO, tenantId, null);
        } catch (RestCallException e) {
            log.error("failed to request user center api", e);
            return new ArrayList<>();
        }
    }


    private List<Map> getUserOrgsWithFullFlag(Long tenantId, Long userId) {
        try {
            return userCenterEnvApi.getUserTenantOrgsWithFullFlag(authTplDTO,
                    tenantId, userId);
        } catch (RestCallException e) {
            log.error("failed to request user center api", e);
            return new ArrayList<>();
        }
    }

    private List<Map> getUserOrgs(Long tenantId, Long userId) {
        try {
            return userCenterEnvApi.getUserTenantOrgs(authTplDTO,
                    tenantId, userId);
        } catch (RestCallException e) {
            log.error("failed to request user center api", e);
            return new ArrayList<>();
        }
    }

    private List<Map> getUserCpys(Long tenantId, Long userId) {
        try {
            return userCenterEnvApi.getUserTenantCpys(authTplDTO, tenantId, userId);
        } catch (RestCallException e) {
            log.error("failed to request user center api", e);
            return new ArrayList<>();
        }
    }

    /**
     * @param tenantId
     * @param userId
     * @param getUserData
     * @param getTenantData
     * @param getTenantDataTotalCount
     * @param <T>
     * @return
     */
    private <T> ApiSmartResult<T> getSmartResult(Long tenantId, Long userId, BiFunction<Long, Long, List<T>> getUserData, Function<Long, List<T>> getTenantData, Function<Long, Integer> getTenantDataTotalCount) {
        ApiSmartResult<T> smartResult = new ApiSmartResult<T>();
        List<T> userData = getUserData.apply(tenantId, userId);
        if (null == userData) {
            // 当前用户拥有租户全部组织
            smartResult.setAll(true);
            return smartResult;
        }
        smartResult.setData(userData);
        if (userData.size() >= inListComputeLimit) {
            int tenantDataTotalCount = getTenantDataTotalCount.apply(tenantId);
            if (tenantDataTotalCount == userData.size()) {
                smartResult.setAll(true);
                return smartResult;
            }
            int notInListSize = tenantDataTotalCount - userData.size();
            //用户中心翻页请求的pageSize上限是1000
            //查询租户使用的翻页查询最多1000条，而查询用户信息能查到全部信息
            //以上情况，会导致 租户数据-用户数据(notInListSize) < 0
            if ((notInListSize > 0) && (notInListSize <= notInListComputeLimit) && (notInListSize < userData.size())) {
                List<T> tenantData = getTenantData.apply(tenantId);
                tenantData.removeAll(userData);
                smartResult.setUseNotIn(true);
                smartResult.setData(tenantData);
                return smartResult;
            }
        }
        return smartResult;
    }
}
