package com.xforceplus.ultraman.datarule.api.usercenter.api.impl;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.tenant.security.core.utils.CompressionUtils;
import com.xforceplus.tenant.security.token.domain.UserType;
import com.xforceplus.tenant.security.token.domain.view.TokenView;
import com.xforceplus.ultraman.datarule.api.usercenter.api.IUserCenterEnvApi;
import com.xforceplus.ultraman.datarule.api.usercenter.api.constant.UserCenterConstant;
import com.xforceplus.ultraman.datarule.api.usercenter.api.convertor.UserCenterConvertor;
import com.xforceplus.ultraman.datarule.api.usercenter.dto.AuthTplDTO;
import com.xforceplus.ultraman.datarule.api.usercenter.dto.ResponseEntity;
import com.xplat.ultraman.api.management.convertor.executor.ConvertWrapper;
import com.xplat.ultraman.api.management.pojo.auth.AuthContentPlaceType;
import com.xplat.ultraman.api.management.pojo.auth.AuthTemplateType;
import com.xplat.ultraman.api.management.pojo.enums.Method;
import com.xplat.ultraman.api.management.restclient.agent.executor.AgentExecutor;
import com.xplat.ultraman.api.management.restclient.dto.AgentClient;
import com.xplat.ultraman.api.management.restclient.dto.AuthTemplate;
import com.xplat.ultraman.api.management.restclient.rest.ParameterTypeReference;
import io.geewit.web.utils.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.util.*;
import java.util.function.Function;

/**
 * CopyRight: 上海云砺信息科技有限公司
 * User: youyifan
 * DateTime: 2022/8/5 18:10
 * Description:
 * History:
 */
@Slf4j
public class UserCenterEnvApiImpl implements IUserCenterEnvApi {

    private AgentExecutor restAgentExecutor;
    private UserCenterConvertor userCenterConverter;

    public UserCenterEnvApiImpl(AgentExecutor restAgentExecutor, ConvertWrapper convertWrapper) {
        this.restAgentExecutor = restAgentExecutor;
        this.userCenterConverter = new UserCenterConvertor(convertWrapper);
    }

    /**
     * 查询租户列表信息
     * @param authTpl
     * @param searchKey
     * @return
     */
    @Override
    public List<Map> getTenants(AuthTplDTO authTpl, String searchKey) {
        String url = StringUtils.isEmpty(searchKey) ?
                UserCenterConstant.TENANT_API: String.format(UserCenterConstant.TENANT_SEARCH_API, searchKey);
        return doRequest(authTpl, url, userCenterConverter::convertTenantMap);
    }

    /**
     * 查询租户角色列表信息
     * @param authTpl
     * @param tenantId
     * @param searchKey
     * @return
     */
    @Override
    public List<Map> getTenantRoles(AuthTplDTO authTpl, Long tenantId, String searchKey) {
        String url = StringUtils.isEmpty(searchKey) ?
                String.format(UserCenterConstant.ROLE_API, tenantId) : String.format(UserCenterConstant.ROLE_SEARCH_API, tenantId, searchKey);
        return doRequest(authTpl, url, userCenterConverter::convertRoleMap);
    }

    /**
     * 查询租户组织总数
     * @param authTpl
     * @param tenantId
     * @return
     */
    @Override
    public Integer getTenantOrgTotalCount(AuthTplDTO authTpl, Long tenantId) {
        String url = String.format(UserCenterConstant.TENANT_ORG_COUNT_API, tenantId);
        return doRequestCount(authTpl, url, userCenterConverter::getTotalElementCount);
    }

    /**
     * 查询租户公司总数
     * @param authTpl
     * @param tenantId
     * @return
     */
    @Override
    public Integer getTenantCpyTotalCount(AuthTplDTO authTpl, Long tenantId) {
        String url = String.format(UserCenterConstant.TENANT_CPY_COUNT_API, tenantId);
        return doRequestCount(authTpl, url, userCenterConverter::getTotalElementCount);
    }

    /**
     * 查询租户组织列表信息
     * @param authTpl
     * @param tenantId
     * @param searchKey
     * @return
     */
    @Override
    public List<Map> getTenantOrgs(AuthTplDTO authTpl, Long tenantId, String searchKey) {
        String url = StringUtils.isEmpty(searchKey) ?
                String.format(UserCenterConstant.TENANT_ORG_API, tenantId) : String.format(UserCenterConstant.TENANT_ORG_SEARCH_API, tenantId, searchKey);
        return doRequest(authTpl, url, userCenterConverter::convertOrgMap);
    }

    /**
     * 查询租户的公司列表信息
     * @param authTpl
     * @param tenantId
     * @return
     */
    @Override
    public List<Map> getTenantCpys(AuthTplDTO authTpl, Long tenantId, String searchKey) {
        String url = StringUtils.isEmpty(searchKey) ?
                String.format(UserCenterConstant.TENANT_CPY_API, tenantId) : String.format(UserCenterConstant.TENANT_CPY_SEARCH_API, tenantId, searchKey);
        return doRequest(authTpl, url, userCenterConverter::convertCpyMap);
    }

    /**
     * 获取用户信息
     *
     * @param authTpl
     * @param tenantId
     * @param userId
     * @return
     */
    @Override
    public Map getUserInfo(AuthTplDTO authTpl, Long tenantId, Long userId) {
        String url = String.format(UserCenterConstant.USER_INFO_API, tenantId, userId);

        ResponseEntity response = invokeRestAgentExecutor(authTpl, url);

        return ResponseEntity.OK.equals(response.getCode())
                ? userCenterConverter.convertUser((Map)response.getResult()) : new HashMap();
    }

    /**
     * 获取用户列表
     * @param authTpl
     * @param tenantId
     * @param params
     * @return
     */
    @Override
    public List<Map> getUserList(AuthTplDTO authTpl, Long tenantId, Map<String, Object> params) {
        String url = String.format(UserCenterConstant.USER_LIST_API, tenantId);
        return doRequest(authTpl, url, userCenterConverter::convertUsers);
    }

    /**
     * 查询用户当前租户所属组织列表信息
     * @param authTpl
     * @param tenantId
     * @param userId
     * @return
     */
    @Override
    public List<Map> getUserTenantOrgs(AuthTplDTO authTpl, Long tenantId, Long userId) {
        String url = String.format(UserCenterConstant.USER_TENANT_ORG_API, tenantId, userId);
        return doRequest(authTpl, url, userCenterConverter::convertUserToOrgMap);
    }

    /**
     * 查询用户当前租户所属组织列表信息带是否全部组织标记
     * @param authTpl
     * @param tenantId
     * @param userId
     * @return
     */
    @Override
    public List<Map> getUserTenantOrgsWithFullFlag(AuthTplDTO authTpl, Long tenantId, Long userId) {
        String url = String.format(UserCenterConstant.USER_TENANT_ORG_WITH_FULL_FLAG_API, tenantId, userId);
        return doRequest(authTpl, url, userCenterConverter::convertUserToOrgMapWithFullFlag);
    }

    /**
     * 查询 用户当前租户所属公司列表信息
     * @param authTpl
     * @param tenantId
     * @param userId
     * @return
     */
    @Override
    public List<Map> getUserTenantCpys(AuthTplDTO authTpl, Long tenantId, Long userId) {
        String url = String.format(UserCenterConstant.USER_TENANT_CPY_API, tenantId, userId);
        return doRequest(authTpl, url, userCenterConverter::convertUserToCpyMap);
    }

    /**
     * 执行请求并解析结果
     * @param authTpl
     * @param url
     * @param converter
     * @return List
     */
    private List<Map> doRequest(AuthTplDTO authTpl, String url, Function<Map, List<Map>> converter){
        ResponseEntity response = invokeRestAgentExecutor(authTpl, url);
        return ResponseEntity.OK.equals(response.getCode())
                ? converter.apply((Map)response.getResult()) : Lists.newArrayList();
    }

    /**
     * 执行请求并解析结果
     * @param authTpl
     * @param url
     * @param converter
     * @return Integer
     */
    private Integer doRequestCount(AuthTplDTO authTpl, String url, Function<Map, Integer> converter){
        ResponseEntity response = invokeRestAgentExecutor(authTpl, url);
        return ResponseEntity.OK.equals(response.getCode())
                ? converter.apply((Map)response.getResult()) : 0;
    }

    /**
     * 组织请求体并执行
     * @param authTpl
     * @param url
     * @return
     */
    private ResponseEntity invokeRestAgentExecutor(AuthTplDTO authTpl, String url) {
        return invokeRestAgentExecutor(authTpl, url, null);
    }

    /**
     * 组织请求体并执行
     * @param authTpl
     * @param url
     * @param params
     * @return
     */
    private ResponseEntity invokeRestAgentExecutor(AuthTplDTO authTpl, String url, Map<String, Object> params) {
        AgentClient.AgentClientBuilder agentClientBuilder = new AgentClient.AgentClientBuilder<ResponseEntity>(
                getAuthTemplate(authTpl), url)
                .method(Method.GET)
                .headers(getUserHeaders())
                .parameterTypeReference(new ParameterTypeReference<ResponseEntity>() {});
        if(null != params) {
            agentClientBuilder.parameters(params);
        }

        return (ResponseEntity) restAgentExecutor.execute(agentClientBuilder.builder());
    }

    /**
     * 组织放置到请求头的用户信息
     * @return
     */
    private Map<String, Object> getUserHeaders(){
        Map<String, Object> headers = new HashMap();

        IAuthorizedUser user = UserInfoHolder.get();
        if(null != user) {
            try {
                String value = CompressionUtils.encode(JsonUtils.toJson(user, TokenView.class));
                headers.put(UserType.USER.userinfoKey(), value);
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
        }
        return headers;
    }

    /**
     * 构造授权模版
     * @param authTpl
     * @return
     */
    private AuthTemplate getAuthTemplate(AuthTplDTO authTpl) {
        return new AuthTemplate(authTpl.getEnv(), authTpl.getTemplateCode(), AuthTemplateType.JWT_AUTH, AuthContentPlaceType.LOCAl);
    }

}
