package com.xforceplus.ultraman.datarule.sync.store;

import com.alicp.jetcache.Cache;
import com.alicp.jetcache.CacheManager;
import com.alicp.jetcache.RefreshPolicy;
import com.alicp.jetcache.anno.CacheType;
import com.alicp.jetcache.template.QuickConfig;
import com.xforceplus.ultraman.datarule.core.property.DataRuleProperties;
import com.xforceplus.ultraman.datarule.domain.dto.v2.AppDataRuleV2DTO;
import com.xforceplus.ultraman.datarule.domain.dto.DataRuleApiAuthTemplateDTO;
import com.xforceplus.ultraman.datarule.domain.dto.v2.EntityActionRuleDetailDTO;
import com.xforceplus.ultraman.datarule.domain.enums.EntityActionType;
import com.xforceplus.ultraman.datarule.domain.pojo.LoginInfo;
import com.xforceplus.ultraman.datarule.sync.provider.AppDataRuleProvider;

import javax.annotation.PostConstruct;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * CopyRight: 上海云砺信息科技有限公司
 * User: youyifan
 * DateTime: 2022/9/11 22:18
 * Description:
 * History: 
 */
public class DataRuleStore {

    public final Integer APP_DATA_RULE_CACHE_LOCAL_LIMIT = 5;

    public final Integer TASK_LOGIN_INFO_CACHE_LOCAL_LIMIT = 100;

    private final Integer STOP_REFRESH_AFTER_LAST_ACCESS = 60;

    public final String APP_DATA_RULE = "appDataRule-env%s";

    private final String APP_RULE_CACHE_KEY = APP_DATA_RULE + "-app%s";

    public final String ENTITY_ACTION_RULE = "entityActionRule-env%s";

    private final String ENTITY_ACTION_RULE_CACHE_KEY = ENTITY_ACTION_RULE + "-user%s-tenant%s-entity%s-action%s";

    private final String ENTITY_ACTION_RULE_COLUMN_CACHE_KEY = ENTITY_ACTION_RULE + "-user%s-tenant%s-entity%s-action%s-columns";

    public final String TASK_ENTITY_ACTION_RULE = "taskEntityActionRule-env%s";

    private final String TASK_ENTITY_ACTION_RULE_CACHE_KEY = TASK_ENTITY_ACTION_RULE + "-user%s-tenant%s-entity%s-action%s";

    private final String TASK_ENTITY_ACTION_RULE_COLUMN_CACHE_KEY = TASK_ENTITY_ACTION_RULE + "-user%s-tenant%s-entity%s-action%s-columns";

    public final String LOGIN_INFO_STORE_KEY = "loginInfoStore-env%s";

    public final String LOGIN_INFO_BY_API_KEY = LOGIN_INFO_STORE_KEY + "-user%s-tenant%s";

    private String appId;
    private String envId;
    private DataRuleProperties dataRuleProperties;
    private String appDataRuleKey;
    private CacheManager cacheManager;
    private AppDataRuleProvider appDataRuleProvider;
    private Cache<String, AppDataRuleV2DTO> appDataRuleCache;
    private Cache<String, EntityActionRuleDetailDTO> entityActionRuleCache;
    private Cache<String, EntityActionRuleDetailDTO> taskEntityActionRuleCache;
    private Cache<String, LoginInfo> loginInfoCache;

    public DataRuleStore(String appId,
                         String env,
                         DataRuleProperties dataRuleProperties,
                         CacheManager cacheManager,
                         AppDataRuleProvider appDataRuleProvider) {
        this.appId = appId;
        this.envId = env;
        this.dataRuleProperties = dataRuleProperties;
        this.appDataRuleProvider = appDataRuleProvider;
        this.cacheManager = cacheManager;

        this.appDataRuleKey = String.format(APP_RULE_CACHE_KEY, this.envId, appId);
    }

    @PostConstruct
    public void init() {
        CacheType cacheType = DataRuleUtil.getCacheType(dataRuleProperties.getCacheType());
        //用于保存应用完整的数据权限
        QuickConfig appDataRuleQC = QuickConfig.newBuilder(String.format(APP_DATA_RULE, envId))
                .cacheType(cacheType)
                .localLimit(APP_DATA_RULE_CACHE_LOCAL_LIMIT)
                .loader((key) -> this.appDataRuleProvider.getAppDataRule(Long.valueOf(appId), Long.valueOf(envId)))
                .refreshPolicy(
                        RefreshPolicy.newPolicy(dataRuleProperties.getCacheRefreshTime(), TimeUnit.MINUTES)
                        .stopRefreshAfterLastAccess(STOP_REFRESH_AFTER_LAST_ACCESS, TimeUnit.MINUTES))
                .penetrationProtect(true)
                .build();
        appDataRuleCache = cacheManager.getOrCreateCache(appDataRuleQC);
        //用于有用上下文的请求，保存解析后的用户对该对象的数据权限
        QuickConfig entityQC = QuickConfig.newBuilder(String.format(ENTITY_ACTION_RULE, envId))
                .cacheType(cacheType)
                .expire(
                        Duration.ofMinutes(dataRuleProperties.getCacheExpireTime()))
                .penetrationProtect(true)
                .build();
        entityActionRuleCache = cacheManager.getOrCreateCache(entityQC);
        //用于内部任务执行时，保存解析后的用户对该对象的数据权限
        QuickConfig taskEntityQC = QuickConfig.newBuilder(String.format(TASK_ENTITY_ACTION_RULE, envId))
                .cacheType(cacheType)
                .expire(
                        Duration.ofMinutes(dataRuleProperties.getCacheExpireTime()))
                .penetrationProtect(true)
                .build();
        taskEntityActionRuleCache = cacheManager.getOrCreateCache(taskEntityQC);
        //用于保存通过接口调用获取的用户信息构造的用户登录信息
        QuickConfig loginInfoQC = QuickConfig.newBuilder(String.format(LOGIN_INFO_STORE_KEY, envId))
                .cacheType(cacheType)
                .localLimit(TASK_LOGIN_INFO_CACHE_LOCAL_LIMIT)
                .expire(
                        Duration.ofMinutes(dataRuleProperties.getCacheExpireTime()))
                .penetrationProtect(true)
                .build();
        loginInfoCache = cacheManager.getOrCreateCache(loginInfoQC);
    }

    /**
     * 获取应用数据规则缓存
     * @return
     */
    public AppDataRuleV2DTO getAppDataRule() {
        return appDataRuleCache.get(this.appDataRuleKey);
    }

    /**
     * 缓存当前用户对象数据规则缓存
     *
     * @param loginId
     * @param tenantCode
     * @param entityId
     * @param entityActionType
     * @param ruleNodeDTOS
     */
    public void putEntityActionRule(String loginId, String tenantCode, Long entityId, EntityActionType entityActionType, EntityActionRuleDetailDTO ruleNodeDTOS) {
        String entityRuleKey = String.format(ENTITY_ACTION_RULE_CACHE_KEY, envId, loginId, tenantCode, entityId, entityActionType);
        entityActionRuleCache.put(entityRuleKey, ruleNodeDTOS);
    }

    public void putEntityActionRuleColumns(String loginId, String tenantCode, Long entityId, EntityActionType entityActionType, EntityActionRuleDetailDTO ruleNodeDTOS) {
        String entityRuleKey = String.format(ENTITY_ACTION_RULE_COLUMN_CACHE_KEY, envId, loginId, tenantCode, entityId, entityActionType);
        entityActionRuleCache.put(entityRuleKey, ruleNodeDTOS);
    }

    /**
     * 获取当前用户对象数据规则缓存
     *
     * @param loginId
     * @param tenantCode
     * @param entityId
     * @param entityActionType
     * @return
     */
    public EntityActionRuleDetailDTO getEntityActionRule(String loginId, String tenantCode, Long entityId, EntityActionType entityActionType) {
        String entityRuleKey = String.format(ENTITY_ACTION_RULE_CACHE_KEY, envId, loginId, tenantCode, entityId, entityActionType);
        return entityActionRuleCache.get(entityRuleKey);
    }

    public EntityActionRuleDetailDTO getEntityActionRuleColumns(String loginId, String tenantCode, Long entityId, EntityActionType entityActionType) {
        String entityRuleKey = String.format(ENTITY_ACTION_RULE_COLUMN_CACHE_KEY, envId, loginId, tenantCode, entityId, entityActionType);
        return entityActionRuleCache.get(entityRuleKey);
    }

    /**
     * 缓存执行任务时给定用户对象数据规则缓存
     *
     * @param userId
     * @param tenantCode
     * @param entityId
     * @param entityActionType
     * @param ruleNodeDTOS
     */
    public void putTaskEntityActionRule(Long userId, String tenantCode, Long entityId, EntityActionType entityActionType, EntityActionRuleDetailDTO ruleNodeDTOS) {
        String entityRuleKey = String.format(TASK_ENTITY_ACTION_RULE_CACHE_KEY, envId, userId, tenantCode, entityId, entityActionType);
        taskEntityActionRuleCache.put(entityRuleKey, ruleNodeDTOS);
    }

    public void putTaskEntityActionRuleColumns(Long userId, String tenantCode, Long entityId, EntityActionType entityActionType, EntityActionRuleDetailDTO ruleNodeDTOS) {
        String entityRuleKey = String.format(TASK_ENTITY_ACTION_RULE_COLUMN_CACHE_KEY, envId, userId, tenantCode, entityId, entityActionType);
        taskEntityActionRuleCache.put(entityRuleKey, ruleNodeDTOS);
    }

    /**
     * 获取执行任务时给定用户对象数据规则缓存
     *
     * @param userId
     * @param tenantCode
     * @param entityId
     * @param entityActionType
     * @return
     */
    public EntityActionRuleDetailDTO getTaskEntityActionRule(Long userId, String tenantCode, Long entityId, EntityActionType entityActionType) {
        String entityRuleKey = String.format(TASK_ENTITY_ACTION_RULE_CACHE_KEY, envId, userId, tenantCode, entityId, entityActionType);
        return taskEntityActionRuleCache.get(entityRuleKey);
    }

    public EntityActionRuleDetailDTO getTaskEntityActionRuleColumns(Long userId, String tenantCode, Long entityId, EntityActionType entityActionType) {
        String entityRuleKey = String.format(TASK_ENTITY_ACTION_RULE_COLUMN_CACHE_KEY, envId, userId, tenantCode, entityId, entityActionType);
        return taskEntityActionRuleCache.get(entityRuleKey);
    }

    /**
     * 清除应用数据权限
     */
    public void clearAppDataRuleCache(){
        appDataRuleCache.remove(this.appDataRuleKey);
    }

    /**
     * 清除当前用户的对象数据权限
     */
    public void clearEntityActionRuleCache(String loginId, String tenantCode, Long entityId, EntityActionType entityActionType) {
        String entityRuleKey = String.format(ENTITY_ACTION_RULE_CACHE_KEY, envId, loginId, tenantCode, entityId, entityActionType);
        entityActionRuleCache.remove(entityRuleKey);
    }

    public void clearEntityActionRuleColumnsCache(String loginId, String tenantCode, Long entityId, EntityActionType entityActionType) {
        String entityRuleKey = String.format(ENTITY_ACTION_RULE_COLUMN_CACHE_KEY, envId, loginId, tenantCode, entityId, entityActionType);
        entityActionRuleCache.remove(entityRuleKey);
    }

    /**
     * 清除执行任务中指定用户的对象数据权限
     */
    public void clearTaskEntityRowRuleCache(String userKey, String tenantCode, Long entityId, EntityActionType entityActionType){
        String entityRuleKey = String.format(ENTITY_ACTION_RULE_CACHE_KEY, envId, userKey, tenantCode, entityId, entityActionType);
        taskEntityActionRuleCache.remove(entityRuleKey);
    }

    /**
     * 保存缓存的用户信息
     * @param tenantId
     * @param userKey  userId or userCode
     * @param loginInfo
     */
    public void putLoginInfoByApi(Long tenantId, String userKey, LoginInfo loginInfo) {
        String loginInfoKey = String.format(LOGIN_INFO_BY_API_KEY, envId, userKey, tenantId);
        loginInfoCache.put(loginInfoKey, loginInfo);
    }

    /**
     * 获取缓存的用户信息
     * @param tenantId
     * @param userKey
     * @return
     */
    public LoginInfo getLoginInfoByApi(Long tenantId, String userKey) {
        String loginInfoKey = String.format(LOGIN_INFO_BY_API_KEY, envId, userKey, tenantId);
        return loginInfoCache.get(loginInfoKey);
    }

    /**
     * 清除登陆信息
     * @param userKey
     * @param tenantId
     */
    public void clearLoginInfoByApi(String userKey, Long tenantId){
        String loginInfoKey = String.format(LOGIN_INFO_BY_API_KEY, envId, userKey, tenantId);
        loginInfoCache.remove(loginInfoKey);
    }

    /**
     * 获取接口授权模版
     * @return
     */
    public List<DataRuleApiAuthTemplateDTO> getApisAuthTemplates() {
        return getAppDataRule().getApiAuthTemplates();
    }

}
