package com.xforceplus.ultraman.oqsengine.devops.rebuild;

import static com.xforceplus.ultraman.oqsengine.devops.rebuild.constant.ConstantDefine.EMPTY_COLLECTION_SIZE;
import static com.xforceplus.ultraman.oqsengine.devops.rebuild.constant.ConstantDefine.MAX_ALLOW_ACTIVE;
import static com.xforceplus.ultraman.oqsengine.devops.rebuild.constant.ConstantDefine.NULL_UPDATE;
import static com.xforceplus.ultraman.oqsengine.devops.rebuild.enums.BatchStatus.DONE;
import static com.xforceplus.ultraman.oqsengine.devops.rebuild.enums.BatchStatus.ERROR;
import static com.xforceplus.ultraman.oqsengine.devops.rebuild.enums.BatchStatus.RUNNING;
import static com.xforceplus.ultraman.oqsengine.pojo.cdc.constant.CDCConstant.DEFAULT_BATCH_SIZE;

import com.xforceplus.ultraman.oqsengine.common.id.LongIdGenerator;
import com.xforceplus.ultraman.oqsengine.common.pool.ExecutorHelper;
import com.xforceplus.ultraman.oqsengine.devops.rebuild.handler.DefaultDevOpsTaskHandler;
import com.xforceplus.ultraman.oqsengine.devops.rebuild.handler.TaskHandler;
import com.xforceplus.ultraman.oqsengine.devops.rebuild.model.DefaultDevOpsTaskInfo;
import com.xforceplus.ultraman.oqsengine.devops.rebuild.model.DevOpsTaskInfo;
import com.xforceplus.ultraman.oqsengine.devops.rebuild.storage.SQLTaskStorage;
import com.xforceplus.ultraman.oqsengine.pojo.devops.DevOpsCdcMetrics;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.pojo.page.Page;
import com.xforceplus.ultraman.oqsengine.storage.index.IndexStorage;
import com.xforceplus.ultraman.oqsengine.storage.master.MasterStorage;
import java.sql.SQLException;
import java.time.LocalDateTime;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * 重建索引处理逻辑.
 *
 * @author : j.xu 2020/8/26.
 * @since : 1.8
 */
public class DevOpsRebuildIndexExecutor implements RebuildIndexExecutor {
    final Logger logger = LoggerFactory.getLogger(DevOpsRebuildIndexExecutor.class);

    @Resource
    private MasterStorage masterStorage;

    @Resource
    private IndexStorage indexStorage;

    @Resource
    private SQLTaskStorage sqlTaskStorage;

    @Resource(name = "longNoContinuousPartialOrderIdGenerator")
    private LongIdGenerator idGenerator;

    private ZoneOffset zoneOffset = OffsetDateTime.now().getOffset();

    private ExecutorService asyncThreadPool;

    //  最小的check数，即cdc一次同步批次的最大数
    private long doubleCheckDistance = DEFAULT_BATCH_SIZE;


    /**
     * construct.
     */
    public DevOpsRebuildIndexExecutor(int taskSize, int maxQueueSize) {
        asyncThreadPool = new ThreadPoolExecutor(taskSize, taskSize,
            0L, TimeUnit.MILLISECONDS,
            new ArrayBlockingQueue<>(maxQueueSize),
            ExecutorHelper.buildNameThreadFactory("task-threads", false));
    }

    /**
     * set value.
     */
    public void resetDoubleCheckDistance(long distance) {
        if (distance > DEFAULT_BATCH_SIZE) {
            doubleCheckDistance = distance;
        }
    }

    @Override
    public void destroy() throws Exception {
        if (null != asyncThreadPool) {
            ExecutorHelper.shutdownAndAwaitTermination(asyncThreadPool, 3600);
        }
    }

    @Override
    public Collection<DevOpsTaskInfo> rebuildIndexes(Collection<IEntityClass> entityClasses, LocalDateTime start,
                                                     LocalDateTime end) throws Exception {
        List<DevOpsTaskInfo> devOps = new ArrayList<>();
        List<DevOpsTaskInfo> errorTasks = new ArrayList<>();
        for (IEntityClass entityClass : entityClasses) {
            DefaultDevOpsTaskInfo devOpsTaskInfo = pending(entityClass, start, end);

            logger.info("pending rebuildIndex task, maintainId {}, entityClass {}, start {}, end {}",
                devOpsTaskInfo.id(), entityClass.id(), start, end
            );
            try {
                if (NULL_UPDATE == sqlTaskStorage.build(devOpsTaskInfo)) {
                    devOpsTaskInfo.resetMessage("init task failed...");
                    devOpsTaskInfo.resetStatus(ERROR.getCode());
                    errorTasks.add(devOpsTaskInfo);
                } else {
                    devOps.add(devOpsTaskInfo);
                }
            } catch (Exception e) {
                devOpsTaskInfo.resetMessage("init task failed...");
                devOpsTaskInfo.resetStatus(ERROR.getCode());
                errorTasks.add(devOpsTaskInfo);
            }
        }

        asyncThreadPool.submit(() -> {
            devOps.forEach(this::handleTask);
        });

        devOps.addAll(errorTasks);

        return devOps;
    }

    @Override
    public DevOpsTaskInfo rebuildIndex(IEntityClass entityClass, LocalDateTime start, LocalDateTime end) throws Exception {

        //  init
        DefaultDevOpsTaskInfo devOpsTaskInfo = pending(entityClass, start, end);

        logger.info("pending rebuildIndex task, maintainId {}, entityClass {}, start {}, end {}",
            devOpsTaskInfo.id(), entityClass.id(), start, end
        );

        if (NULL_UPDATE == sqlTaskStorage.build(devOpsTaskInfo)) {
            return null;
        }

        asyncThreadPool.submit(() -> {
            handleTask(devOpsTaskInfo);
        });

        return devOpsTaskInfo;
    }

    private void handleTask(DevOpsTaskInfo devOpsTaskInfo) {
        try {
            //  执行主表更新
            int rebuildCount =
                masterStorage.rebuild(devOpsTaskInfo.getEntity(), devOpsTaskInfo.getMaintainid(),
                    devOpsTaskInfo.getStarts(), devOpsTaskInfo.getEnds());

            if (rebuildCount > 0) {
                devOpsTaskInfo.setBatchSize(rebuildCount);
                devOpsTaskInfo.resetStatus(RUNNING.getCode());
                devOpsTaskInfo.resetMessage("TASK PROCESSING");
            } else {
                devOpsTaskInfo.setBatchSize(0);
                devOpsTaskInfo.resetStatus(DONE.getCode());
                devOpsTaskInfo.resetMessage("TASK END");
            }

            sqlTaskStorage.update(devOpsTaskInfo);
        } catch (Exception e) {
            devOpsTaskInfo.resetMessage(e.getMessage());

            try {
                sqlTaskStorage.error(devOpsTaskInfo);
            } catch (SQLException ex) {
                ex.printStackTrace();
            }
        }
    }

    @Override
    public boolean cancel(long maintainId) throws Exception {
        return sqlTaskStorage.cancel(maintainId) > 0;
    }

    @Override
    public Optional<TaskHandler> taskHandler(Long maintainId) throws SQLException {
        return sqlTaskStorage.selectUnique(maintainId).map(this::newTaskHandler);
    }

    @Override
    public Collection<TaskHandler> listActiveTasks(Page page) throws SQLException {
        Collection<DevOpsTaskInfo> taskInfoList = sqlTaskStorage.listActives(page);

        return (null != taskInfoList && EMPTY_COLLECTION_SIZE < taskInfoList.size())
            ? taskInfoList.stream().map(this::newTaskHandler).collect(Collectors.toList()) : new ArrayList<>();
    }

    @Override
    public Optional<TaskHandler> getActiveTask(IEntityClass entityClass) throws SQLException {
        Collection<DevOpsTaskInfo> taskInfoCollection = sqlTaskStorage.selectActive(entityClass.id());
        if (MAX_ALLOW_ACTIVE < taskInfoCollection.size()) {
            throw new SQLException("more than 1 active task error.");
        }

        if (!taskInfoCollection.isEmpty()) {
            return Optional.of(newTaskHandler(taskInfoCollection.iterator().next()));
        }
        return Optional.empty();
    }

    @Override
    public Collection<TaskHandler> listAllTasks(Page page) throws SQLException {
        Collection<DevOpsTaskInfo> taskInfoList = sqlTaskStorage.listAll(page);

        return (null != taskInfoList && EMPTY_COLLECTION_SIZE < taskInfoList.size())
            ? taskInfoList.stream().map(this::newTaskHandler).collect(Collectors.toList()) : new ArrayList<>();
    }

    @Override
    public void sync(Map<Long, DevOpsCdcMetrics> devOpsCdcMetrics) throws SQLException {
        //  提交到异步执行.
        devOpsCdcMetrics.forEach(
            (maintainId, devOpsMetrics) -> {
                asyncThreadPool.submit(() -> {
                    Optional<DevOpsTaskInfo> devOpsTaskInfoOp = null;
                    try {
                        devOpsTaskInfoOp = sqlTaskStorage.selectUnique(maintainId);
                    } catch (SQLException ex) {
                        logger.warn("query task exception, maintainId {}.", maintainId);
                        return;
                    }

                    if (devOpsTaskInfoOp.isPresent()) {
                        DevOpsTaskInfo dt = devOpsTaskInfoOp.get();
                        if (dt.isEnd()) {
                            if (dt.isDone() && dt.getStatus() != DONE.getCode()) {
                                try {
                                    done(dt);
                                } catch (Exception e) {
                                    logger.warn("done task exception, maintainId {}, message {}.", maintainId, e.getMessage());
                                }
                            }
                            return;
                        }

                        boolean needUpdate = false;

                        if (devOpsMetrics.getSuccess() > 0) {
                            needUpdate = true;
                            dt.resetIncrementSize(devOpsMetrics.getSuccess());

                            //  完成数量
                            dt.addFinishSize(devOpsMetrics.getSuccess());
                        }

                        if (devOpsMetrics.getFails() > 0) {
                            needUpdate = true;
                            dt.addErrorSize(devOpsMetrics.getFails());
                        }

                        if (needUpdate) {
                            //  任务存在失败数据.
                            if (dt.getErrorSize() > 0) {
                                try {
                                    dt.resetMessage("task end with error.");
                                    sqlTaskStorage.error(dt);
                                } catch (SQLException ex) {
                                    logger.warn("do task-error exception, maintainId {}.", dt.getMaintainid());
                                }
                            } else {
                                try {
                                    //  任务已完成.
                                    long distance = dt.getBatchSize() - dt.getFinishSize();

                                    if (distance <= 0) {
                                        done(dt);
                                    } else {
                                        dt.resetStatus(RUNNING.getCode());
                                        sqlTaskStorage.update(dt);
                                        //  当条数小于 2048 后启动更新后复检，这样哪怕并发造成状态不同步，最终也会由一个线程完成Done.
                                        if (distance < doubleCheckDistance) {
                                            try {
                                                devOpsTaskInfoOp = sqlTaskStorage.selectUnique(maintainId);
                                                if (devOpsTaskInfoOp.isPresent()) {
                                                    if (devOpsTaskInfoOp.get().getBatchSize()
                                                        <= devOpsTaskInfoOp.get().getFinishSize()) {
                                                        done(devOpsTaskInfoOp.get());
                                                    }
                                                }
                                            } catch (SQLException ex) {
                                                logger.warn("query task exception, maintainId {}.", maintainId);
                                            }
                                        }
                                    }
                                } catch (SQLException ex) {
                                    logger.warn("do task-update exception, maintainId {}.", dt.getMaintainid());
                                }
                            }
                        }
                    }
                });
            }
        );
    }

    private DefaultDevOpsTaskInfo pending(IEntityClass entityClass, LocalDateTime start, LocalDateTime end) {
        return new DefaultDevOpsTaskInfo(
            idGenerator.next(),
            entityClass,
            start.toInstant(zoneOffset).toEpochMilli(),
            end.toInstant(zoneOffset).toEpochMilli()
        );
    }

    private boolean done(DevOpsTaskInfo devOpsTaskInfo) throws SQLException {
        //  删除index脏数据
        try {
            indexStorage.clean(devOpsTaskInfo.getEntity(), devOpsTaskInfo.getMaintainid(),
                devOpsTaskInfo.getStarts(), devOpsTaskInfo.getEnds());
        } catch (Exception e) {
            logger.warn(e.getMessage());
        }

        //  更新状态为完成
        devOpsTaskInfo.resetMessage("success");

        if (sqlTaskStorage.done(devOpsTaskInfo) > NULL_UPDATE) {
            logger.info("task done, maintainId {}, finish batchSize {}",
                devOpsTaskInfo.getMaintainid(), devOpsTaskInfo.getFinishSize());
            ((DefaultDevOpsTaskInfo) devOpsTaskInfo).setStatus(DONE.getCode());
        } else {
            logger.warn("task done error, task update finish status error, maintainId {}",
                devOpsTaskInfo.getMaintainid());
        }
        return true;
    }

    private TaskHandler newTaskHandler(DevOpsTaskInfo taskInfo) {
        return new DefaultDevOpsTaskHandler(sqlTaskStorage, taskInfo);
    }
}
