package com.xforceplus.ultraman.oqsengine.sdk.controller;

import akka.NotUsed;
import akka.stream.javadsl.Source;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.sdk.*;
import com.xforceplus.ultraman.oqsengine.sdk.service.EntityService;
import com.xforceplus.ultraman.oqsengine.sdk.util.EntityClassToGrpcConverter;
import com.xforceplus.ultraman.oqsengine.sdk.vo.dto.ConditionQueryRequest;
import com.xforceplus.ultraman.oqsengine.sdk.vo.dto.RebuildRequestDTO;
import com.xforceplus.ultraman.oqsengine.sdk.vo.dto.TaskInfoDTO;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;

/**
 *
 */
@RequestMapping
public class EntityRebuildController {

    @Autowired
    private EntityRebuildService rebuildService;

    @Autowired
    private EntityService entityService;

    private static TaskInfoDTO MISSING = new TaskInfoDTO("-1", "对象不存在");

    @GetMapping(value = "/rebuild/{taskId}/taskinfo", produces = MediaType.APPLICATION_STREAM_JSON_VALUE)
    public Source<TaskInfoDTO, NotUsed> getTaskStream(@PathVariable Long taskId) {
        return rebuildService.showProgress(ShowTask.newBuilder().setId(taskId).build()).map(this::toTaskInfoDTO);
    }

    @GetMapping(value = "/rebuild/list/active", produces = MediaType.APPLICATION_STREAM_JSON_VALUE)
    public Source<TaskInfoDTO, NotUsed> listActiveTask(ConditionQueryRequest queryRequest) {
        return rebuildService
                .listActiveTasks(QueryPage.newBuilder().setNumber(queryRequest.getPageNo()).setSize(queryRequest.getPageSize()).build())
                .map(this::toTaskInfoDTO);
    }

    @GetMapping(value = "/rebuild/entity/id/{id}/active")
    public CompletionStage<TaskInfoDTO> showCurrentTaskbyId(@PathVariable String id) {
        return entityService.load(id).map(this::getActiveTask)
                .orElse(CompletableFuture.completedFuture(MISSING));
    }

    @GetMapping(value = "/rebuild/entity/code/{code}/active")
    public CompletionStage<TaskInfoDTO> showCurrentTaskByCode(@PathVariable String code) {
        return entityService.loadByCode(code).map(this::getActiveTask)
                .orElse(CompletableFuture.completedFuture(MISSING));
    }

    private CompletionStage<TaskInfoDTO> getActiveTask(IEntityClass entityClass) {
        return rebuildService.getActiveTask(EntityClassToGrpcConverter.toEntityUp(entityClass)).thenApply(this::toTaskInfoDTO);
    }

    @GetMapping(value = "/rebuild/list/all", produces = MediaType.APPLICATION_STREAM_JSON_VALUE)
    public Source<TaskInfoDTO, NotUsed> listAllTask(ConditionQueryRequest queryRequest) {
        return rebuildService.listAllTasks(QueryPage
                .newBuilder().setNumber(queryRequest.getPageNo()).setSize(queryRequest.getPageSize()).build())
                .map(this::toTaskInfoDTO);
    }

    @PostMapping("/rebuild/{taskId}/cancel")
    public CompletionStage<TaskInfoDTO> cancel(@PathVariable String taskId) {
        return rebuildService.cancelTask(ShowTask.newBuilder().setId(Long.parseLong(taskId)).build()).thenApply(this::toTaskInfoDTO);
    }

    @PostMapping(value = "/rebuild/")
    public CompletionStage<TaskInfoDTO> requestTask(@RequestBody RebuildRequestDTO rebuildRequest) {

        String code = rebuildRequest.getCode();
        String id = rebuildRequest.getId();

        /**
         * yyyyMMddHHmmSS
         */
        String startDate = rebuildRequest.getStartDate();
        String endDate = rebuildRequest.getEndDate();

        /**
         *
         */
        Optional<IEntityClass> entityClass = Optional.empty();

        if (StringUtils.isEmpty(id)) {
            if (StringUtils.isNoneEmpty(code)) {
                entityClass = entityService.loadByCode(code);
            }
        } else {
            entityClass = entityService.load(id);
        }

        return entityClass.map(x -> {

            EntityUp entityUp = EntityClassToGrpcConverter.toEntityUp(x);

            return rebuildService.rebuildIndex(RebuildRequest.newBuilder()
                    .setEntity(entityUp)
                    .setStart(startDate)
                    .setEnd(endDate)
                    .build()).thenApply(this::toTaskInfoDTO);
        }).orElse(CompletableFuture.completedFuture(MISSING));
    }

    private TaskInfoDTO toTaskInfoDTO(RebuildTaskInfo rebuildTaskInfo) {
        TaskInfoDTO dto = new TaskInfoDTO();
        if (StringUtils.isNoneEmpty(rebuildTaskInfo.getErrCode()) && Integer.parseInt(rebuildTaskInfo.getErrCode()) < 0) {
            //error condition
            dto.setErrCode(rebuildTaskInfo.getErrCode());
            dto.setMessage(rebuildTaskInfo.getMessage());
            return dto;
        } else {
            dto.setBatchSize(rebuildTaskInfo.getBatchSize());
            dto.setCancel(rebuildTaskInfo.getIsCancel());
            dto.setDone(rebuildTaskInfo.getIsDone());
            dto.setEnds(Long.toString(rebuildTaskInfo.getEnds()));
            dto.setStarts(Long.toString(rebuildTaskInfo.getStarts()));
            dto.setTid(Long.toString(rebuildTaskInfo.getTid()));
            dto.setPercentage(rebuildTaskInfo.getPercentage());
            dto.setFinishSize(rebuildTaskInfo.getFinishSize());
            dto.setEntityId(Long.toString(rebuildTaskInfo.getEntityId()));
            dto.setStatus(rebuildTaskInfo.getStatus());
        }
        return dto;
    }
}
