package com.xforceplus.ultraman.oqsengine.sdk.handler;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityField;
import com.xforceplus.ultraman.oqsengine.pojo.reader.IEntityClassReader;
import com.xforceplus.ultraman.oqsengine.pojo.reader.record.Record;
import com.xforceplus.ultraman.oqsengine.sdk.command.*;
import com.xforceplus.ultraman.oqsengine.sdk.config.AuthSearcherConfig;
import com.xforceplus.ultraman.oqsengine.sdk.query.dsl.ExpContext;
import com.xforceplus.ultraman.oqsengine.sdk.query.dsl.ExpFactory;
import com.xforceplus.ultraman.oqsengine.sdk.query.dsl.ExpField;
import com.xforceplus.ultraman.oqsengine.sdk.query.dsl.ExpRel;
import com.xforceplus.ultraman.oqsengine.sdk.service.EntityService;
import com.xforceplus.ultraman.oqsengine.sdk.service.ExecutionService;
import com.xforceplus.ultraman.oqsengine.sdk.service.PagePermissionService;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.EntityExportService;
import com.xforceplus.ultraman.oqsengine.sdk.store.repository.PageBoMapLocalStore;
import com.xforceplus.ultraman.oqsengine.sdk.ui.DefaultUiService;
import com.xforceplus.ultraman.oqsengine.sdk.vo.DataCollection;
import com.xforceplus.ultraman.oqsengine.sdk.vo.dto.ConditionQueryRequest;
import com.xforceplus.ultraman.oqsengine.sdk.vo.dto.permission.ConditionExp;
import com.xforceplus.ultraman.oqsengine.sdk.vo.dto.permission.RuleResult;
import com.xforceplus.xplat.galaxy.framework.context.ContextKeys;
import com.xforceplus.xplat.galaxy.framework.context.ContextService;
import com.xforceplus.xplat.galaxy.framework.dispatcher.anno.QueryHandler;
import com.xforceplus.xplat.galaxy.framework.dispatcher.messaging.MetaData;
import com.xforceplus.xplat.galaxy.framework.dispatcher.messaging.QueryMessage;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import io.vavr.control.Either;
import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVRecord;
import org.apache.commons.text.StringEscapeUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * default ui service handler
 */
public class DefaultEntityServiceHandler implements DefaultUiService {

    @Autowired
    private EntityService entityService;

    @Autowired
    private ExecutionService executionService;

    @Autowired
    private EntityExportService exportService;

    @Autowired
    private PagePermissionService pagePermissionService;

    @Autowired
    private PageBoMapLocalStore pageBoMapLocalStore;

    @Autowired
    private ObjectMapper mapper;

    @Autowired
    private ContextService contextService;

    @Autowired
    private AuthSearcherConfig auth;

    @Value("${xplat.oqsengine.sdk.permission.enabled:false}")
    private boolean enabled;

    private Logger logger = LoggerFactory.getLogger(DefaultUiService.class);

    private static final String MISSING_ENTITIES = "查询对象不存在";

    private Logger log = LoggerFactory.getLogger(DefaultUiService.class);

    public Optional<IEntityClass> getEntityClass(MetaDataLikeCmd cmd) {
        return
                Optional
                        .ofNullable(cmd.version()).map(x -> {
                    return entityService.load(cmd.getBoId(), cmd.version());
                }).orElseGet(() -> entityService.load(cmd.getBoId()));
    }

    @QueryHandler(isDefault = true)
    @Override
    public Either<String, Map<String, Object>> singleQuery(SingleQueryCmd cmd) {

        Optional<IEntityClass> entityClassOp = getEntityClass(cmd);

        if (entityClassOp.isPresent()) {
            return entityService.findOne(entityClassOp.get(), Long.parseLong(cmd.getId()));
        } else {
            return Either.left(MISSING_ENTITIES);
        }
    }

    @QueryHandler(isDefault = true)
    @Override
    public Either<String, Integer> singleDelete(SingleDeleteCmd cmd) {
        Optional<IEntityClass> entityClassOp = getEntityClass(cmd);

        if (entityClassOp.isPresent()) {
            return entityService.deleteOne(entityClassOp.get(), Long.valueOf(cmd.getId()));
        } else {
            return Either.left(MISSING_ENTITIES);
        }
    }

    @QueryHandler(isDefault = true)
    @Override
    public Either<String, Long> singleCreate(SingleCreateCmd cmd) {

        Optional<IEntityClass> entityClassOp = getEntityClass(cmd);

        if (entityClassOp.isPresent()) {
            return entityService.create(entityClassOp.get(), cmd.getBody());
        } else {
            return Either.left(MISSING_ENTITIES);
        }
    }

    @QueryHandler(isDefault = true)
    @Override
    public Either<String, Integer> singleUpdate(SingleUpdateCmd cmd) {

        Optional<IEntityClass> entityClassOp = getEntityClass(cmd);

        if (entityClassOp.isPresent()) {
            return entityService.updateById(entityClassOp.get(), cmd.getId(), cmd.getBody());
        } else {
            return Either.left(MISSING_ENTITIES);
        }
    }

    @QueryHandler(isDefault = true)
    @Override
    public Either<String, Tuple2<Integer, List<Map<String, Object>>>> conditionSearch(ConditionSearchCmd cmd) {

        /**
         * get related entityClass
         */
        Optional<IEntityClass> entityClassOp = getEntityClass(cmd);

        /**
         * relList
         */
        Optional<ExpRel> queryFromPermission = Optional.empty();

        /**
         * check has page to check the permssion
         */
        String pageCode = cmd.getPageCode();

        if (pageCode != null && !pageCode.isEmpty() && enabled) {

            //turn page code to id
//            DataSet ds = pageBoMapLocalStore.query().select("id")
//                    .where("code").eq(pageCode).execute();
//            List<Row> rows = ds.toRows();
//            if (!rows.isEmpty()) {
//                Row row = rows.get(0);
//                Long pageId = RowUtils.getRowValue(row, "id").map(Object::toString).map(Long::parseLong).orElse((long) 0);
            Long tenantId = contextService.get(ContextKeys.LongKeys.TENANT_ID);
            String tenantCode = contextService.get(ContextKeys.StringKeys.TENANTCODE_KEY);
            Collection<Long> roles = contextService.get(ContextKeys.CollectionKeys.ROLE_IDS);

            if (tenantId != null && roles != null && !StringUtils.isEmpty(tenantCode)) {

                long start = System.currentTimeMillis();
                List<RuleResult> ruleResults = pagePermissionService.getRuleResults(
                        tenantId
                        , null
                        , auth.getAppId()
                        , pageCode
                        , tenantCode
                        , null
                        , new ArrayList<>(roles));
                logger.debug("Get Permission using {}ms", System.currentTimeMillis() - start);

                /**
                 * turn permission to ExpRel
                 * filter only this entityId
                 */
                queryFromPermission = ruleResults.stream()
                        .filter(x -> x.getEntityId().equals(Long.parseLong(cmd.getBoId())))
                        .flatMap(x -> {
                            try {
                                //group by entityId
                                String rowRule = x.getRowRule();
                                String columnRule = x.getColumnRule();
                                /**
                                 *  List OR
                                 *     List AND
                                 *        <ConditionExp>
                                 */
                                List<List<ConditionExp>> conditionExps = mapper.readValue(rowRule, new TypeReference<List<List<ConditionExp>>>() {
                                });
                                List<String> columns = mapper.readValue(columnRule, new TypeReference<List<String>>() {
                                });

                                return conditionExps.stream().map(andList -> ExpFactory.createFrom(columns, andList));

                            } catch (Exception ex) {
                                return null;
                            }
                        }).filter(Objects::nonNull)
                        .map(x -> (ExpRel) x)
                        .reduce(ExpRel::mergeOr);
//                }
            }
        }

        ConditionQueryRequest conditionQueryRequest = cmd.getConditionQueryRequest();

        /**
         * exp rel connected by and
         */
        ExpRel condition = ExpFactory.createFrom(conditionQueryRequest);

        //combine expRel to AND tree

        if (queryFromPermission.isPresent()) {
            condition = condition.mergeAnd(queryFromPermission.get());
        }

        if (entityClassOp.isPresent()) {

            /**
             * build context for current execution env
             */
            ExpContext expContext = new ExpContext()
                    .withSchema(entityClassOp.get())
                    .withContext(contextService.getAll());

            Long queryStart = System.currentTimeMillis();
            Either<String, DataCollection<Record>> query = executionService.query(expContext, condition);
            logger.info("Query using {}ms", System.currentTimeMillis() - queryStart);

            //TODO!!!!
            ExpRel finalCondition = condition;
            return query.map(x -> {
                return Tuple.of(x.getRowNum(), x.getRows().stream()
                        //.map(y -> y.toMap(finalCondition.getProjects().stream().map(p -> ((ExpField) p).getName()).collect(Collectors.toSet())))
                        //TODO
                        .map(y -> y.toMap(finalCondition.getProjects().stream().map(p -> ((ExpField) p).getName()).collect(Collectors.toSet())))
                        .collect(Collectors.toList()));
            });
        } else {
            return Either.left(MISSING_ENTITIES);
        }
    }

    /**
     * export command
     * either a error msg and a link
     * Notice this is a async calling
     *
     * @param message
     * @return
     */
    @QueryHandler(isDefault = true)
    @Override
    public CompletableFuture<Either<String, String>> conditionExport(QueryMessage<ConditionExportCmd, ?> message) {
        //TODO check this performance

        ConditionExportCmd cmd = message.getPayload();

        MetaData metaData = message.getMetaData();

        //#26 user-center not support Chinese
        //remove
        Long currentTime = System.nanoTime();

        String token = Optional.ofNullable(metaData.get("code")).map(Object::toString)
                .orElse(cmd.getBoId()).trim() + "-" + currentTime;

        String fileName = Optional.ofNullable(metaData.get("name")).map(Object::toString).map(String::trim)
                .orElse(Optional.ofNullable(metaData.get("code")).map(Object::toString)
                        .orElse(cmd.getBoId())).trim() + "-" + currentTime;

        Optional<? extends IEntityClass> iEntityClassOp;

        if (cmd.getBoId() != null) {
            iEntityClassOp = entityService.load(cmd.getBoId());
        } else {
            iEntityClassOp = entityService.load(cmd.getBoId(), cmd.version());
        }

        if (!iEntityClassOp.isPresent()) {
            return CompletableFuture.completedFuture(Either.left(MISSING_ENTITIES));
        } else {
            //TODO
            return exportService.export(iEntityClassOp.get(), cmd.getConditionQueryRequest()
                    , token, fileName, new HashMap<>(metaData), cmd.getExportType(), cmd.getAppId());
        }
    }

    @QueryHandler(isDefault = true)
    @Override
    public Either<String, InputStream> importTemplate(GetImportTemplateCmd cmd) {

        String boId = cmd.getBoId();

        Optional<IEntityClass> entityClass = entityService.load(boId);

        if (entityClass.isPresent()) {
            IEntityClassReader reader = new IEntityClassReader(entityClass.get());
            Collection<? extends IEntityField> fields = reader.columns();

            byte[] bom = new byte[]{(byte) 0xef, (byte) 0xbb, (byte) 0xbf};
            String columns = fields.stream()
                    .map(x -> x.cnName() + "[" + x.name() + "]")
                    .map(StringEscapeUtils::escapeCsv)
                    .collect(Collectors.joining(","));
            byte[] bytes = columns.getBytes(StandardCharsets.UTF_8);

            byte[] allBytes = new byte[bom.length + bytes.length];

            for (int i = 0; i < allBytes.length; ++i) {
                allBytes[i] = i < bom.length ? bom[i] : bytes[i - bom.length];
            }

            return Either.right(new ByteArrayInputStream(allBytes));

        } else {
            return Either.left(MISSING_ENTITIES);
        }
    }

    /**
     * extract name from headerPart
     *
     * @param headerPart
     * @return
     */
    private String getKeyFromHeader(String headerPart) {
        int start = headerPart.indexOf("[");
        int end = headerPart.indexOf("]");

        if (start < 0 || end < 0) {
            return headerPart;
        } else {
            return headerPart.substring(start + 1, end);
        }
    }

    /**
     * @param cmd
     * @return
     */
    @QueryHandler(isDefault = true)
    @Override
    public Either<String, String> batchImport(ImportCmd cmd) {

        String boId = cmd.getBoId();

        Optional<IEntityClass> entityClassOp = entityService.load(boId);

        if (entityClassOp.isPresent()) {

            MultipartFile file = cmd.getFile();

            Either<String, String> ret;
            try {
                CSVParser parser = CSVParser.parse(file.getInputStream()
                        , StandardCharsets.UTF_8, CSVFormat.EXCEL);

                List<CSVRecord> list = parser.getRecords();

                Map<String, Object> map = new HashMap<>();


                if (list.size() > 1) {
                    CSVRecord header = list.get(0);

                    ret = entityService.transactionalExecute(() -> {
                        for (int i = 1; i < list.size(); i++) {
                            CSVRecord record = list.get(i);
                            for (int j = 0; j < header.size(); j++) {
                                map.put(getKeyFromHeader(header.get(j)), StringUtils.isEmpty(record.get(j)) ? null : record.get(j));
                            }
                            entityService.create(entityClassOp.get(), map);
                        }
                        return "ok";
                    });
                } else {
                    ret = Either.right("ok");
                }
                parser.close();
            } catch (IOException e) {
                e.printStackTrace();
                ret = Either.left(e.getMessage());
            }

            return ret;
        } else {
            return Either.left(MISSING_ENTITIES);
        }
    }
}
