package com.xforceplus.ultraman.oqsengine.sdk.listener;

import com.xforceplus.ultraman.metadata.grpc.Base;
import com.xforceplus.ultraman.metadata.grpc.CheckServiceClient;
import com.xforceplus.ultraman.metadata.grpc.ModuleUp;
import com.xforceplus.ultraman.metadata.grpc.ModuleUpResult;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.pojo.error.ERROR;
import com.xforceplus.ultraman.oqsengine.sdk.EntityService;
import com.xforceplus.ultraman.oqsengine.sdk.EntityUp;
import com.xforceplus.ultraman.oqsengine.sdk.OperationResult;
import com.xforceplus.ultraman.oqsengine.sdk.autoconfigurer.GlobalInited;
import com.xforceplus.ultraman.oqsengine.sdk.config.AuthSearcherConfig;
import com.xforceplus.ultraman.oqsengine.sdk.event.*;
import com.xforceplus.ultraman.oqsengine.sdk.store.repository.CurrentUpdateService;
import com.xforceplus.ultraman.oqsengine.sdk.store.repository.MetadataRepository;
import com.xforceplus.ultraman.oqsengine.sdk.store.repository.impl.MetadataRepositoryInMemoryImpl;
import com.xforceplus.ultraman.oqsengine.sdk.util.EntityClassToGrpcConverter;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.event.EventListener;
import org.springframework.scheduling.annotation.Async;
import org.springframework.util.concurrent.CompletableToListenableFutureAdapter;
import org.springframework.util.concurrent.ListenableFuture;

import java.time.Duration;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import static com.xforceplus.ultraman.oqsengine.sdk.util.CompletableFutureUtils.sequence;
import static com.xforceplus.ultraman.oqsengine.sdk.util.CompletableFutureUtils.within;

/**
 * module event listener
 */
public class ModuleEventListener {

    @Autowired
    private MetadataRepository store;

    @Autowired
    private CheckServiceClient checkServiceClient;

    @Autowired
    private EntityService entityService;

    @Autowired
    private AuthSearcherConfig config;

    Logger logger = LoggerFactory.getLogger(ModuleEventListener.class);

    @Autowired
    public ApplicationEventPublisher publisher;

    @Autowired
    public CurrentUpdateService currentUpdateService;

    @Autowired
    public GlobalInited globalInited;

    @Value("${xplat.meta.oqsengine.sdk.preparation:10}")
    public int prepareTimeInSec;

    //@Async
    @EventListener(MetadataModuleGotEvent.class)
    public ListenableFuture<List<MetadataModulePrepareResultEvent>> saveMetadata(MetadataModuleGotEvent event) {

        List<ModuleUpResult> moduleUpResults = Optional.ofNullable(event.getResponse()).orElseGet(Collections::emptyList);

        MetadataRepository metadataRepository = new MetadataRepositoryInMemoryImpl(1, null);

        List<CompletableFuture<MetadataModulePrepareResultEvent>> retFutures = moduleUpResults.stream().map(module -> {
            logger.debug("Got New Module {}", event);
            //current no
            metadataRepository.save(module, "1", "1");
            List<IEntityClass> allEntities = metadataRepository.findAllEntities();

            List<CompletableFuture<Tuple2<EntityUp, OperationResult>>> results = new LinkedList<>();

            allEntities.forEach(x -> {

                EntityUp entityUp = EntityClassToGrpcConverter.toEntityUp(x);
                CompletableFuture<Tuple2<EntityUp, OperationResult>> prepare = entityService
                        .prepare(entityUp).thenApply(rep -> Tuple.of(entityUp, rep)).toCompletableFuture();
                CompletableFuture<Tuple2<EntityUp, OperationResult>> withinTimeout = within(prepare, Duration.ofSeconds(prepareTimeInSec))
                        .exceptionally(throwable -> {
                            logger.error("prepare {} timeout, throwable {}", x.code(), throwable);
                            return Tuple.of(entityUp, OperationResult.newBuilder()
                                    .setCode(OperationResult.Code.NETWORK_ERR)
                                    .build());
                        });
                results.add(withinTimeout);
            });

            return sequence(results).thenApply(x -> {
                if (x.stream().allMatch(r -> r._2().getCode() == OperationResult.Code.OK)) {
                    //store.save(module, event.getRequest().getTenantId(), event.getRequest().getAppId());
                    logger.info("Module {}:{} saved ", module.getId(), module.getCode());
                    //retList.add();
                    return new MetadataModulePreparedEvent(module
                            , event.getRequest().getTenantId()
                            , event.getRequest().getAppId());
                } else {
                    logger.error("Module {} version {} is invalid retry later", module.getCode(), module.getVersion());
                    boolean isFatal = x.stream()
                            .filter(r -> (r._2().getCode() != OperationResult.Code.OK))
                            .anyMatch(r -> r._2().getAffectedRow() == ERROR.VALIDATION_ERROR.ordinal()
                                    || r._2().getAffectedRow() == ERROR.REACH_MAX_SLOT.ordinal());

                    if (isFatal) {
                        //do nothing
                        logger.error("Fatal {}", module);
                        return new MetadataModuleFatalErrorEvent(module
                                , event.getRequest().getTenantId()
                                , event.getRequest().getAppId()
                                , x.stream().filter(r -> r._2().getCode() != OperationResult.Code.OK).collect(Collectors.toList()));
                    } else {
                        return new MetadataModulePreparedErrorEvent(module
                                , event.getRequest().getTenantId()
                                , event.getRequest().getAppId()
                                , x.stream().filter(r -> r._2().getCode() != OperationResult.Code.OK).collect(Collectors.toList()));
                    }
                }
            });
        }).collect(Collectors.toList());

        return new CompletableToListenableFutureAdapter<>(sequence(retFutures));
    }

//    @Async
    //@EventListener(value = MetadataModulePrepareResultEvent.class, condition = "#preparedEvent instanceof  T(MetadataModulePreparedEvent)")
    @EventListener(MetadataModulePreparedEvent.class)
    public void cleanUpdateAndSave(MetadataModulePreparedEvent preparedEvent) {
        MetadataModulePreparedEvent preparedEventType = (MetadataModulePreparedEvent) preparedEvent;
        ModuleUpResult module = preparedEventType.getModuleUpResult();
        store.save(module, preparedEventType.getTenantId(), preparedEventType.getAppId());
        globalInited.moduleSaved();
    }

    @Async
    @EventListener(MetadataModulePreparedErrorEvent.class)
    public void recordFailed(MetadataModulePreparedErrorEvent errorEvent) {
        MetadataModulePreparedErrorEvent errorEventType = (MetadataModulePreparedErrorEvent) errorEvent;
        currentUpdateService.recordResults(errorEventType.getModuleUpResult(), errorEventType.getErrorResult());
    }

    @Async
    @EventListener(MetadataModuleVersionMissingEvent.class)
    public void requestMetadata(MetadataModuleVersionMissingEvent event) {

        logger.debug("Got Module Missing {}", event);
        Base.Authorization request = com.xforceplus
                .ultraman.metadata.grpc.Base.Authorization.newBuilder()
                .setAppId(config.getAppId())
                .setEnv(config.getEnv())
                .setTenantId(config.getTenant())
                .build();

        Long module = event.getModuleId();
        String version = event.getVersion();
        ModuleUpResult result = checkServiceClient.check(
                ModuleUp.newBuilder()
                        .setModuleId(module.toString())
                        .setModuleVersion(version)
                        .addAuthorization(request)
                        .build()).toCompletableFuture().join();

        logger.debug("Got Versioned Module {}", result);
        store.save(result, config.getTenant(), config.getAppId());
        logger.debug("Versioned Module saved {}", Optional.ofNullable(result).map(x -> x.getVersion()).orElseGet(() -> "none"));
    }
}
