package com.xforceplus.ultraman.oqsengine.sdk.query.dsl;

import com.xforceplus.ultraman.oqsengine.sdk.vo.dto.*;
import com.xforceplus.ultraman.oqsengine.sdk.vo.dto.permission.ConditionExp;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * factory to transfer other format to ExpDSL
 */
public class ExpFactory {

    private static ExpCondition toCondition(ConditionExp exp) {
        List<ExpNode> expNodes = new LinkedList<>();
        expNodes.add(ExpField.field(exp.getField()));

        if (exp.getValue() != null) {
            List<ExpValue> values = exp.getValue().stream()
                    .filter(Objects::nonNull)
                    .map(ExpValue::literal)
                    .collect(Collectors.toList());
            expNodes.addAll(values);
        } else {
            if (exp.getFrom() != null) {
                expNodes.add(ExpValue.literal(exp.getFrom()));
            }

            if (exp.getTo() != null) {
                expNodes.add(ExpValue.literal(exp.getTo()));
            }
        }

        return ExpCondition.call(ExpOperator.from(ConditionOp.valueOf(exp.getOperator())), expNodes);
    }

    /**
     * from conditionExp
     *
     * @return
     */
    public static ExpQuery createFrom(List<String> columns, List<ConditionExp> exps) {
        Objects.requireNonNull(exps, "Condition expression cannot be null");

        return new ExpQuery()
                .project(columns.stream()
                        .map(ExpField::field)
                        .collect(Collectors.toList()))
                .filters(exps.stream().map(ExpFactory::toCondition).collect(Collectors.toList()));
    }

    /**
     * from ConditionQueryRequest
     *
     * @param request
     * @return
     */
    public static ExpQuery createFrom(ConditionQueryRequest request) {
        Objects.requireNonNull(request, "ConditionQueryRequest cannot be null");
        Conditions conditions = request.getConditions();

        /**
         * default value
         */
        ExpCondition expCondition = ExpCondition.alwaysTrue();

        if (conditions != null) {
            List<SubFieldCondition> entities = conditions.getEntities();
            List<FieldCondition> fields = conditions.getFields();

            /**
             * filter to rel
             * TODO current not use fieldType
             */
            Stream<ExpCondition> expRelStream = Optional.ofNullable(fields)
                    .orElseGet(Collections::emptyList)
                    .stream().map(condition -> {

                        List<ExpValue> nodes = condition.getValue().stream().map(ExpValue::literal).collect(Collectors.toList());
                        ExpField fieldExp = ExpField.field(condition.getCode());

                        List<ExpNode> expNodes = new LinkedList<>();
                        expNodes.add(fieldExp);
                        expNodes.addAll(nodes);

                        return ExpCondition.call(ExpOperator.from(condition.getOperation()), expNodes);
                    });

            /**
             * sub filter to rel
             */
            Stream<ExpCondition> expRelSubStream = Optional.ofNullable(entities)
                    .orElseGet(Collections::emptyList)
                    .stream().flatMap(sub -> {
                        return sub.getFields().stream().map(subCondition -> {
                            List<ExpValue> nodes = subCondition.getValue().stream().map(ExpValue::literal).collect(Collectors.toList());
                            ExpField fieldExp = ExpField.field(sub.getCode() + "." + subCondition.getCode());

                            List<ExpNode> expNodes = new LinkedList<>();
                            expNodes.add(fieldExp);
                            expNodes.addAll(nodes);

                            return ExpCondition.call(ExpOperator.from(subCondition.getOperation()), expNodes);
                        });
                    });

            //combine two stream
            List<ExpNode> expRelList = Stream.concat(expRelStream, expRelSubStream).collect(Collectors.toList());

            if (expRelList.isEmpty()) {
                expCondition = ExpCondition.alwaysTrue();
            } else {
                expCondition = ExpCondition.AND(expRelList);
            }
        }

        /**
         * deal with projection
         */
        List<ExpNode> fieldProject = null;

        /**
         * entityItem is a project
         */
        EntityItem entityItem = request.getEntity();
        List<String> fields = new ArrayList<>();
        if (entityItem != null) {
            List<String> projects = entityItem.getFields();
            fields.addAll(projects);
            if (entityItem.getEntities() != null) {
                List<String> subProjects = entityItem.getEntities().stream()
                        .flatMap(x -> x.getFields().stream().map(y -> "_" + x.getCode() + "." + y))
                        .collect(Collectors.toList());
                fields.addAll(subProjects);
            }

            //convert to projects
            //find the name_mapping and do the mapping
            List<NameMapping> mapping = request.getMapping();

            Map<String, String> nameMapping = Optional.ofNullable(mapping)
                    .orElseGet(Collections::emptyList)
                    .stream().collect(Collectors.toMap(NameMapping::getCode, NameMapping::getText));

            fieldProject = fields.stream().map(x -> {
                ExpField field = ExpField.field(x);
                nameMapping.computeIfPresent(x, (k, v) -> {
                    //side-effect
                    field.as(v);
                    return v;
                });

                return field;
            }).collect(Collectors.toList());
        }

        ExpSort sort = ExpSort.init();

        Optional.ofNullable(request.getSort())
                .orElseGet(Collections::emptyList).forEach(x -> {
            sort.withSort(x.getField(), x.getOrder());
        });


        ExpQuery expQuery = new ExpQuery();

        expQuery.project(Optional.ofNullable(fieldProject).orElseGet(Collections::emptyList));
        expQuery.filters(expCondition);
        expQuery.sort(sort);

        expQuery.range(request.getPageNo(), request.getPageSize());

        return expQuery;
    }
}
