package com.xforceplus.ultraman.oqsengine.sdk.query.dsl;

import com.google.common.collect.Sets;

import java.util.*;
import java.util.stream.Collectors;

/**
 * a query represent
 * - project
 * - filter
 * - sort
 * - range
 */
public class ExpQuery implements ExpRel {

    private List<ExpNode> projects = new LinkedList<>();

    private List<ExpNode> filters = new LinkedList<>();

    private ExpSort sorts;

    private ExpRange range;

    public ExpQuery project(List<ExpNode> nodes) {
        this.projects.addAll(nodes);
        return this;
    }

    public ExpQuery filters(List<ExpNode> nodes) {
        this.filters.addAll(nodes);
        return this;
    }

    public ExpQuery filters(ExpNode condition) {
        this.filters.add(condition);
        return this;
    }

    public ExpQuery sort(ExpSort sorts) {
        this.sorts = sorts;
        return this;
    }

    public ExpQuery range(Integer pageNo, Integer pageSize) {
        range = new ExpRange(pageNo, pageSize);
        return this;
    }

    @Override
    public List<ExpNode> getProjects() {
        return projects;
    }

    @Override
    public List<ExpNode> getFilters() {
        return filters;
    }

    @Override
    public ExpSort getSorts() {
        return sorts;
    }

    @Override
    public ExpRange getRange() {
        return range;
    }

    /**
     * default merge
     * union all projects
     * union all condition
     * union all sorts
     * left range
     *
     * @param rel
     * @return
     */
    @Override
    public ExpRel mergeOr(ExpRel rel) {
        return mergeInternal(rel, 0);
    }

    @Override
    public ExpRel mergeAnd(ExpRel rel) {
        return mergeInternal(rel, 1);
    }

    private ExpRel mergeInternal(ExpRel rel, int type) {

        if (rel instanceof ExpQuery) {

            ExpQuery merged = new ExpQuery();

            ExpQuery left = this;
            ExpQuery right = (ExpQuery) rel;

            Set<ExpNode> projectsLeft = new HashSet<>(left.getProjects());
            Set<ExpNode> projectsRight = new HashSet<>(right.getProjects());

            Set<ExpNode> mergedProjects;
            if (type == 0) {
                //union all
                mergedProjects = Sets.newHashSet(projectsLeft);
                mergedProjects.addAll(projectsRight);
            } else {
                //interceptor
                mergedProjects = Sets.intersection(projectsLeft, projectsRight);
            }

            //make sure ExpNode can be reduced
            List<ExpNode> newProjects = new ArrayList<>(mergedProjects);

            //merge condition
            List<ExpNode> filtersLeft = left.getFilters();
            List<ExpNode> filtersRight = right.getFilters();

            ExpCondition leftAnd = filtersLeft.isEmpty() ? ExpCondition.alwaysTrue() : ExpCondition.call(ExpOperator.AND, filtersLeft);
            ExpCondition rightAnd = filtersRight.isEmpty() ? ExpCondition.alwaysTrue() : ExpCondition.call(ExpOperator.AND, filtersRight);


            ExpCondition unionOr = ExpCondition.call(type == 0 ? ExpOperator.OR : ExpOperator.AND, Arrays.asList(leftAnd, rightAnd));

            /**
             * add all sorts
             */
            List<ExpSort.FieldSort> newSorts = new LinkedList<>(Optional.ofNullable(left.sorts)
                    .map(ExpSort::getSorts)
                    .orElseGet(Collections::emptyList));
            newSorts.addAll(Optional.ofNullable(right.sorts)
                    .map(ExpSort::getSorts)
                    .orElseGet(Collections::emptyList));

            /**
             * ignore page
             */
            merged.project(newProjects)
                    .filters(unionOr)
                    .sort(ExpSort.init().setSorts(newSorts));

            if (this.range != null) {
                merged.range(this.range.getIndex(), this.range.getSize());
            }
            return merged;
        }
        return this;
    }

    /**
     * TODO cannot be
     * all visitor result is store in instance
     * or
     *
     * @param visitor
     * @param <T>
     * @return
     */
    @Override
    public <T> T accept(ExpVisitor<T> visitor) {
        List<T> projectsT = projects.stream().map(visitor::visitGeneral).collect(Collectors.toList());
        List<T> filtersT = filters.stream().map(visitor::visitGeneral).collect(Collectors.toList());
        if (sorts != null) {
            T visit = visitor.visit(sorts);
        }

        if (range != null) {
            T visit = visitor.visit(range);
        }

        return null;
    }
}
