package com.xforceplus.ultraman.oqsengine.sdk.query.transformer.optimizer.utils;

import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.sdk.query.dsl.*;
import org.apache.calcite.rel.RelNode;
import org.apache.calcite.rel.logical.LogicalFilter;
import org.apache.calcite.rel.type.RelDataTypeField;
import org.apache.calcite.rex.*;
import org.apache.calcite.sql.SqlKind;
import org.apache.calcite.sql.SqlOperator;
import org.apache.calcite.tools.FrameworkConfig;
import org.apache.calcite.tools.RelBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import static org.apache.calcite.sql.fun.SqlStdOperatorTable.*;

/**
 * tree to rel
 */
public class ExpTreeToRel {

    private static Logger logger = LoggerFactory.getLogger(ExpTreeToRel.class);

    public static RelNode toRelTree(IEntityClass schema, ExpRel root, FrameworkConfig config) {
        long builderStart = System.currentTimeMillis();
        RelBuilder builder = getBuilder(config);
        long builderEnd = System.currentTimeMillis();
        logger.info("Builder time {}", builderEnd - builderStart);

        long relNodeBuildEnd = System.currentTimeMillis();
        List<ExpNode> filters = root.getFilters();
        logger.info("Builder1 time {}", System.currentTimeMillis() - relNodeBuildEnd);
        RelBuilder scan = builder.scan(schema.code());
        logger.info("Builder2 time {}", System.currentTimeMillis() - relNodeBuildEnd);
        ExpVisitor<RexNode> visitor = new RelTreeVisitor(scan);
        logger.info("Builder3 time {}", System.currentTimeMillis() - relNodeBuildEnd);
        RexNode condition = ExpCondition.call(ExpOperator.OR, filters).accept(visitor);
        logger.info("Builder4 time {}", System.currentTimeMillis() - relNodeBuildEnd);
        RelNode build = scan.filter(condition).build();
        logger.info("Builder5 time {}", System.currentTimeMillis() - relNodeBuildEnd);
        return build;
    }

    private static RelBuilder getBuilder(FrameworkConfig config) {


        long current = System.currentTimeMillis();

        RelBuilder relBuilder = RelBuilder.create(config);

        logger.info("Create Time {}", System.currentTimeMillis() - current);

        return relBuilder;
    }

    public static ExpNode toExpTree(RelNode root, List<RelDataTypeField> fields) {

        if (root instanceof LogicalFilter) {
            return ((LogicalFilter) root).getCondition().accept(new ExpTreeVisitor(fields));
        } else {
            return ExpCondition.alwaysTrue();
        }
    }

    static class ExpTreeVisitor implements RexVisitor<ExpNode> {

        List<RelDataTypeField> fields;

        public ExpTreeVisitor(List<RelDataTypeField> fields) {
            this.fields = fields;
        }

        @Override
        public ExpNode visitInputRef(RexInputRef inputRef) {
            int index = inputRef.getIndex();
            return ExpField.field(fields.get(index).getName());
        }

        @Override
        public ExpNode visitLocalRef(RexLocalRef localRef) {
            return null;
        }

        @Override
        public ExpNode visitLiteral(RexLiteral literal) {
            return ExpValue.literal(literal.getValueAs(String.class));
        }

        private ExpOperator toExpOperator(SqlOperator op) {
            if (op.equals(EQUALS)) {
                return ExpOperator.EQUALS;
            } else if (op.equals(NOT_EQUALS)) {
                return ExpOperator.NOT_EQUALS;
            } else if (op.equals(IN)) {
                return ExpOperator.IN;
            } else if (op.equals(GREATER_THAN)) {
                return ExpOperator.GREATER_THAN;
            } else if (op.equals(GREATER_THAN_OR_EQUAL)) {
                return ExpOperator.GREATER_EQUALS;
            } else if (op.equals(LESS_THAN)) {
                return ExpOperator.LESS_THAN;
            } else if (op.equals(LESS_THAN_OR_EQUAL)) {
                return ExpOperator.LESS_EQUALS;
            } else if (op.equals(LIKE)) {
                return ExpOperator.LIKE;
            } else if (op.equals(NOT_IN)) {
                return ExpOperator.NOT_IN;
            }

            return null;
        }

        @Override
        public ExpNode visitCall(RexCall call) {
            if (call.isA(SqlKind.AND)) {
                //return builder.and(rel.getExpNodes().stream().map(x -> x.accept(this)).collect(Collectors.toList()));
                return ExpCondition.call(ExpOperator.AND, call.getOperands()
                        .stream()
                        .map(x -> x.accept(this))
                        .collect(Collectors.toList()));
            } else if (call.isA(SqlKind.OR)) {
                return ExpCondition.call(ExpOperator.OR, call.getOperands().stream().map(x -> x.accept(this)).collect(Collectors.toList()));
            } else {
                ExpOperator expOperator = toExpOperator(call.getOperator());
                if (expOperator != null) {
                    return ExpCondition.call(expOperator, call.getOperands().stream().map(x -> x.accept(this)).collect(Collectors.toList()));
                } else {
                    logger.warn(call.getOperands() + " is not support ");
                    throw new RuntimeException(call.getOperands() + " is not support ");
                }
            }
        }

        @Override
        public ExpNode visitOver(RexOver over) {
            return null;
        }

        @Override
        public ExpNode visitCorrelVariable(RexCorrelVariable correlVariable) {
            return null;
        }

        @Override
        public ExpNode visitDynamicParam(RexDynamicParam dynamicParam) {
            return null;
        }

        @Override
        public ExpNode visitRangeRef(RexRangeRef rangeRef) {
            return null;
        }

        @Override
        public ExpNode visitFieldAccess(RexFieldAccess fieldAccess) {
            return null;
        }

        @Override
        public ExpNode visitSubQuery(RexSubQuery subQuery) {
            return null;
        }

        @Override
        public ExpNode visitTableInputRef(RexTableInputRef fieldRef) {
            return null;
        }

        @Override
        public ExpNode visitPatternFieldRef(RexPatternFieldRef fieldRef) {
            return null;
        }
    }

    static class RelTreeVisitor implements ExpVisitor<RexNode> {

        private RelBuilder builder;

        public RelTreeVisitor(RelBuilder builder) {
            this.builder = builder;
        }

        @Override
        public RexNode visit(ExpField field) {
            return builder.field(field.getName());
        }

        @Override
        public RexNode visit(ExpCondition rel) {
            if (rel.getOperator() == ExpOperator.AND) {
                return builder.and(rel.getExpNodes().stream().map(x -> x.accept(this)).collect(Collectors.toList()));
            } else if (rel.getOperator() == ExpOperator.OR) {
                return builder.or(rel.getExpNodes().stream().map(x -> x.accept(this)).collect(Collectors.toList()));
            } else {
                if (rel.isAlwaysTrue()) {
                    return builder.call(EQUALS, builder.literal(1), builder.literal(1));
                }

                if (rel.isAlwaysFalse()) {
                    return builder.call(EQUALS, builder.literal(1), builder.literal(0));
                }
                List<SqlOperator> sqlOperators = toSqlOperator(rel.getOperator());
                if (sqlOperators.size() > 0) {
                    //TODO
                    if (sqlOperators.size() == 2) {
                        if (rel.getExpNodes().size() == 3) {
                            //find
                            RexNode left = builder.call(sqlOperators.get(0)
                                    , rel.getExpNodes().get(0).accept(this)
                                    , rel.getExpNodes().get(1).accept(this));

                            RexNode right = builder.call(sqlOperators.get(1)
                                    , rel.getExpNodes().get(0).accept(this)
                                    , rel.getExpNodes().get(2).accept(this));
                            return builder.and(left, right);
                        }
                        //in fact the expNodes should only 3
                    } else {
                        //in fact the expNodes should only 2
                        return builder.call(sqlOperators.get(0)
                                , rel.getExpNodes().stream().map(x -> x.accept(this)).collect(Collectors.toList()));
                    }
                }
            }
            throw new RuntimeException("Invalid tree");
        }

        @Override
        public RexNode visit(ExpValue value) {
            return builder.literal(value.getStrValue());
        }

        @Override
        public RexNode visit(ExpBi bi) {
            return null;
        }

        @Override
        public RexNode visit(ExpSort expSort) {
            return null;
        }

        @Override
        public RexNode visit(ExpRange range) {
            return null;
        }

        /**
         * EQUALS(eq, binary),
         * <p>
         * LIKE(like, binary),
         * <p>
         * IN(in, atLeast1),
         * <p>
         * NOT_IN(ni, atLeast1),
         * <p>
         * GREATER_EQ_AND_LESS_EQ(ge_le, ternary),
         * <p>
         * GREATER_EQ_AND_LESS_THAN(ge_lt, ternary),
         * <p>
         * GREATER_THAN_AND_LESS_EQ(gt_le, ternary),
         * <p>
         * GREATER_THAN_AND_LESS_THAN(gt_lt, ternary),
         * <p>
         * GREATER_THAN(gt, binary),
         * <p>
         * GREATER_EQ(gt, binary),
         * <p>
         * LESS_THAN(lt, binary),
         * <p>
         * LESS_EQUALS(le, binary),
         * <p>
         * NOT_EQUALS(ne, binary);
         *
         * @param operator
         * @return
         */
        private List<SqlOperator> toSqlOperator(ExpOperator operator) {
            switch (operator) {
                case EQUALS:
                    return Collections.singletonList(EQUALS);
                case NOT_EQUALS:
                    return Collections.singletonList(NOT_EQUALS);
                case LIKE:
                    return Collections.singletonList(LIKE);
                case NOT_IN:
                    return Collections.singletonList(NOT_IN);
                case IN:
                    return Collections.singletonList(IN);
                case GREATER_EQUALS:
                    return Collections.singletonList(GREATER_THAN_OR_EQUAL);
                case GREATER_THAN:
                    return Collections.singletonList(GREATER_THAN);
                case LESS_EQUALS:
                    return Collections.singletonList(LESS_THAN_OR_EQUAL);
                case LESS_THAN:
                    return Collections.singletonList(LESS_THAN);
                case GREATER_EQ_AND_LESS_EQ:
                    return Arrays.asList(GREATER_THAN_OR_EQUAL, LESS_THAN_OR_EQUAL);
                case GREATER_THAN_AND_LESS_EQ:
                    return Arrays.asList(GREATER_THAN, LESS_THAN_OR_EQUAL);
                case GREATER_THAN_AND_LESS_THAN:
                    return Arrays.asList(GREATER_THAN, LESS_THAN);
                case GREATER_EQ_AND_LESS_THAN:
                    return Arrays.asList(LESS_THAN_OR_EQUAL, LESS_THAN);
                default:
                    return Collections.emptyList();
            }
        }
    }
}
