package com.xforceplus.ultraman.oqsengine.sdk.query.utils;

import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.impl.ColumnField;
import com.xforceplus.ultraman.oqsengine.pojo.reader.IEntityClassReader;
import com.xforceplus.ultraman.oqsengine.sdk.*;
import com.xforceplus.ultraman.oqsengine.sdk.query.dsl.*;
import com.xforceplus.ultraman.oqsengine.sdk.util.EntityClassToGrpcConverter;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * a helper for rel Tree
 */
public class RelTreeHelper {

    /**
     * tree to grpc tree
     *
     * @param tree
     * @return
     */
    public static SelectByTree relToTree(IEntityClass schema, ExpRel tree, ExpContext context) {
        List<ExpNode> projects = tree.getProjects();
        List<ExpNode> filters = tree.getFilters();
        ExpSort sorts = tree.getSorts();
        ExpRange range = tree.getRange();

        return SelectByTree.newBuilder()
                .setProjects(toProjects(projects, context))
                .setFilters(toFilters(filters, context))
                .setSorts(toSorts(sorts, context))
                .setRange(toRange(range))
                .setEntity(EntityClassToGrpcConverter.toEntityUp(schema))
                .build();
    }

    /**
     * filter node
     * nodes is a and clause
     *
     * @param nodes
     * @return
     */
    private static Filters toFilters(List<ExpNode> nodes, ExpContext context) {
        List<FilterNode> filterNode = nodes.stream()
                .peek(x -> x.setExpContext(context))
                .map(RelTreeHelper::toFilterNode)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());

        return Filters.newBuilder().addAllNodes(filterNode).build();
    }

    private static FilterNode toFilterNode(ExpNode node) {
        if (node instanceof ExpCondition) {
            if (((ExpCondition) node).isAlwaysTrue()) {
                //omit all true
                return null;
            } else {
                return FilterNode.newBuilder()
                        .setNodeType(0)
                        .setOperator(FilterNode.Operator.valueOf(((ExpCondition) node).getOperator().getShortName()))
                        .addAllNodes(((ExpCondition) node)
                                .getExpNodes().stream()
                                .map(RelTreeHelper::toFilterNode)
                                .filter(Objects::nonNull)
                                .collect(Collectors.toList()))
                        .build();
            }
        } else if (node instanceof ExpValue) {
            return FilterNode.newBuilder()
                    .setNodeType(2)
                    .setPayload(((ExpValue) node).getStrValue())
                    .build();
        } else if (node instanceof ExpField) {
            return FilterNode.newBuilder()
                    .setNodeType(1)
                    .setPayload(((ExpField) node).getName())
                    .build();
        }

        return null;
    }

    private static Projects toProjects(List<ExpNode> projects, ExpContext context) {
        return Projects.newBuilder()
                .addAllQueryFields(projects.stream()
                        .peek(x -> x.setExpContext(context))
                        .map(RelTreeHelper::toQueryFieldUp)
                        .filter(Objects::nonNull)
                        .collect(Collectors.toList()))
                .build();
    }

    private static Optional<ColumnField> getColumnField(ExpContext context, String code) {

        if (context != null && context.getSchema() != null) {
            IEntityClassReader schema = context.getSchema();
            return schema.column(code);
        }

        return Optional.empty();
    }

    private static QueryFieldsUp toQueryFieldUp(ExpNode expNode) {
        if (expNode instanceof ExpField) {

            ExpContext context = expNode.getExpContext();
            Optional<ColumnField> column = getColumnField(context, ((ExpField) expNode).getName());
            if (column.isPresent()) {
                return QueryFieldsUp.newBuilder()
                        .setId(column.get().id())
                        .setEntityId(column.get().originEntityClass().id())
                        .setCode(((ExpField) expNode).getName())
                        .build();
            }
        }
        return null;
    }

    private static Sorts toSorts(ExpSort sorts, ExpContext context) {
        return Sorts.newBuilder()
                .addAllSort(sorts.getSorts().stream()
                        .map(x -> RelTreeHelper.toSortNode(x, context))
                        .filter(Objects::nonNull)
                        .collect(Collectors.toList()))
                .build();
    }

    private static SortNode toSortNode(ExpSort.FieldSort sort, ExpContext context) {
        Optional<ColumnField> column = getColumnField(context, sort.getCode());
        return column.map(columnField -> SortNode.newBuilder()
                .setCode(sort.getCode())
                .setFieldId(columnField.id())
                .setOrder(sort.getSort() == ExpSort.Sort.ASCEND ?
                        SortNode.Order.asc : SortNode.Order.desc)
                .build()).orElse(null);
    }

    private static Range toRange(ExpRange range) {

        Range.Builder builder = Range.newBuilder();

        if (range.getIndex() != null) {
            builder.setPageIndex(range.getIndex());
        }

        if (range.getSize() != null) {
            builder.setPageSize(range.getSize());
        }

        return builder.buildPartial();
    }
}
