package com.xforceplus.phoenix.split.domain;

import com.xforceplus.phoenix.split.exception.SplitRuleParamException;
import com.xforceplus.phoenix.split.pojo.OrderItemPojo;
import com.xforceplus.phoenix.split.pojo.OrderPojo;
import com.xforceplus.phoenix.split.pojo.SplitOrderRulePojo;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.List;

public class OrderSplitDomain {

    private static final BigDecimal MAX_ERROR_AMOUNT = new BigDecimal("0.01");

    private OrderPojo order;

    private List<OrderItemPojo> orderItems;

    private SplitOrderRulePojo splitOrderRulePojo;

    private OrderItemPojo lastOrderItem;

    private BigDecimal rate;

    public OrderSplitDomain(OrderPojo order,
                            SplitOrderRulePojo splitOrderRulePojo) {
        this.rate = splitOrderRulePojo.getSplitAmount().divide(order.getAmountWithTax(), 11, BigDecimal.ROUND_DOWN);
        if (rate.compareTo(BigDecimal.ONE) >= 0) {
            throw new SplitRuleParamException("拆分金额必须小于订单含税总金额!");
        }
        this.order = order;
        this.splitOrderRulePojo = splitOrderRulePojo;
        this.orderItems = new ArrayList<>();
        for (int i = 0; i < order.getOrderItems().size() - 1; i++) {
            this.orderItems.add(order.getOrderItems().get(i));
        }
        this.lastOrderItem = order.getOrderItems().get(order.getOrderItems().size() - 1);
    }


    /**
     * 拆分明细逻辑
     * 1.每条明细金额按照比例拆分成两条明细
     * 2.反算单价或者数量
     * 3.累计从每条明细拆分的含税金额
     * 4.计算拆分金额与累计含税金额的差 C
     * 5.从最后一条明细拆分出金额C
     *
     * @return
     */
    public List<OrderPojo> splitOrder() {

        List<OrderItemPojo> firstItems = new ArrayList<>();
        List<OrderItemPojo> secondItems = new ArrayList<>();

        BigDecimal totalSplitAmount = BigDecimal.ZERO;

        for (OrderItemPojo orderItemPojo : orderItems) {
            OrderItemPojo splitOrderItem = splitItem(orderItemPojo);
            firstItems.add(splitOrderItem);
            secondItems.add(orderItemPojo);
            totalSplitAmount = totalSplitAmount.add(splitOrderItem.getAmountWithTax());
        }

        BigDecimal surplusSplitAmount = splitOrderRulePojo.getSplitAmount().subtract(totalSplitAmount);

        firstItems.add(splitLastItem(surplusSplitAmount));
        secondItems.add(lastOrderItem);

        List<OrderPojo> result = new ArrayList<>(2);
        result.add(createOrder(firstItems));
        result.add(createOrder(secondItems));

        return result;
    }

    private OrderPojo createOrder(List<OrderItemPojo> orderItems) {
        OrderPojo orderPojo = new OrderPojo();
        orderPojo.setOrderItems(orderItems);
        orderPojo.setExtInfo(this.order.getExtInfo());

        BigDecimal amountWithTax = BigDecimal.ZERO;
        BigDecimal amountWithoutTax = BigDecimal.ZERO;
        BigDecimal taxAmount = BigDecimal.ZERO;
        for (OrderItemPojo orderItemPojo : orderItems) {
            amountWithTax = amountWithTax.add(orderItemPojo.getAmountWithTax());
            amountWithoutTax = amountWithoutTax.add(orderItemPojo.getAmountWithoutTax());
            taxAmount = taxAmount.add(orderItemPojo.getTaxAmount());
        }

        orderPojo.setAmountWithoutTax(amountWithoutTax);
        orderPojo.setAmountWithTax(amountWithTax);
        orderPojo.setTaxAmount(taxAmount);

        return orderPojo;
    }

    private OrderItemPojo splitLastItem(BigDecimal surplusSplitAmount) {
        OrderItemPojo newOrderItem = new OrderItemPojo();

        Amount amount = calculationAmountByWithTaxAmount(this.lastOrderItem, surplusSplitAmount);
        newOrderItem.setAmountWithoutTax(amount.getWithoutTaxAmount());
        newOrderItem.setTaxAmount(amount.getTaxAmount());
        newOrderItem.setAmountWithTax(amount.getWithTaxAmount());

        splitInnerDiscountAmount(this.lastOrderItem, newOrderItem);
        splitOutterDiscountAmount(this.lastOrderItem, newOrderItem);

        subtractSplitAmount(newOrderItem, this.lastOrderItem);
        processQuantityAndUnitPrice(newOrderItem, this.lastOrderItem);
        newOrderItem.setTaxRate(this.lastOrderItem.getTaxRate());
        newOrderItem.setExtInfo(this.lastOrderItem.getExtInfo());
        return newOrderItem;
    }

    private Amount calculationAmountByWithTaxAmount(OrderItemPojo lastOrderItem, BigDecimal surplusSplitAmount) {
        BigDecimal splitAmountWithoutTax = surplusSplitAmount.divide(BigDecimal.ONE.add(lastOrderItem.getTaxRate()), 2, RoundingMode.HALF_UP);
        BigDecimal splitTaxAmount = surplusSplitAmount.subtract(splitAmountWithoutTax);


        return new Amount(surplusSplitAmount, splitAmountWithoutTax, splitTaxAmount);
    }

    private OrderItemPojo splitItem(OrderItemPojo orderItemPojo) {
        OrderItemPojo newOrderItem = new OrderItemPojo();

        Amount amount = calculationAmountByRate(orderItemPojo.getAmountWithoutTax(), orderItemPojo.getTaxRate());
        newOrderItem.setAmountWithoutTax(amount.getWithoutTaxAmount());
        newOrderItem.setTaxAmount(amount.getTaxAmount());
        newOrderItem.setAmountWithTax(amount.getWithTaxAmount());

        splitInnerDiscountAmount(orderItemPojo, newOrderItem);

        splitOutterDiscountAmount(orderItemPojo, newOrderItem);

        subtractSplitAmount(newOrderItem, orderItemPojo);
        processQuantityAndUnitPrice(newOrderItem, orderItemPojo);
        newOrderItem.setTaxRate(orderItemPojo.getTaxRate());
        newOrderItem.setExtInfo(orderItemPojo.getExtInfo());
        return newOrderItem;
    }

    private void splitOutterDiscountAmount(OrderItemPojo orderItemPojo, OrderItemPojo newOrderItem) {
        Amount amount = calculationAmountByRate(orderItemPojo.getOutterDiscountWithoutTax(), orderItemPojo.getTaxRate());
        newOrderItem.setOutterDiscountWithoutTax(amount.getWithoutTaxAmount());
        newOrderItem.setOutterDiscountTax(amount.getTaxAmount());
        newOrderItem.setOutterDiscountWithTax(amount.getWithTaxAmount());
    }

    private void splitInnerDiscountAmount(OrderItemPojo orderItemPojo, OrderItemPojo newOrderItem) {
        Amount amount = calculationAmountByRate(orderItemPojo.getInnerDiscountWithoutTax(), orderItemPojo.getTaxRate());
        newOrderItem.setInnerDiscountWithoutTax(amount.getWithoutTaxAmount());
        newOrderItem.setInnerDiscountTax(amount.getTaxAmount());
        newOrderItem.setInnerDiscountWithTax(amount.getWithTaxAmount());
    }

    private void subtractSplitAmount(OrderItemPojo newOrderItem, OrderItemPojo orderItemPojo) {
        orderItemPojo.setAmountWithoutTax(orderItemPojo.getAmountWithoutTax().subtract(newOrderItem.getAmountWithoutTax()));
        orderItemPojo.setAmountWithTax(orderItemPojo.getAmountWithTax().subtract(newOrderItem.getAmountWithTax()));
        orderItemPojo.setTaxAmount(orderItemPojo.getTaxAmount().subtract(newOrderItem.getTaxAmount()));

        orderItemPojo.setOutterDiscountWithoutTax(orderItemPojo.getOutterDiscountWithoutTax().subtract(newOrderItem.getOutterDiscountWithoutTax()));
        orderItemPojo.setOutterDiscountWithTax(orderItemPojo.getOutterDiscountWithTax().subtract(newOrderItem.getOutterDiscountWithTax()));
        orderItemPojo.setOutterDiscountTax(orderItemPojo.getOutterDiscountTax().subtract(newOrderItem.getOutterDiscountTax()));

        orderItemPojo.setInnerDiscountWithoutTax(orderItemPojo.getInnerDiscountWithoutTax().subtract(newOrderItem.getInnerDiscountWithoutTax()));
        orderItemPojo.setInnerDiscountWithTax(orderItemPojo.getInnerDiscountWithTax().subtract(newOrderItem.getInnerDiscountWithTax()));
        orderItemPojo.setInnerDiscountTax(orderItemPojo.getInnerDiscountTax().subtract(newOrderItem.getInnerDiscountTax()));

    }

    private void processQuantityAndUnitPrice(OrderItemPojo newOrderItem, OrderItemPojo orderItemPojo) {
        BigDecimal quantity = orderItemPojo.getQuantity();
        BigDecimal unitPrice = orderItemPojo.getUnitPrice();
        if (splitQuantity()) {
            if (nonZero(orderItemPojo.getUnitPrice())) {
                quantity = newOrderItem.getAmountWithoutTax()
                        .divide(orderItemPojo.getUnitPrice(), splitOrderRulePojo.getQuantityScale(), BigDecimal.ROUND_DOWN);

                orderItemPojo.setQuantity(orderItemPojo.getAmountWithoutTax().divide(orderItemPojo.getUnitPrice(), splitOrderRulePojo.getQuantityScale(), BigDecimal.ROUND_DOWN));
                checkQuantityAndUnitPrice(quantity, unitPrice, newOrderItem.getAmountWithoutTax());
                checkQuantityAndUnitPrice(orderItemPojo.getQuantity(), orderItemPojo.getUnitPrice(), orderItemPojo.getAmountWithoutTax());
            }
        } else if (splitUnitPrice()) {
            if (nonZero(orderItemPojo.getQuantity())) {
                unitPrice = newOrderItem.getAmountWithoutTax().divide(orderItemPojo.getQuantity(), splitOrderRulePojo.getUnitPriceScale(), BigDecimal.ROUND_DOWN);
                orderItemPojo.setUnitPrice(orderItemPojo.getAmountWithoutTax().divide(orderItemPojo.getQuantity(), splitOrderRulePojo.getUnitPriceScale(), BigDecimal.ROUND_DOWN));
                checkQuantityAndUnitPrice(quantity, unitPrice, newOrderItem.getAmountWithoutTax());
                checkQuantityAndUnitPrice(orderItemPojo.getQuantity(), orderItemPojo.getUnitPrice(), orderItemPojo.getAmountWithoutTax());
            }
        } else {
            throw new SplitRuleParamException("amountSplitRule值不合法, 合法值[1,2]");
        }
        newOrderItem.setQuantity(quantity);
        newOrderItem.setUnitPrice(unitPrice);

    }

    private void checkQuantityAndUnitPrice(BigDecimal quantity, BigDecimal unitPrice, BigDecimal amountWithoutTax) {
        if (quantity.multiply(unitPrice)
                .setScale(2, BigDecimal.ROUND_HALF_UP)
                .subtract(amountWithoutTax).abs()
                .compareTo(MAX_ERROR_AMOUNT) > 0) {
            throw new SplitRuleParamException("无法拆分订单，拆分后单价*数量与不含税金额相差超过一分");
        }
    }

    private boolean nonZero(BigDecimal bigDecimal) {
        return bigDecimal != null && bigDecimal.compareTo(BigDecimal.ZERO) != 0;
    }

    private boolean splitUnitPrice() {
        return "1".equals(splitOrderRulePojo.getAmountSplitRule());
    }

    private boolean splitQuantity() {
        return "2".equals(splitOrderRulePojo.getAmountSplitRule());
    }


    private Amount calculationAmountByRate(BigDecimal amountWithoutTax, BigDecimal taxRate) {
        BigDecimal splitAmountWithoutTax = amountWithoutTax.multiply(this.rate).setScale(2, RoundingMode.HALF_UP);
        BigDecimal splitTaxAmount = splitAmountWithoutTax.multiply(taxRate).setScale(2, RoundingMode.HALF_UP);
        BigDecimal splitAmountWithTax = splitAmountWithoutTax.add(splitTaxAmount);

        return new Amount(splitAmountWithTax, splitAmountWithoutTax, splitTaxAmount);
    }


    private class Amount {
        private BigDecimal withTaxAmount;
        private BigDecimal withoutTaxAmount;
        private BigDecimal taxAmount;

        Amount(BigDecimal withTaxAmount, BigDecimal withoutTaxAmount, BigDecimal taxAmount) {
            this.withTaxAmount = withTaxAmount;
            this.withoutTaxAmount = withoutTaxAmount;
            this.taxAmount = taxAmount;
        }

        BigDecimal getWithTaxAmount() {
            return withTaxAmount;
        }

        BigDecimal getWithoutTaxAmount() {
            return withoutTaxAmount;
        }

        BigDecimal getTaxAmount() {
            return taxAmount;
        }
    }
}
