package com.xforceplus.phoenix.split.service.dataflow.impl;

import com.xforceplus.phoenix.split.domain.ItemGroup;
import com.xforceplus.phoenix.split.domain.RuleInfo;
import com.xforceplus.phoenix.split.exception.SplitBizException;
import com.xforceplus.phoenix.split.model.BillInfo;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.model.FieldOfObj;
import com.xforceplus.phoenix.split.model.RemarkFieldMetadata;
import com.xforceplus.phoenix.split.service.dataflow.DataProcessPlugin;
import com.xforceplus.phoenix.split.util.CommonTools;
import com.xforceplus.phoenix.split.util.FiledUtils;
import org.apache.ibatis.reflection.ReflectionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.nio.charset.Charset;
import java.util.List;
import java.util.Objects;

/**
 * 明细名称处理，可根据拆票规则配置字段拼接名称
 */
@Component
public class FieldProcessPlugin implements DataProcessPlugin {

    private Logger logger = LoggerFactory.getLogger(FieldProcessPlugin.class);

    @Value("${cargoNameLengthLimit:100}")
    private int cargoNameLengthLimit;


    @Value("${cargoNameLengthLimit:40}")
    private int itemSpecNameLengthLimit;


    private static final Charset GBK = Charset.forName("GBK");


    @Override
    public List<ItemGroup> processData(List<ItemGroup> itemGroups, BillInfo billInfo, RuleInfo ruleInfo) {
        List<RemarkFieldMetadata> cargoNameFieldMetadatas = ruleInfo.getSplitRule().getCargoNameFiledMetadatas();
        List<RemarkFieldMetadata> itemSpecFieldMetadatas = ruleInfo.getSplitRule().getItemSpecFiledMetadatas();

        if (CollectionUtils.isEmpty(cargoNameFieldMetadatas) && CollectionUtils.isEmpty(itemSpecFieldMetadatas)) {
            return itemGroups;
        }


        int itemSpecNameLengthLimit = ruleInfo.getSplitRule().getItemSpecNameLength() == null ?
                this.itemSpecNameLengthLimit : (ruleInfo.getSplitRule().getItemSpecNameLength() > this.itemSpecNameLengthLimit ?
                this.itemSpecNameLengthLimit : ruleInfo.getSplitRule().getItemSpecNameLength());


        for (ItemGroup itemGroup : itemGroups) {
            for (BillItem billItem : itemGroup.getBillItems()) {

                String joinName = parseFieldValue(cargoNameFieldMetadatas, billInfo, billItem);
                if (StringUtils.hasText(joinName)) {
                    billItem.setJoinName(joinName);
                }

                String itemSpecName = parseFieldValue(itemSpecFieldMetadatas, billInfo, billItem);
                if (gtLimit(itemSpecName, itemSpecNameLengthLimit)) {
                    itemSpecName = CommonTools.substring(itemSpecName, itemSpecNameLengthLimit, GBK);
                }
                if (StringUtils.hasText(itemSpecName)) {
                    billItem.setItemSpec(itemSpecName);
                }
            }
        }

        return itemGroups;
    }

    private String parseFieldValue(List<RemarkFieldMetadata> cargoNameFieldMetadatas, BillInfo billInfo, BillItem billItem) {
        if (CollectionUtils.isEmpty(cargoNameFieldMetadatas)) {
            return "";
        }
        StringBuilder sb = new StringBuilder();
        for (RemarkFieldMetadata fieldMetadata : cargoNameFieldMetadatas) {
            String fieldName = fieldMetadata.getFieldName();
            Object obj = null;
            if (isBillInfoField(fieldMetadata)) {
                obj = billInfo;
            }
            if (isBillItemField(fieldMetadata)) {
                obj = billItem;
            }
            if (obj == null) {
                throw new SplitBizException("fieldGroupIndex value is illegal!");
            }
            Object fieldValue = null;
            try {
                fieldValue = FiledUtils.getFieldValue(obj, fieldName);
            } catch (ReflectionException ignore) {
                logger.warn(String.format("field = [%s] not found from obj = [%s]!", fieldName, obj.getClass()));
            }
            sb.append(Objects.toString(fieldValue, ""));
        }
        return sb.toString();
    }

    private boolean gtLimit(String str, int lengthLimit) {
        return str.getBytes(GBK).length > lengthLimit;
    }

    private boolean isBillItemField(RemarkFieldMetadata remarkFieldMetadata) {
        return remarkFieldMetadata.getFieldGroupIndex() == FieldOfObj.BILL_ITEM;
    }

    private boolean isBillInfoField(RemarkFieldMetadata remarkFieldMetadata) {
        return remarkFieldMetadata.getFieldGroupIndex() == FieldOfObj.BILL_INFO;
    }

    public void setCargoNameLengthLimit(int cargoNameLengthLimit) {
        this.cargoNameLengthLimit = cargoNameLengthLimit;
    }

    public void setItemSpecNameLengthLimit(int itemSpecNameLengthLimit) {
        this.itemSpecNameLengthLimit = itemSpecNameLengthLimit;
    }
}
