package com.xforceplus.phoenix.split.service.dataflow.impl;

import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.google.common.collect.MultimapBuilder;
import com.xforceplus.phoenix.split.domain.ItemGroup;
import com.xforceplus.phoenix.split.domain.RuleInfo;
import com.xforceplus.phoenix.split.domain.SplitGroupLimit;
import com.xforceplus.phoenix.split.model.BillInfo;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.service.SplitRuleUtil;
import com.xforceplus.phoenix.split.service.dataflow.DataProcessPlugin;
import com.xforceplus.phoenix.split.service.dataflow.MinInvoiceService;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 立邦拆票合并发票实现
 */
@Service
public class MergeBySplitFieldPlugin implements DataProcessPlugin {

    @Override
    public List<ItemGroup> processData(List<ItemGroup> itemGroups, BillInfo billInfo, RuleInfo ruleInfo) {

        if (!ruleInfo.getSplitRule().isMergeBySplitFiled()) {
            return itemGroups;
        }

        List<ItemGroup> result = Lists.newLinkedList();
        Map<String, Multimap<String, ItemGroup>> temp = new HashMap<>();


        for (ItemGroup itemGroup : itemGroups) {
            String parentGroupFlag = itemGroup.getParentGroupFlag();
            String invoiceOfGroupKey = itemGroup.getInvoiceOfGroupKey();

            Multimap<String, ItemGroup> invoices = temp.get(parentGroupFlag);
            if (invoices == null) {
                invoices = MultimapBuilder.linkedHashKeys().linkedListValues().build();
                temp.put(parentGroupFlag, invoices);
            }
            invoices.put(invoiceOfGroupKey, itemGroup);

        }

        for (Map.Entry<String, Multimap<String, ItemGroup>> entry : temp.entrySet()) {
            Multimap<String, ItemGroup> invoices = entry.getValue();
            result.addAll(merge(invoices, ruleInfo));
        }
        return result;
    }

    private List<ItemGroup> merge(Multimap<String, ItemGroup> invoices, RuleInfo ruleInfo) {

        List<ItemGroup> result = Lists.newLinkedList();

        List<ItemGroup> toBeProcessItemGroup = new ArrayList<>();
        for (String key : invoices.keySet()) {
            if (invoices.get(key).size() > 1) {
                result.addAll(invoices.get(key));
            } else {
                toBeProcessItemGroup.addAll(invoices.get(key));
            }
        }

        if (toBeProcessItemGroup.size() > 1) {
            toBeProcessItemGroup = mergerInvoice(toBeProcessItemGroup, ruleInfo);
        }

        result.addAll(toBeProcessItemGroup);
        return result;
    }

    private List<ItemGroup> mergerInvoice(List<ItemGroup> toBeProcessItemGroup, RuleInfo ruleInfo) {
        SplitGroupLimit splitGroupLimit = SplitRuleUtil.createSplitGroupLimit(ruleInfo.getSplitRule(),
                toBeProcessItemGroup.get(0).getBillItems().get(0).getGoodsTaxNo());

        MinInvoiceService minInvoiceService = new MinInvoiceService(splitGroupLimit);
        List<List<BillItem>> invoices = minInvoiceService.minMergeInvoices(toBeProcessItemGroup);

        List<ItemGroup> result = Lists.newLinkedList();

        invoices.forEach(invoice -> {
            ItemGroup itemGroup = new ItemGroup();
            itemGroup.setBillItems(invoice);
            result.add(itemGroup);
        });
        return result;
    }
}
