package com.xforceplus.phoenix.split.service.dataflow.impl;

import com.alibaba.fastjson.JSON;
import com.google.common.base.Stopwatch;
import com.xforceplus.phoenix.split.domain.ItemAmountInfo;
import com.xforceplus.phoenix.split.domain.ItemGroup;
import com.xforceplus.phoenix.split.domain.RuleInfo;
import com.xforceplus.phoenix.split.exception.SplitRuleParamException;
import com.xforceplus.phoenix.split.model.BillInfo;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.model.SplitRule;
import com.xforceplus.phoenix.split.service.SplitBillItemAmountServiceFactory;
import com.xforceplus.phoenix.split.service.dataflow.DataProcessPlugin;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

@Service
public class SplitBillItemPlugin implements DataProcessPlugin {

    private final Logger logger = LoggerFactory.getLogger(SplitBillItemPlugin.class);

    public static final String ORIGIN_INVOICE_CODE = "originInvoiceCode";

    public static final String ORIGIN_INVOICE_NO = "originInvoiceNo";

    public static final String RED_NOTIFICATION_NO = "redNotificationNo";


    @Autowired
    private SplitBillItemAmountServiceFactory splitBillItemAmountServiceFactory;


    @Override
    public List<ItemGroup> processData(List<ItemGroup> itemGroups, BillInfo billInfo, RuleInfo ruleInfo) {
        Stopwatch stopwatch = Stopwatch.createStarted();

        itemGroups.forEach(itemGroup -> {
            List<BillItem> billItemList = new LinkedList<>();
            for (BillItem billItem : itemGroup.getBillItems()) {
                checkPriceAndQuantity(billItem);
                SplitRule rule = ruleInfo.getSplitRule();
                if (split(ruleInfo.getExtRule(), billItem, rule)) {
                    billItemList.addAll(splitBillItemByLimitAmount(billItem, rule));
                } else {
                    billItemList.add(billItem);
                }
            }
            itemGroup.setBillItems(billItemList);
        });
        logger.info("process bill items elapsed time = {} ms", stopwatch.elapsed().toMillis());

        return itemGroups;
    }

    private void checkPriceAndQuantity(BillItem billItem) {
        BigDecimal price = billItem.getUnitPrice();
        BigDecimal quantity = billItem.getQuantity();
        if ((price.compareTo(BigDecimal.ZERO) != 0 && quantity.compareTo(BigDecimal.ZERO) == 0) ||
                (price.compareTo(BigDecimal.ZERO) == 0 && quantity.compareTo(BigDecimal.ZERO) != 0)) {
            throw new SplitRuleParamException(String.format("salesbillItemId = [%s] 单价跟数量必须同时为零或者同时不为零!", billItem.getSalesbillItemId()));
        }

    }


    private boolean split(Map<String, Object> extRule, BillItem billItem, SplitRule rule) {
        if (gtLimitAmount(billItem, rule)) {
            return !billItem.isRedItem() || splitRed(extRule, billItem);
        }
        return false;
    }

    private boolean splitRed(Map<String, Object> extRule, BillItem billItem) {
        return billItem.isRedItem() &&
                StringUtils.isEmpty(extRule.get(ORIGIN_INVOICE_CODE)) &&
                StringUtils.isEmpty(extRule.get(ORIGIN_INVOICE_NO)) &&
                StringUtils.isEmpty(extRule.get(RED_NOTIFICATION_NO));
    }


    private List<BillItem> splitBillItemByLimitAmount(BillItem billItem, SplitRule rule) {
        ItemAmountInfo itemAmountInfo = copyBillItemAmountInfo(billItem, rule);
        List<ItemAmountInfo> itemAmountInfoList = splitBillItemAmountServiceFactory.getByRule(rule)
                .splitAmount(itemAmountInfo, rule);
        logger.debug("split id = {} itemAmount, result = {}", billItem.getSalesbillItemId(), itemAmountInfoList);
        return reCreateBillItem(itemAmountInfoList, billItem);
    }

    private List<BillItem> reCreateBillItem(List<ItemAmountInfo> itemAmountInfoList, BillItem billItem) {
        List<BillItem> result = new ArrayList<>(itemAmountInfoList.size());
        for (ItemAmountInfo itemAmountInfo : itemAmountInfoList) {
            BillItem newBillItem = createNewBillItem(itemAmountInfo, billItem);
            result.add(newBillItem);
        }

        return result;
    }

    private BillItem createNewBillItem(ItemAmountInfo itemAmountInfo, BillItem billItem) {
        BillItem result = JSON.parseObject(JSON.toJSONString(billItem), BillItem.class);

        result.setOriginalAmountWithoutTax(billItem.getAmountWithoutTax());

        result.setAmountWithoutTax(itemAmountInfo.getAmountWithoutTax());
        result.setAmountWithTax(itemAmountInfo.getAmountWithTax());
        result.setTaxAmount(itemAmountInfo.getTaxAmount());

        result.setDeductions(itemAmountInfo.getDeductions());

        result.setDiscountTax(itemAmountInfo.getDiscountTax());
        result.setDiscountWithoutTax(itemAmountInfo.getDiscountWithoutTax());
        result.setDiscountWithTax(itemAmountInfo.getDiscountWithTax());


        result.setUnitPrice(itemAmountInfo.getUnitPrice());
        result.setQuantity(itemAmountInfo.getQuantity());
        result.setSplitItem(true);

        return result;
    }


    private ItemAmountInfo copyBillItemAmountInfo(BillItem billItem, SplitRule rule) {
        ItemAmountInfo itemAmountInfo = JSON.parseObject(JSON.toJSONString(billItem), ItemAmountInfo.class);
        itemAmountInfo.setPriceMethod(rule.getPriceMethod());
        return itemAmountInfo;
    }

    /**
     * 判断是否超限额
     *
     * @param billItem 明细
     * @param rule     规则
     */
    private boolean gtLimitAmount(BillItem billItem, SplitRule rule) {
        boolean limitIsAmountWithTax = rule.isLimitIsAmountWithTax();
        BigDecimal limitAmount = rule.getInvoiceLimit();
        boolean gtLimitAmount;
        if (limitIsAmountWithTax) {
            BigDecimal amountWithTax = billItem.getAmountWithTax();
            BigDecimal discount = billItem.getInnerDiscountWithTax()
                    .add(billItem.getInnerPrepayAmountWithTax())
                    .add(billItem.getOutterDiscountWithTax())
                    .add(billItem.getOutterPrepayAmountWithTax());
            gtLimitAmount = amountWithTax.subtract(discount).compareTo(limitAmount) > 0;
        } else {
            BigDecimal amountWithoutTax = billItem.getAmountWithoutTax();
            BigDecimal discount = billItem.getInnerDiscountWithoutTax()
                    .add(billItem.getInnerPrepayAmountWithoutTax())
                    .add(billItem.getOutterDiscountWithoutTax())
                    .add(billItem.getOutterPrepayAmountWithoutTax());
            gtLimitAmount = amountWithoutTax.subtract(discount).compareTo(limitAmount) > 0;
        }
        return gtLimitAmount;
    }
}
