package com.xforceplus.phoenix.split.service.impl;

import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.xforceplus.phoenix.split.constant.InvoiceType;
import com.xforceplus.phoenix.split.constant.PreInvoiceTemplateVersionStatus;
import com.xforceplus.phoenix.split.exception.SplitBizException;
import com.xforceplus.phoenix.split.model.*;
import com.xforceplus.phoenix.split.service.PreInvoiceGenerateService;
import com.xforceplus.phoenix.split.service.RemarkService;
import com.xforceplus.phoenix.split.util.CommonTools;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static java.math.BigDecimal.ROUND_HALF_UP;


/**
 * 创建预制发票抽象类
 */
@Service
public class PreInvoiceGenerateServiceImpl implements PreInvoiceGenerateService {

    private static final Logger logger = LoggerFactory.getLogger(PreInvoiceGenerateServiceImpl.class);


    @Autowired
    protected RemarkService remarkService;

    @Value("${cargoNameLengthLimit:100}")
    private int cargoNameLengthLimit;

    private static final Charset GBK = Charset.forName("GBK");

    /**
     * 生成预制发票
     *
     * @param billInfo 单据信息
     * @param rule     规则
     * @return 预制发票list
     */
    @Override
    public List<SplitPreInvoiceInfo> createPreInvoice(BillInfo billInfo, SplitRule rule) {
        List<BillItem> billItemList = billInfo.getBillItems();
        if (CollectionUtils.isEmpty(billItemList)) {
            return Lists.newArrayList();
        }

        if ("1".equals(rule.getSaleListOption()) || "2".equals(rule.getSaleListOption())) {
            //单据已按成品油分组，所以判断任意一条明细即可判断本组是否为成品油
            if (billInfo.getInvoiceType().equals(InvoiceType.NORMAL_ROLL.value())) {
                throw new SplitBizException("卷票不支持销货清单");
            }
        }
        //生成预制发票
        SplitPreInvoiceInfo splitPreInvoiceInfo = generatePreInvoice(billInfo, rule);
        //追加备注
        splitPreInvoiceInfo.getPreInvoiceMain().setRemark(toLegalRemark(billInfo, rule));
        return Lists.newArrayList(splitPreInvoiceInfo);
    }

    boolean isRedInvoice(PreInvoiceMain preInvoiceMain) {
        return StringUtils.isNotBlank(preInvoiceMain.getOriginInvoiceNo())
                || StringUtils.isNotBlank(preInvoiceMain.getOriginInvoiceCode())
                || StringUtils.isNotBlank(preInvoiceMain.getRedNotificationNo());
    }

    /**
     * 生成预制发票 mei
     *
     * @param billInfo 单据信
     * @param rule     拆票规则
     * @return 预制发票列表
     */
    private SplitPreInvoiceInfo generatePreInvoice(BillInfo billInfo, SplitRule rule) {
        BillItem billItem0 = billInfo.getBillItems().get(0);
        //是否为红字业务单
        boolean isRedItem = billInfo.getBillItems().get(0).isRedItem();
        //零税率发票专改普
        if (rule.getZeroTaxOption().equals(ZeroTaxOption.PROCESS)) {
            if (billItem0.getTaxRate().equals(BigDecimal.ZERO)) {
                billInfo.setInvoiceType(InvoiceType.NORMAL.value());
            }
        }
        //特殊发票标记
        String specialInvoiceFlag = "0";
        if (HBBM_SET.contains(billItem0.getGoodsTaxNo())) {
            specialInvoiceFlag = "2";
        }

        SplitPreInvoiceInfo splitPreInvoiceInfo = new SplitPreInvoiceInfo();
        splitPreInvoiceInfo.setRuleId(String.valueOf(rule.getRuleId()));

        splitPreInvoiceInfo.setPreInvoiceMain(createPreInvoiceMain(billInfo, rule, specialInvoiceFlag));
        splitPreInvoiceInfo.setPreInvoiceItems(createPreInvoiceItems(billInfo.getBillItems(), rule, isRedItem));
        splitPreInvoiceInfo.getPreInvoiceMain().setSaleListFileFlag(Byte.valueOf(getPrintSalesListFlag(splitPreInvoiceInfo, rule) ? "1" : "0"));

        if (splitPreInvoiceInfo.getPreInvoiceMain().getSaleListFileFlag().equals(Byte.valueOf("1"))) {
            if (splitPreInvoiceInfo.getPreInvoiceItems().size() > rule.getSalesListMaxRow()) {
                throw new SplitBizException("明细行数超过销货清单行数！");
            }
        }
        splitPreInvoiceInfo.getPreInvoiceMain().setListGoodsName(Byte.valueOf("1").equals(splitPreInvoiceInfo.getPreInvoiceMain().getSaleListFileFlag()) ? LIST_GOODS_NAME : "");
        //设置主信息税率
        List<String> taxRateList = splitPreInvoiceInfo.getPreInvoiceItems().stream()
                .map(PreInvoiceItem::getTaxRate).distinct().map(BigDecimal::toPlainString).collect(Collectors.toList());
        splitPreInvoiceInfo.getPreInvoiceMain().setTaxRate(Joiner.on(",").skipNulls().join(taxRateList));
        //设置主信息含税金额
        BigDecimal amountWithTax = splitPreInvoiceInfo.getPreInvoiceItems().stream()
                .map(PreInvoiceItem::getAmountWithTax).reduce(BigDecimal.ZERO, BigDecimal::add);
        BigDecimal discountWithTax = splitPreInvoiceInfo.getPreInvoiceItems().stream()
                .map(PreInvoiceItem::getDiscountWithTax).reduce(BigDecimal.ZERO, BigDecimal::add);
        splitPreInvoiceInfo.getPreInvoiceMain().setAmountWithTax(amountWithTax.subtract(discountWithTax).setScale(2, ROUND_HALF_UP));


        //设置主信息不含税金额
        BigDecimal amountWithoutTax = splitPreInvoiceInfo.getPreInvoiceItems().stream()
                .map(PreInvoiceItem::getAmountWithoutTax).reduce(BigDecimal.ZERO, BigDecimal::add);
        BigDecimal discountWithoutTax = splitPreInvoiceInfo.getPreInvoiceItems().stream()
                .map(PreInvoiceItem::getDiscountWithoutTax).reduce(BigDecimal.ZERO, BigDecimal::add);
        splitPreInvoiceInfo.getPreInvoiceMain().setAmountWithoutTax(amountWithoutTax.subtract(discountWithoutTax).setScale(2, ROUND_HALF_UP));
        //设置主信息税额
        BigDecimal taxAmount = splitPreInvoiceInfo.getPreInvoiceItems().stream()
                .map(PreInvoiceItem::getTaxAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
        BigDecimal discountTax = splitPreInvoiceInfo.getPreInvoiceItems().stream()
                .map(PreInvoiceItem::getDiscountTax).reduce(BigDecimal.ZERO, BigDecimal::add);
        splitPreInvoiceInfo.getPreInvoiceMain().setTaxAmount(taxAmount.subtract(discountTax).setScale(2, ROUND_HALF_UP));

//            splitPreInvoiceInfo.setPreInvoiceExt(createPreInvoiceExt());
        return splitPreInvoiceInfo;
    }

    /**
     * 是否打印销货清单
     *
     * @param splitPreInvoiceInfo
     * @param rule
     * @return
     */


    private boolean getPrintSalesListFlag(SplitPreInvoiceInfo splitPreInvoiceInfo, SplitRule rule) {
        int itemSize = 0;
        for (PreInvoiceItem preInvoiceItem : splitPreInvoiceInfo.getPreInvoiceItems()) {
            if (preInvoiceItem.getDiscountWithoutTax() != null && preInvoiceItem.getDiscountWithoutTax().compareTo(BigDecimal.ZERO) > 0) {
                itemSize = itemSize + 2;
            } else {
                itemSize++;
            }
        }


        //成品油不支持销货清单
        if ("2".equals(splitPreInvoiceInfo.getPreInvoiceMain().getSpecialInvoiceFlag())) {
            return false;
        }
        //红冲发票不拆票  如果行数超过发票限制 强制打印销货清单
        if (isRedInvoice(splitPreInvoiceInfo.getPreInvoiceMain())) {
            if (itemSize > rule.getInvoiceItemMaxRow()) {
                return true;
            }
        }

        if ("2".equals(rule.getSaleListOption())) {
            return true;
        } else if ("1".equals(rule.getSaleListOption())) {
            if (rule.getInvoiceItemMaxRow() != null) {
                if (itemSize > rule.getInvoiceItemMaxRow()) {
                    return true;
                }
            } else {
                return true;
            }
        }
        return false;
    }

    /**
     * 生成预制发票扩展信息
     *
     * @return 预制发票扩展信息
     */
    private PreInvoiceExt createPreInvoiceExt() {
        return null;
    }


    /**
     * 生成预制发票主信息
     *
     * @param billInfo           单据信息
     * @param rule               拆票规则
     * @param specialInvoiceFlag 特殊发票标记
     * @return 预制发票主信息
     */
    private PreInvoiceMain createPreInvoiceMain(BillInfo billInfo, SplitRule rule, String specialInvoiceFlag) {
        PreInvoiceMain preInvoiceMain = new PreInvoiceMain();
        BeanUtils.copyProperties(billInfo, preInvoiceMain);
        preInvoiceMain.setRuleId(rule.getRuleId());
        preInvoiceMain.setDisplayPriceQuality(Byte.valueOf(rule.getUnitPriceAmountOps()));
        preInvoiceMain.setSpecialInvoiceFlag(specialInvoiceFlag);
        preInvoiceMain.setTemplateVersion(null != rule.getInvoiceItemMaxRow() && rule.getInvoiceItemMaxRow() > 5
                ? PreInvoiceTemplateVersionStatus.EIGHT.getValue()
                : PreInvoiceTemplateVersionStatus.FIVE.getValue());
        return preInvoiceMain;
    }

    public static void negateBigDecimalField(Object object) {
        try {
            for (Field field : object.getClass().getDeclaredFields()) {
                if (field.getType() == BigDecimal.class) {
                    field.setAccessible(true);
                    BigDecimal bigDecimal = (BigDecimal) field.get(object);
                    if (Objects.nonNull(bigDecimal)) {
                        field.set(object, bigDecimal.negate());
                    }
                }
            }
        } catch (IllegalAccessException e) {
            logger.error("negateBigDecimalField error:{}", e);
            throw new RuntimeException(e);
        }
    }

    /**
     * 生成预制发票明细
     *
     * @param itemList  单据明细列表
     * @param rule      拆票规则
     * @param isRedItem
     * @return 预制发票明细列表
     */
    private List<PreInvoiceItem> createPreInvoiceItems(List<BillItem> itemList, SplitRule rule, boolean isRedItem) {
        List<PreInvoiceItem> preInvoiceItems = new ArrayList<>(itemList.size());
        int cargoNameLengthLimit = rule.getCargoNameLength() == null ?
                this.cargoNameLengthLimit : (rule.getCargoNameLength() > this.cargoNameLengthLimit ?
                this.cargoNameLengthLimit : rule.getCargoNameLength());

        for (BillItem billItem : itemList) {

            PreInvoiceItem preInvoiceItem = new PreInvoiceItem();
            preInvoiceItems.add(preInvoiceItem);
            BeanUtils.copyProperties(billItem, preInvoiceItem);

            billItem.setAmountWithoutTax(billItem.getAmountWithoutTax().setScale(2, ROUND_HALF_UP));
            billItem.setAmountWithTax(billItem.getAmountWithTax().setScale(2, ROUND_HALF_UP));
            billItem.setTaxAmount(billItem.getAmountWithTax().setScale(2, ROUND_HALF_UP));
            billItem.setDiscountWithoutTax(billItem.getDiscountWithoutTax().setScale(2, ROUND_HALF_UP));
            billItem.setDiscountWithTax(billItem.getDiscountWithTax().setScale(2, ROUND_HALF_UP));
            billItem.setDiscountTax(billItem.getDiscountTax().setScale(2, ROUND_HALF_UP));

            //红字业务单，金额设置为负数
            if (isRedItem) {
                negateBigDecimalField(preInvoiceItem);
            }
            preInvoiceItem.setUnitPrice(preInvoiceItem.getUnitPrice().abs());
            preInvoiceItem.setTaxRate(preInvoiceItem.getTaxRate().abs());

            preInvoiceItem.setCargoCode(billItem.getItemCode());
            if (StringUtils.isNotBlank(billItem.getItemShortName())) {
                String cargoName = TAX_CODE_SHORT_NAME_JOINER_PREFIX + billItem.getItemShortName()
                        + TAX_CODE_SHORT_NAME_JOINER_SUFFIX;

                if (StringUtils.isNotBlank(billItem.getJoinName())) {
                    cargoName = cargoName + billItem.getJoinName();
                } else {
                    cargoName = cargoName + billItem.getItemName();
                }
                preInvoiceItem.setCargoName(cargoName);
            } else {
                preInvoiceItem.setCargoName(billItem.getItemName());
                if (StringUtils.isNotBlank(billItem.getJoinName())) {
                    preInvoiceItem.setCargoName(billItem.getJoinName());
                }
            }
            if (cargoNameGtLimit(preInvoiceItem.getCargoName(), cargoNameLengthLimit)) {
                preInvoiceItem.setCargoName(CommonTools.substring(preInvoiceItem.getCargoName(), cargoNameLengthLimit, GBK));
            }
            preInvoiceItem.setPrintContentFlag(BigDecimal.ZERO.compareTo(billItem.getQuantity()) == 0 && BigDecimal.ZERO.compareTo(billItem.getUnitPrice()) == 0
                    ? "1" : StringUtils.isNotEmpty(rule.getUnitPriceAmountOps()) ? rule.getUnitPriceAmountOps() : "0");
            preInvoiceItem.setTaxItem("");
            preInvoiceItem.setDeduction(billItem.getDeductions());
            preInvoiceItem.setGoodsTaxNo(StringUtils.isEmpty(billItem.getGoodsTaxNo()) ? "" : billItem.getGoodsTaxNo());
            preInvoiceItem.setTaxPre(StringUtils.isEmpty(billItem.getTaxPre()) ? "0" : billItem.getTaxPre());
            preInvoiceItem.setZeroTax(StringUtils.isEmpty(billItem.getZeroTax()) ? "" : billItem.getZeroTax());

            preInvoiceItem.setPriceMethod(String.valueOf(rule.getPriceMethod().value()));

            if (HBBM_SET.contains(preInvoiceItem.getGoodsTaxNo())) {
                Objects.requireNonNull(preInvoiceItem.getQuantity(), "成品油发票数量，单位为必填");
                Objects.requireNonNull(preInvoiceItem.getQuantityUnit(), "成品油发票数量，单位为必填");
                if (preInvoiceItem.getQuantity().compareTo(BigDecimal.ZERO) == 0) {
                    throw new SplitBizException("成品油发票数量，单位为必填");
                }
            }
            // 计算单价
//            BigDecimal tempUnitPrice = preInvoiceItem.getAmountWithoutTax().divide(billItem.getQuantity(), 15,
//                    BigDecimal.ROUND_HALF_UP);
//            preInvoiceItem.setUnitPrice(tempUnitPrice);
        }
        return preInvoiceItems;
    }

    private boolean cargoNameGtLimit(String cargoName, int cargoNameLengthLimit) {
        return cargoName.getBytes(GBK).length > cargoNameLengthLimit;
    }


    /**
     * 追加备注
     *
     * @param billInfo 单据信息
     * @param rule     拆票规则
     */
    private String toLegalRemark(BillInfo billInfo, SplitRule rule) {
        String remark = remarkService.splice(billInfo, rule);
        Integer remarkSizeLimit = rule.getCustomRemarkSize();
        if (Objects.isNull(remarkSizeLimit)) {
            if (InvoiceType.ELECTRONIC.value().equals(billInfo.getInvoiceType())) {
                remarkSizeLimit = ELECTRONIC_SIZE;
            } else {
                remarkSizeLimit = DEFAULT_SIZE;
            }
        }
        return RemarkService.toLegalRemark(remark, remarkSizeLimit);
    }


}
