package com.xforceplus.phoenix.split.service.impl;

import com.alibaba.fastjson.JSON;
import com.xforceplus.phoenix.split.domain.ItemAmountInfo;
import com.xforceplus.phoenix.split.model.PriceMethod;
import com.xforceplus.phoenix.split.model.SplitRule;
import com.xforceplus.phoenix.split.service.SplitBillItemAmountService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.LinkedList;
import java.util.List;

import static java.math.BigDecimal.ROUND_DOWN;
import static java.math.BigDecimal.ROUND_HALF_UP;
import static java.math.RoundingMode.HALF_UP;

/**
 * 拆分超限额单据明细
 */
@Service
public class DefaultSplitBillItemAmountServiceImpl implements SplitBillItemAmountService {

    private Logger logger = LoggerFactory.getLogger(DefaultSplitBillItemAmountServiceImpl.class);

    public static final BigDecimal MIN_QUANTITY = BigDecimal.ONE.movePointLeft(6);

    private static final BigDecimal ERROR_AMOUNT_LIMIT = BigDecimal.ONE.movePointLeft(2);

    @Override
    public List<ItemAmountInfo> splitAmount(ItemAmountInfo itemAmountInfo, SplitRule rule) {
        ItemAmountInfo splitAmountInfo = JSON.parseObject(JSON.toJSONString(itemAmountInfo), ItemAmountInfo.class);

        List<ItemAmountInfo> result = new LinkedList<>();
        calculateAmountByPriceMethod(splitAmountInfo);
        boolean hasQuantity = itemAmountInfo.getQuantity().compareTo(BigDecimal.ZERO) != 0;
        mergeDiscountAmount(splitAmountInfo);
        BigDecimal limitAmount = rule.getInvoiceLimit();
        ItemAmountInfo splitItemAmountInfo = null;
        while (gtLimitAmount(splitAmountInfo, limitAmount)) {
            if (splitItemAmountInfo == null) {
                splitItemAmountInfo = splitFirstItemAmountInfo(rule, splitAmountInfo, limitAmount);
            } else {
                splitItemAmountInfo = copyFirstItemAmount(limitAmount, rule, splitAmountInfo, splitItemAmountInfo);
            }
            deductSplitItemAmount(rule, splitAmountInfo, splitItemAmountInfo);
            result.add(splitItemAmountInfo);
        }
        processLastItemAmountInfo(splitAmountInfo, hasQuantity, rule);
        result.add(splitAmountInfo);
        logger.debug("before process error amount result = {}", result);
        processErrorAmount(result);
        logger.debug("after process error amount result = {}", result);
        return result;
    }

    /**
     * 根据价格方式计算金额
     *
     * @param splitAmountInfo
     */
    protected void calculateAmountByPriceMethod(ItemAmountInfo splitAmountInfo) {
        calculateWithoutTaxAmount(splitAmountInfo);
    }

    /**
     * 处理金额误差
     *
     * @param itemAmountInfoList
     */
    protected void processErrorAmount(List<ItemAmountInfo> itemAmountInfoList) {
        ItemAmountInfo lastItemAmountInfo = itemAmountInfoList.get(itemAmountInfoList.size() - 1);

        BigDecimal taxAmountError = lastItemAmountInfo.getAmountWithoutTax().subtract(lastItemAmountInfo.getDeductions())
                .multiply(lastItemAmountInfo.getTaxRate()).setScale(2, HALF_UP)
                .subtract(lastItemAmountInfo.getTaxAmount()).abs();

        BigDecimal discountTaxAmount = BigDecimal.ZERO;
        ItemAmountInfo lastDiscountItemAmountInfo = null;

        if (lastItemAmountInfo.getOutterDiscountWithoutTax().compareTo(BigDecimal.ZERO) > 0 ||
                lastItemAmountInfo.getOutterPrepayAmountWithoutTax().compareTo(BigDecimal.ZERO) > 0) {

            for (int i = itemAmountInfoList.size() - 1; i >= 0; i--) {
                ItemAmountInfo itemAmountInfo = itemAmountInfoList.get(i);
                if (itemAmountInfo.getDiscountWithoutTax().compareTo(BigDecimal.ZERO) > 0) {
                    lastDiscountItemAmountInfo = itemAmountInfo;
                    break;
                }
            }
            if (lastDiscountItemAmountInfo == null) {
                throw new RuntimeException("未找到最后一笔折扣金额明细!");
            }
            discountTaxAmount = lastDiscountItemAmountInfo.getDiscountWithoutTax()
                    .multiply(lastDiscountItemAmountInfo.getTaxRate()).setScale(2, HALF_UP)
                    .subtract(lastDiscountItemAmountInfo.getDiscountTax()).abs();
        }


        logger.info("taxAmountError = {}, discountTaxAmount = {}", taxAmountError, discountTaxAmount);
        int index = 0;
        while (hasErrorAmount(taxAmountError, discountTaxAmount)) {
            ItemAmountInfo itemAmountInfo = itemAmountInfoList.get(index++);
            taxAmountError = processTaxAmountError(taxAmountError, itemAmountInfo, lastItemAmountInfo);
            discountTaxAmount = processDiscountTaxAmountError(discountTaxAmount, itemAmountInfo, lastDiscountItemAmountInfo);
        }
    }

    /**
     * 处理折扣金额误差
     *
     * @param discountTaxAmount
     * @param itemAmountInfo
     * @param lastItemAmountInfo
     * @return
     */
    private BigDecimal processDiscountTaxAmountError(BigDecimal discountTaxAmount,
                                                     ItemAmountInfo itemAmountInfo,
                                                     ItemAmountInfo lastItemAmountInfo) {
        if (discountTaxAmount.compareTo(ERROR_AMOUNT_LIMIT) > 0) {
            discountTaxAmount = discountTaxAmount.subtract(ERROR_AMOUNT_LIMIT);

            itemAmountInfo.setDiscountTax(itemAmountInfo.getDiscountTax().add(ERROR_AMOUNT_LIMIT));
            itemAmountInfo.setDiscountWithTax(itemAmountInfo.getDiscountWithTax().add(ERROR_AMOUNT_LIMIT));

            lastItemAmountInfo.setDiscountTax(lastItemAmountInfo.getDiscountTax().subtract(ERROR_AMOUNT_LIMIT));
            lastItemAmountInfo.setDiscountWithTax(lastItemAmountInfo.getDiscountWithTax().subtract(ERROR_AMOUNT_LIMIT));
        }
        return discountTaxAmount;
    }

    /**
     * 处理税额误差
     *
     * @param taxAmountError
     * @param itemAmountInfo
     * @param lastItemAmountInfo
     * @return
     */
    private BigDecimal processTaxAmountError(BigDecimal taxAmountError,
                                             ItemAmountInfo itemAmountInfo,
                                             ItemAmountInfo lastItemAmountInfo) {
        if (taxAmountError.compareTo(ERROR_AMOUNT_LIMIT) > 0) {
            taxAmountError = taxAmountError.subtract(ERROR_AMOUNT_LIMIT);

            itemAmountInfo.setTaxAmount(itemAmountInfo.getTaxAmount().add(ERROR_AMOUNT_LIMIT));
            itemAmountInfo.setAmountWithTax(itemAmountInfo.getAmountWithTax().add(ERROR_AMOUNT_LIMIT));

            lastItemAmountInfo.setTaxAmount(lastItemAmountInfo.getTaxAmount().subtract(ERROR_AMOUNT_LIMIT));
            lastItemAmountInfo.setAmountWithTax(lastItemAmountInfo.getAmountWithTax().subtract(ERROR_AMOUNT_LIMIT));
        }
        return taxAmountError;

    }

    /**
     * 是否有金额误差
     *
     * @param taxAmountError
     * @param discountTaxAmount
     * @return
     */
    private boolean hasErrorAmount(BigDecimal taxAmountError, BigDecimal discountTaxAmount) {

        return taxAmountError.compareTo(ERROR_AMOUNT_LIMIT) > 0 ||
                discountTaxAmount.compareTo(ERROR_AMOUNT_LIMIT) > 0;
    }

    /**
     * 处理最好一条
     *  @param itemAmountInfo
     * @param hasQuantity
     * @param rule
     */
    protected void processLastItemAmountInfo(ItemAmountInfo itemAmountInfo, boolean hasQuantity, SplitRule rule) {
        if (hasQuantity) {
            BigDecimal quantity = itemAmountInfo.getQuantity();
            if (quantity.compareTo(BigDecimal.ZERO) == 0) {
                quantity = MIN_QUANTITY;
            }
            itemAmountInfo.setQuantity(quantity);
            itemAmountInfo.setUnitPrice(itemAmountInfo.getAmountWithoutTax()
                    .divide(quantity, rule.getUnitPriceScale(), ROUND_HALF_UP));
        }
    }

    /**
     * 扣减拆分金额
     *
     * @param rule
     * @param itemAmountInfo
     * @param splitItemAmountInfo
     */
    private void deductSplitItemAmount(SplitRule rule, ItemAmountInfo itemAmountInfo, ItemAmountInfo splitItemAmountInfo) {
        itemAmountInfo.deductNewItemAmount(splitItemAmountInfo);
        if (isUnitPriceAndQuantityInteger(rule.getAmountSplitRule()) ||
                isUnitPrice(rule.getAmountSplitRule())) {
            itemAmountInfo.setQuantity(itemAmountInfo.getQuantity().subtract(splitItemAmountInfo.getQuantity()));
        }
    }

    private ItemAmountInfo copyFirstItemAmount(BigDecimal limitAmount,
                                               SplitRule rule, ItemAmountInfo itemAmountInfo,
                                               ItemAmountInfo splitItemAmountInfo) {
        ItemAmountInfo result = JSON.parseObject(JSON.toJSONString(splitItemAmountInfo), ItemAmountInfo.class);

        if (leftDiscountAmountLtSplitDiscountAmount(itemAmountInfo, result)) {
            reCalculateAmount(limitAmount, rule, itemAmountInfo, result);
        }
        return result;
    }

    protected void reCalculateAmount(BigDecimal limitAmount, SplitRule rule, ItemAmountInfo itemAmountInfo, ItemAmountInfo result) {
        result.setDiscountWithoutTax(itemAmountInfo.getDiscountWithoutTax());
        result.setDiscountTax(itemAmountInfo.getDiscountTax());
        result.setDiscountWithTax(itemAmountInfo.getDiscountWithTax());

        if (result.getAmountWithoutTax().subtract(result.getDiscountWithoutTax()).compareTo(limitAmount) > 0) {
            BigDecimal splitAmountWithoutTax = limitAmount.add(result.getDiscountWithoutTax()).setScale(2, ROUND_DOWN);
            BigDecimal splitTaxAmount = splitAmountWithoutTax.subtract(result.getDeductions())
                    .multiply(itemAmountInfo.getTaxRate())
                    .setScale(2, ROUND_DOWN);
            BigDecimal splitAmountWithTax = splitAmountWithoutTax.add(splitTaxAmount).setScale(2, ROUND_HALF_UP);

            result.setAmountWithTax(splitAmountWithTax);
            result.setAmountWithoutTax(splitAmountWithoutTax);
            result.setTaxAmount(splitTaxAmount);

            processUnitPriceOrQuantity(rule, result);
        }
    }

    /**
     * 判断剩余折扣是否小于拆分折扣
     */
    protected boolean leftDiscountAmountLtSplitDiscountAmount(ItemAmountInfo itemAmountInfo, ItemAmountInfo result) {
        return itemAmountInfo.getDiscountWithoutTax().compareTo(result.getDiscountWithoutTax()) <= 0;
    }

    /**
     * 按限额拆分金额
     *
     * @param rule
     * @param itemAmountInfo
     * @param limitAmount
     * @return
     */
    protected ItemAmountInfo splitFirstItemAmountInfo(SplitRule rule, ItemAmountInfo itemAmountInfo, BigDecimal limitAmount) {
        return itemAmountInfo.createItemAmountByAmountWithoutTax(rule, limitAmount);
    }

    /**
     * 处理单价数量
     *
     * @param rule
     * @param newItemAmountInfo
     */
    protected void processUnitPriceOrQuantity(SplitRule rule, ItemAmountInfo newItemAmountInfo) {
        if (isUnitPrice(rule.getAmountSplitRule())) {
            BigDecimal unitPrice = newItemAmountInfo.getUnitPrice();
            if (unitPrice != null && unitPrice.compareTo(BigDecimal.ZERO) > 0) {
                BigDecimal quantity = newItemAmountInfo.getAmountWithoutTax().divide(unitPrice, 6,
                        ROUND_HALF_UP);
                if (quantity.compareTo(BigDecimal.ZERO) == 0) {
                    quantity = MIN_QUANTITY;
                    newItemAmountInfo.setUnitPrice(newItemAmountInfo.getAmountWithoutTax().divide(quantity, rule.getUnitPriceScale(), ROUND_HALF_UP));
                }
                newItemAmountInfo.setQuantity(quantity);
            }
        } else if (isUnitPriceAndQuantityInteger(rule.getAmountSplitRule())) {
            BigDecimal quantity = newItemAmountInfo.getAmountWithoutTax().divide(newItemAmountInfo.getUnitPrice(), 0, ROUND_HALF_UP);

            newItemAmountInfo.setQuantity(quantity);

        } else if (isQuantity(rule.getAmountSplitRule())) {
            if (newItemAmountInfo.getQuantity() != null && newItemAmountInfo.getQuantity().compareTo(BigDecimal.ZERO) > 0) {
                newItemAmountInfo.setUnitPrice(newItemAmountInfo.getAmountWithoutTax().divide(newItemAmountInfo.getQuantity(), rule.getUnitPriceScale(), ROUND_HALF_UP));
            }
        } else {
            throw new IllegalArgumentException();
        }
    }

    /**
     * 是否大于限额
     *
     * @param itemAmountInfo
     * @param limitAmount
     * @return
     */
    protected boolean gtLimitAmount(ItemAmountInfo itemAmountInfo, BigDecimal limitAmount) {
        return itemAmountInfo.getAmountWithoutTax()
                .subtract(itemAmountInfo.getDiscountWithoutTax())
                .compareTo(limitAmount) > 0;
    }

    /**
     * 合并折扣金额
     *
     * @param itemAmountInfo
     */
    private void mergeDiscountAmount(ItemAmountInfo itemAmountInfo) {
        itemAmountInfo.setAmountWithoutTax(itemAmountInfo.getAmountWithoutTax()
                .subtract(itemAmountInfo.getInnerDiscountWithoutTax())
                .subtract(itemAmountInfo.getInnerPrepayAmountWithoutTax())
        );

        itemAmountInfo.setAmountWithTax(itemAmountInfo.getAmountWithTax()
                .subtract(itemAmountInfo.getInnerDiscountWithTax())
                .subtract(itemAmountInfo.getInnerPrepayAmountWithTax())
        );

        itemAmountInfo.setTaxAmount(itemAmountInfo.getTaxAmount()
                .subtract(itemAmountInfo.getInnerDiscountTax())
                .subtract(itemAmountInfo.getInnerPrepayAmountTax())
        );

        itemAmountInfo.setDiscountWithoutTax(itemAmountInfo.getOutterDiscountWithoutTax()
                .add(itemAmountInfo.getOutterPrepayAmountWithoutTax())
        );

        itemAmountInfo.setDiscountWithTax(itemAmountInfo.getOutterDiscountWithTax()
                .add(itemAmountInfo.getOutterPrepayAmountWithTax())
        );

        itemAmountInfo.setDiscountTax(itemAmountInfo.getOutterDiscountTax()
                .add(itemAmountInfo.getOutterPrepayAmountTax())
        );
    }

    /**
     * 计算 不含税金额
     *
     * @param itemAmountInfo
     */
    private void calculateWithoutTaxAmount(ItemAmountInfo itemAmountInfo) {
        if (itemAmountInfo.getPriceMethod() == PriceMethod.WITH_TAX) {
            BigDecimal amountWithTax = itemAmountInfo.getAmountWithTax();
            BigDecimal taxRate = itemAmountInfo.getTaxRate();
            itemAmountInfo.setAmountWithoutTax((amountWithTax.add(itemAmountInfo.getDeductions().multiply(taxRate))).divide(BigDecimal.ONE.add(taxRate), 2, HALF_UP));
            itemAmountInfo.setTaxAmount(amountWithTax.subtract(itemAmountInfo.getAmountWithoutTax()));

            BigDecimal outterDiscountWithTax = itemAmountInfo.getOutterDiscountWithTax();
            if (outterDiscountWithTax.compareTo(BigDecimal.ZERO) > 0) {
                itemAmountInfo.setOutterDiscountWithoutTax(outterDiscountWithTax.divide(BigDecimal.ONE.add(taxRate), 2, HALF_UP));
                itemAmountInfo.setOutterDiscountTax(itemAmountInfo.getOutterDiscountWithTax().subtract(itemAmountInfo.getOutterDiscountWithoutTax()));
            }

            BigDecimal outterPrepayAmountWithTax = itemAmountInfo.getOutterPrepayAmountWithTax();
            if (outterPrepayAmountWithTax.compareTo(BigDecimal.ZERO) > 0) {
                itemAmountInfo.setOutterPrepayAmountWithoutTax(outterPrepayAmountWithTax.divide(BigDecimal.ONE.add(taxRate), 2, HALF_UP));
                itemAmountInfo.setOutterPrepayAmountTax(itemAmountInfo.getOutterPrepayAmountWithTax().subtract(itemAmountInfo.getOutterPrepayAmountWithoutTax()));
            }

            BigDecimal innerDiscountWithTax = itemAmountInfo.getInnerDiscountWithTax();
            if (innerDiscountWithTax.compareTo(BigDecimal.ZERO) > 0) {
                itemAmountInfo.setInnerDiscountWithoutTax(innerDiscountWithTax.divide(BigDecimal.ONE.add(taxRate), 2, HALF_UP));
                itemAmountInfo.setInnerDiscountTax(itemAmountInfo.getInnerDiscountWithTax().subtract(itemAmountInfo.getInnerDiscountWithoutTax()));
            }

            BigDecimal innerPrepayAmountWithTax = itemAmountInfo.getInnerPrepayAmountWithTax();
            if (innerPrepayAmountWithTax.compareTo(BigDecimal.ZERO) > 0) {
                itemAmountInfo.setInnerPrepayAmountWithoutTax(innerPrepayAmountWithTax.divide(BigDecimal.ONE.add(taxRate), 2, HALF_UP));
                itemAmountInfo.setInnerPrepayAmountTax(itemAmountInfo.getInnerPrepayAmountWithTax().subtract(itemAmountInfo.getInnerPrepayAmountWithoutTax()));
            }

        }
    }

    /**
     * 按数量拆单价
     */
    public static boolean isQuantity(String amountSplitRule) {
        return "1".equals(amountSplitRule);
    }

    /**
     * 按单价拆数量
     */
    private boolean isUnitPrice(String amountSplitRule) {
        return "3".equals(amountSplitRule);
    }

    /**
     * 按单价拆数量，并保证数量为整数， 最后一笔不保证
     */
    public static boolean isUnitPriceAndQuantityInteger(String amountSplitRule) {
        return "2".equals(amountSplitRule);
    }
}
