package com.xforceplus.phoenix.split.service;


import com.google.common.collect.ImmutableSet;
import com.xforceplus.phoenix.split.constant.InvoiceType;
import com.xforceplus.phoenix.split.constant.PreInvoiceTemplateVersionStatus;
import com.xforceplus.phoenix.split.exception.SplitRuleParamException;
import com.xforceplus.phoenix.split.model.SplitRule;

import java.util.Objects;

public class SplitRuleUtil {
    /**
     * 电票-销货清单最大行
     */
    private static final int ELECTRONIC_SALES_LIST_MAX_ROW = 99;
    /**
     * 销货清单最大行
     */
    private static final int SALES_LIST_MAX_ROW = 2000;

    public static void validateSaleListOption(SplitRule rule) {
        if (!ImmutableSet.of("0", "1", "2").contains(rule.getSaleListOption())) {
            throw new SplitRuleParamException("销货清单参数只能为：0-不启动销货清单,1-启动销货清单,2-强制启用销货清单");
        }
    }

    public static void validateInvoiceItemMaxRow(SplitRule rule) {
        Integer invoiceItemMaxRow = rule.getInvoiceItemMaxRow();
        if (invoiceItemMaxRow == null) {
            throw new SplitRuleParamException("发票明细最大行数不能为空");
        }
        if (invoiceItemMaxRow < 1) {
            throw new SplitRuleParamException("发票明细最大行数必须大于0");
        }
        if (InvoiceType.ELECTRONIC.value().equals(rule.getInvoiceType())) {
            if (invoiceItemMaxRow > PreInvoiceTemplateVersionStatus.FIVE.getValue()) {
                throw new SplitRuleParamException("电票明细最大行数不能超过5");
            }
        } else {
            if (invoiceItemMaxRow > PreInvoiceTemplateVersionStatus.EIGHT.getValue()) {
                throw new SplitRuleParamException("发票明细最大行数不能超过8");
            }
        }
    }

    /**
     * @param rule
     * @return
     */
    public static void validateSalesListMaxRow(SplitRule rule) {
        if ("0".equals(rule.getSaleListOption())) {
            return;
        }
        Integer salesListMaxRow = rule.getSalesListMaxRow();
        if (salesListMaxRow == null) {
            throw new SplitRuleParamException("启用销货清时,单销货清单最大行数不能为空");
        }

        if (salesListMaxRow < 1) {
            throw new SplitRuleParamException("启用销货清时,，销货清单最大行数必须大于0");
        }
    }

    /**
     * @param rule
     * @return
     */
    public static int getSalesListMaxRow(SplitRule rule) {
        if ("0".equals(rule.getSaleListOption())) {
            return 0;
        }
        Integer salesListMaxRow = rule.getSalesListMaxRow();
        if (InvoiceType.ELECTRONIC.value().equals(rule.getInvoiceType()) && salesListMaxRow > ELECTRONIC_SALES_LIST_MAX_ROW) {
            salesListMaxRow = ELECTRONIC_SALES_LIST_MAX_ROW;
        } else if (salesListMaxRow > SALES_LIST_MAX_ROW) {
            salesListMaxRow = SALES_LIST_MAX_ROW;
        }
        return salesListMaxRow;
    }

    public static void validateUnitPriceAmountOps(SplitRule rule) {
        try {
            Byte.valueOf(rule.getUnitPriceAmountOps());
        } catch (Exception e) {
            throw new SplitRuleParamException("打印单价数量规则有误");
        }
    }


    public static void validateCustomRemarkSize(SplitRule rule) {
        if (Objects.nonNull(rule.getCustomRemarkSize()) && rule.getCustomRemarkSize() < 0) {
            throw new SplitRuleParamException("自定义备注长度不能小于0");
        }
    }

    public static void validateSplitRule(SplitRule rule) {
        validateSaleListOption(rule);
        validateInvoiceItemMaxRow(rule);
        validateSalesListMaxRow(rule);
        validateUnitPriceAmountOps(rule);
    }


}
