package com.xforceplus.phoenix.split.service.dataflow.impl;

import com.google.common.collect.Lists;
import com.xforceplus.phoenix.split.constant.InvoiceItemOrder;
import com.xforceplus.phoenix.split.domain.ItemGroup;
import com.xforceplus.phoenix.split.domain.RuleInfo;
import com.xforceplus.phoenix.split.domain.SplitGroupLimit;
import com.xforceplus.phoenix.split.exception.SplitBizException;
import com.xforceplus.phoenix.split.exception.SplitRuleParamException;
import com.xforceplus.phoenix.split.model.BillInfo;
import com.xforceplus.phoenix.split.model.BillItem;
import com.xforceplus.phoenix.split.model.SplitRule;
import com.xforceplus.phoenix.split.service.PreInvoiceGenerateService;
import com.xforceplus.phoenix.split.service.SplitRuleUtil;
import com.xforceplus.phoenix.split.service.dataflow.DataProcessPlugin;
import com.xforceplus.phoenix.split.service.impl.DefaultSplitBillItemAmountServiceImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

import static com.xforceplus.phoenix.split.service.impl.DefaultSplitBillItemAmountServiceImpl.MIN_QUANTITY;
import static java.math.BigDecimal.ROUND_HALF_UP;

/**
 * 预制发票分组
 */
@Service
public class InvoiceLimitProcessPlugin implements DataProcessPlugin {

    private static final Logger logger = LoggerFactory.getLogger(InvoiceLimitProcessPlugin.class);

    @Override
    public List<ItemGroup> processData(List<ItemGroup> itemGroups, BillInfo billInfo, RuleInfo rule) {
        SplitRuleUtil.validateSplitRule(rule.getSplitRule());


        List<ItemGroup> allItemGroups = Lists.newLinkedList();
        for (ItemGroup itemGroup : itemGroups) {
            for (BillItem billItem : itemGroup.getBillItems()) {
                //未拆票金额计算
                if (!billItem.isSplitItem()) {
                    billItem.setAmountWithoutTax(billItem.getAmountWithoutTax().subtract(billItem.getInnerDiscountWithoutTax()).subtract(billItem.getInnerPrepayAmountWithoutTax()));
                    if (billItem.getInnerDiscountWithoutTax().compareTo(BigDecimal.ZERO) > 0 || billItem.getInnerPrepayAmountWithoutTax().compareTo(BigDecimal.ZERO) > 0) {
                        processPriceOrQuantity(billItem, rule);
                    }
                    billItem.setAmountWithTax(billItem.getAmountWithTax().subtract(billItem.getInnerDiscountWithTax()).subtract(billItem.getInnerPrepayAmountWithTax()));
                    billItem.setTaxAmount(billItem.getTaxAmount().subtract(billItem.getInnerPrepayAmountTax()).subtract(billItem.getInnerDiscountTax()));
                    billItem.setDiscountWithoutTax(billItem.getOutterDiscountWithoutTax().add(billItem.getOutterPrepayAmountWithoutTax()));
                    billItem.setDiscountWithTax(billItem.getOutterDiscountWithTax().add(billItem.getOutterPrepayAmountWithTax()));
                    billItem.setDiscountTax(billItem.getOutterDiscountTax().add(billItem.getOutterPrepayAmountTax()));
                }
            }


            boolean isCreateSalesList = true;
            String goodsTaxNo = itemGroup.getBillItems().get(0).getGoodsTaxNo();
            if (PreInvoiceGenerateService.HBBM_SET.contains(goodsTaxNo)) {
                isCreateSalesList = false;
            }
            int limitLine = this.getLimitLine(rule.getSplitRule(), isCreateSalesList);
            SplitGroupLimit splitGroupLimit = new SplitGroupLimit();
            splitGroupLimit.setInvoiceItemMode(rule.getSplitRule().getSaleListOption());
            splitGroupLimit.setLimitAmount(rule.getSplitRule().getInvoiceLimit());
            splitGroupLimit.setLimitLine(limitLine);
            splitGroupLimit.setLimitIsAmountWithTax(rule.getSplitRule().isLimitIsAmountWithTax());
            if (InvoiceItemOrder.ITEM_NO_ORDER.value().equals(rule.getSplitRule().getItemSort())) {
                splitGroupLimit.setInvoiceItemOrder(InvoiceItemOrder.ITEM_NO_ORDER);
                allItemGroups.addAll(orderSplitItemGroup(itemGroup, splitGroupLimit));
            } else if (InvoiceItemOrder.MINIMUM_INVOICES.value().equals(rule.getSplitRule().getItemSort())) {
                splitGroupLimit.setInvoiceItemOrder(InvoiceItemOrder.MINIMUM_INVOICES);
                allItemGroups.addAll(leastSplitItemGroup(itemGroup, splitGroupLimit));
            } else {
                throw new SplitRuleParamException("发票明细顺序规则有误");
            }
        }
        return allItemGroups;
    }

    private boolean splitRedInvoice(Map<String, Object> extRule) {
        return !StringUtils.isEmpty(extRule.get(SplitBillItemPlugin.ORIGIN_INVOICE_CODE)) ||
                !StringUtils.isEmpty(extRule.get(SplitBillItemPlugin.ORIGIN_INVOICE_NO)) ||
                !StringUtils.isEmpty(extRule.get(SplitBillItemPlugin.RED_NOTIFICATION_NO));
    }

    private void processPriceOrQuantity(BillItem billItem, RuleInfo rule) {
        if (DefaultSplitBillItemAmountServiceImpl.isQuantity(rule.getSplitRule().getAmountSplitRule())) {
            if (billItem.getQuantity() != null && billItem.getQuantity().compareTo(BigDecimal.ZERO) > 0) {
                billItem.setUnitPrice(billItem.getAmountWithoutTax().divide(billItem.getQuantity(), 15, ROUND_HALF_UP));
            }
        } else {
            if (billItem.getUnitPrice() != null && billItem.getUnitPrice().compareTo(BigDecimal.ZERO) > 0) {
                BigDecimal quantity = billItem.getAmountWithoutTax().divide(billItem.getUnitPrice(), 6,
                        ROUND_HALF_UP);
                if (quantity.compareTo(BigDecimal.ZERO) == 0) {
                    quantity = MIN_QUANTITY;
                }
                if (quantity.multiply(billItem.getUnitPrice()).setScale(2, ROUND_HALF_UP)
                        .compareTo(billItem.getAmountWithoutTax()) != 0) {
                    billItem.setUnitPrice(billItem.getAmountWithoutTax().divide(quantity, 15, ROUND_HALF_UP));
                }
                billItem.setQuantity(quantity);
            }
        }
    }


    private List<ItemGroup> orderSplitItemGroup(ItemGroup itemGroup, SplitGroupLimit splitGroupLimit) {
        List<ItemGroup> processedItemGroups = Lists.newLinkedList();

        logger.debug("itemGroup.getBillItems size:{}", itemGroup.getBillItems().size());
        while (itemGroup.getBillItems().size() > 0) {
            List<BillItem> processedItemList = splitByAmountAndLineLimit(itemGroup.getBillItems(), splitGroupLimit);
            if (processedItemList.size() == 0) {
                throw new SplitBizException("拆票请求数据异常，请检查价内外折扣以及税额是否正确");
            }
            ItemGroup processedItemGroup = new ItemGroup();
            processedItemGroup.setBillItems(processedItemList);
            processedItemGroups.add(processedItemGroup);
        }

        return processedItemGroups;
    }


    private List<ItemGroup> leastSplitItemGroup(ItemGroup itemGroup, SplitGroupLimit splitGroupLimit) {
        List<ItemGroup> allItemGroups = Lists.newLinkedList();

        List<BillItem> billItems = itemGroup.getBillItems();
        Map<Boolean, List<BillItem>> isSplitItemMap = billItems.stream().collect(Collectors.groupingBy(BillItem::isSplitItem));

        List<BillItem> processItems = new LinkedList<>();
        //未拆分过金额的明细,需要处理分组
        List<BillItem> isNotSplitBillItems = isSplitItemMap.get(false);
        if (!CollectionUtils.isEmpty(isNotSplitBillItems)) {
            processItems.addAll(isNotSplitBillItems);
        }
        //拆分过金额的明细,每张单据最后一条需要处理分组
        List<BillItem> isSplitBillItems = isSplitItemMap.get(true);

        List<BillItem> noNeedProcessItems = Lists.newLinkedList();

        if (!CollectionUtils.isEmpty(isSplitBillItems)) {
            Map<String, List<BillItem>> billIdItemsMap = isSplitBillItems.stream().collect(Collectors.groupingBy(BillItem::getSalesbillItemId));
            for (Map.Entry<String, List<BillItem>> stringListEntry : billIdItemsMap.entrySet()) {
                List<BillItem> items = stringListEntry.getValue();
                BillItem item = items.get(items.size() - 1);
                items.remove(item);
                processItems.add(item);
                noNeedProcessItems.addAll(items);
            }
            for (BillItem noNeedProcessItem : noNeedProcessItems) {
                ItemGroup notNeedProcessItemGroup = new ItemGroup();
                notNeedProcessItemGroup.setBillItems(Lists.newArrayList(noNeedProcessItem));
                allItemGroups.add(notNeedProcessItemGroup);
            }
        }
        if (!CollectionUtils.isEmpty(processItems)) {
            //按照金额倒排
            processItems = processItems.stream()
                    .sorted(((Comparator<BillItem>) (o1, o2) -> {
                        BigDecimal value1 = actualAmount(o1, splitGroupLimit.isLimitIsAmountWithTax());
                        BigDecimal value2 = actualAmount(o2, splitGroupLimit.isLimitIsAmountWithTax());
                        return value1.compareTo(value2);
                    }).reversed())
                    .collect(Collectors.toList());
            logger.debug("itemGroup.processItems size:{}", processItems.size());
            while (processItems.size() > 0) {
                List<BillItem> processedItemList = splitByAmountAndLineLimit(processItems, splitGroupLimit);
                if (processedItemList.size() == 0) {
                    throw new SplitBizException("拆票请求数据异常，请检查价内外折扣以及税额是否正确");
                }
                ItemGroup processedItemGroup = new ItemGroup();
                processedItemGroup.setBillItems(processedItemList);
                allItemGroups.add(processedItemGroup);
            }
        }


        return allItemGroups;
    }


    /**
     * 明细实际金额
     *
     * @param item 单据明细
     * @param limitIsAmountWithTax
     * @return 明细实际金额
     */
    private BigDecimal actualAmount(BillItem item, boolean limitIsAmountWithTax) {
        BigDecimal result;

        if (limitIsAmountWithTax) {
            result = item.getAmountWithTax().subtract(item.getDiscountWithTax());
        } else {
            result = item.getAmountWithoutTax().subtract(item.getDiscountWithoutTax());
        }

        return result;
    }

    /**
     * 明细实际double金额
     *
     * @param item 单据明细
     * @return 明细实际金额
     */
    private double actualAmountDoubleValue(BillItem item, boolean limitIsAmountWithTax) {
        return actualAmount(item, limitIsAmountWithTax).doubleValue();
    }

    /**
     * 1.迭代billItems
     * 2.从billItems中选第一个元素a
     * 3 判断是否为差额征税
     * 4.如果是，splitGroup.add(a),billItems remove a,结束
     * 5.如果不是，计算billItems加上a之后是否满足行数和限额要求以及税额误差不超过1.27
     * 6.如果满足，splitGroup.add(a),billItems remove a,回到第一步。
     * 7.如果不满足,判断是否为顺序优先，不是:【回到第2步】，是:【迭代结束】。
     * 8.迭代结束。
     *
     * @param billItems 待分组单据明细列表
     * @return 包括billItems中第一条明细的一组明细, 其合计金额不超过限额，数量不超过限制行数，且剩余明细中【张数：任一条明细】/【顺序：下一条明细】再添加到此结果中将会不满足限额或者行数限制。
     */
    protected List<BillItem> splitByAmountAndLineLimit(List<BillItem> billItems, SplitGroupLimit splitGroupLimit) {
        logger.debug("splitByAmountAndLineLimit billItems.size:{}", billItems.size());
        BigDecimal amount = BigDecimal.valueOf(0);
        int lineNum = 0;
        // 不四舍五入得到的总税额
        BigDecimal sum1 = BigDecimal.valueOf(0);
        // 四舍五入得到的总税额
        BigDecimal sum2 = BigDecimal.valueOf(0);
        List<BillItem> splitGroup = Lists.newLinkedList();
        for (Iterator<BillItem> it = billItems.iterator(); it.hasNext(); ) {
            BillItem item = it.next();
            // 差额征税时，单条开票
            if (item.getDeductions().compareTo(BigDecimal.ZERO) > 0) {
                if (splitGroup.size() > 0) {
                    continue;
                }
                splitGroup.add(item);
                it.remove();
                return splitGroup;
            }
            // 金额限制
            boolean isAmountGtLimit = amount.add(actualAmount(item, splitGroupLimit.isLimitIsAmountWithTax())).compareTo(splitGroupLimit.getLimitAmount()) > 0;
            // 行数限制
            boolean isLineGtLimit = lineNum + this.needLineNum(item) > splitGroupLimit.getLimitLine();
            // 税额误差不超过1.27
            BigDecimal tempSum1 = sum1.add(item.getTaxAmount().subtract(item.getDiscountTax()));
            BigDecimal tempSum2 = sum2.add((item.getAmountWithoutTax().subtract(item.getDiscountWithoutTax())).multiply(item.getTaxRate()).setScale(2, BigDecimal.ROUND_HALF_UP));
            boolean rateFor1dot27 = tempSum1.subtract(tempSum2).abs().compareTo(BigDecimal.valueOf(1.27)) < 0;
            //当行数超限 或 税额误差超过1.27
            if (isLineGtLimit || !rateFor1dot27) {
                return splitGroup;
            }
            if (!isAmountGtLimit) {
                amount = amount.add(actualAmount(item, splitGroupLimit.isLimitIsAmountWithTax()));
                lineNum += this.needLineNum(item);
                sum1 = sum1.add(item.getAmountWithoutTax().multiply(item.getTaxRate()));
                sum2 = sum2.add(item.getAmountWithoutTax().multiply(item.getTaxRate()).setScale(2, BigDecimal.ROUND_HALF_UP));
                splitGroup.add(item);
                it.remove();
            } else if (splitGroupLimit.getInvoiceItemOrder() == InvoiceItemOrder.ITEM_NO_ORDER) {
                return splitGroup;
            }

        }
        return splitGroup;
    }


    /**
     * 计算行数限制
     *
     * @param rule 拆票规则
     * @return 发票行数限制
     */
    private int getLimitLine(SplitRule rule, boolean isCreateSalesList) {
        if (!isCreateSalesList) {
            return rule.getInvoiceItemMaxRow();
        }
        if ("0".equals(rule.getSaleListOption())) {
            return rule.getInvoiceItemMaxRow();
        } else if ("1".equals(rule.getSaleListOption())) {
            return Math.max(SplitRuleUtil.getSalesListMaxRow(rule), rule.getInvoiceItemMaxRow());
        } else if ("2".equals(rule.getSaleListOption())) {
            return SplitRuleUtil.getSalesListMaxRow(rule);
        }
        return 0;
    }

    /**
     * 返回明细生成预制发票时占用行数
     *
     * @param item 单据明细
     * @return 明细生成预制发票时占用行数
     */
    private int needLineNum(BillItem item) {

        if (item.getDiscountWithoutTax().compareTo(BigDecimal.ZERO) > 0
                && (item.getOutterDiscountWithoutTax().compareTo(BigDecimal.ZERO) > 0
                // 有折扣时,判断税率汇总计数行是否存在
                || item.getOutterPrepayAmountWithoutTax().compareTo(BigDecimal.ZERO) > 0)) {
            return 2;
        }
        return 1;
    }
}
