package com.xforceplus.phoenix.split.service.impl;

import com.google.common.base.Strings;
import com.xforceplus.phoenix.split.constant.InvoiceType;
import com.xforceplus.phoenix.split.model.*;
import com.xforceplus.phoenix.split.service.PreInvoiceGenerateService;
import com.xforceplus.phoenix.split.service.RemarkService;
import com.xforceplus.phoenix.split.util.CommonTools;
import com.xforceplus.phoenix.split.util.FiledUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.ibatis.reflection.ReflectionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.xforceplus.phoenix.split.service.PreInvoiceGenerateService.*;

/**
 * 备注ServiceImpl
 */
@Service
public class RemarkServiceImpl implements RemarkService {
    private static final Logger logger = LoggerFactory.getLogger(RemarkServiceImpl.class);

    @Override
    /**
     * 备注生成
     *
     * @param billInfo 单据信息
     * @param rule                拆票规则
     * @return 备注
     */
    public String splice(BillInfo billInfo, SplitRule rule) {
        // initialize.
        StringBuilder sb = new StringBuilder();

        PriceMethod priceMethod = rule.getPriceMethod();

        List<BillItem> billItems = billInfo.getBillItems();

        //红字发票备注
        if (InvoiceType.SPECIAL.value().equals(billInfo.getInvoiceType())) {
            if (StringUtils.isNotBlank(billInfo.getRedNotificationNo())) {
                sb.append(CommonTools.format(PreInvoiceGenerateService.RED_SPECIAL_REMARK, billInfo.getRedNotificationNo()));
            }
        } else if (InvoiceType.NORMAL.value().equals(billInfo.getInvoiceType()) || InvoiceType.ELECTRONIC.value().equals(billInfo.getInvoiceType())) {
            if (StringUtils.isNotBlank(billInfo.getOriginInvoiceCode())) {
                sb.append(RED_NORMAL_REMARK).append(CommonTools.format(RED_NORMAL_REMARK_CODE, billInfo.getOriginInvoiceCode()));
                if (StringUtils.isNotBlank(billInfo.getOriginInvoiceNo())) {
                    sb.append(CommonTools.format(RED_NORMAL_REMARK_NO, billInfo.getOriginInvoiceNo()));
                }
            } else if (StringUtils.isNotBlank(billInfo.getOriginInvoiceNo())) {
                sb.append(RED_NORMAL_REMARK).append(CommonTools.format(RED_NORMAL_REMARK_NO, billInfo.getOriginInvoiceNo()));
            }
        }


//        if (null2Zero(billItems.get(0).getDeductions()).compareTo(BigDecimal.ZERO) > 0) {
//            appendSpace(sb);
//            BigDecimal deduction = null2Zero(billItems.get(0).getDeductions());
//            //差额征税：100.00
//            sb.append(CommonTools.format(DEDUCTION_REMARK, deduction.setScale(2, ROUND_HALF_UP).toPlainString()));
//        }

        List<RemarkFieldMetadata> remarkFiledMetadataBeanList = rule.getRemarkFiledMetadataBeanList();
        if (CollectionUtils.isNotEmpty(remarkFiledMetadataBeanList)) {
            //业务单备注
            long size = remarkFiledMetadataBeanList.stream()
                    .filter(remarkFieldMetadata -> "remark".equalsIgnoreCase(remarkFieldMetadata.getFieldName())).count();

            if (size > 0) {
                appendSpace(sb);
                sb.append(Objects.toString(billInfo.getRemark(), ""));
            }
            appendSpace(sb);


            //主信息明细规则备注
            for (RemarkFieldMetadata metadataBean : remarkFiledMetadataBeanList) {
                if (metadataBean.getFieldGroupIndex() == FieldOfObj.BILL_INFO) {

                    if (StringUtils.equals(metadataBean.getFieldName(), "remark")) {
                        continue;
                    }
                    if (StringUtils.equals(metadataBean.getFieldName(), "fixedRemark")) {
                        sb.append(Strings.nullToEmpty(rule.getFixedRemarkText()));
                    } else if (StringUtils.equals(metadataBean.getFieldName(), "priceMethod")) {
                        sb.append(metadataBean.getFieldDisplayName()).append("[").append(this.getPriceMethod(priceMethod)).append("]");
                    } else {
                        sb.append(metadataBean.getFieldDisplayName()).append("[").append(getReMark(billInfo, metadataBean.getFieldName())).append("]");
                    }
                } else {
                    //明细备注
                    String fieldValue = "";
                    if (StringUtils.equals(metadataBean.getFieldName(), "taxPre")) {
                        fieldValue = billItems.stream().map(billItem -> this.getTaxPre(billItem.getTaxPre())).distinct().collect(Collectors.joining(RemarkService.DAWN));
                    } else if (StringUtils.equals(metadataBean.getFieldName(), "zeroTax")) {
                        fieldValue = billItems.stream().map(billItem -> this.getZeroTaxMark(billItem.getZeroTax())).distinct().collect(Collectors.joining(RemarkService.DAWN));
                    } else {
                        fieldValue = billItems.stream()
                                .map(billItem -> this.getReMark(billItem, metadataBean.getFieldName())).distinct()
                                .collect(Collectors.joining(RemarkService.DAWN));
                    }

                    sb.append(metadataBean.getFieldDisplayName()).append("[").append(fieldValue).append("]");

                }
                //最后一个规则备注不加","
                appendSpace(sb);
            }
        }
        appendSpace(sb);
        if (StringUtils.isNotBlank(rule.getExtRemark())) {
            sb.append(rule.getExtRemark());
        }
        return sb.toString();
    }

    private void appendSpace(StringBuilder sb) {
        if (StringUtils.isNotBlank(sb.toString()) && !StringUtils.endsWith(sb.toString(), RemarkService.SPACE)) {
            sb.append(RemarkService.SPACE);
        }
    }


    /**
     * 获取发票主信息备注
     *
     * @param billInfo  单据主信息
     * @param fieldName 字段名称
     * @return 发票主信息备注
     */
    private String getReMark(BillInfo billInfo, String fieldName) {
        try {
            return Objects.toString(FiledUtils.getFieldValue(billInfo, fieldName), "");
        } catch (ReflectionException e) {
            logger.warn("元数据获取字段值错误:{}", e.getMessage());
        }
        return "";
    }

    /**
     * 获取发票明细备注
     *
     * @param billItem  单据明细
     * @param fieldName 字段名称
     * @return 发票明细备注
     */
    private String getReMark(BillItem billItem, String fieldName) {
        try {
            return Objects.toString(FiledUtils.getFieldValue(billItem, fieldName), "");
        } catch (ReflectionException e) {
            logger.warn("元数据获取字段值错误:{}", e.getMessage());
        }
        return "";
    }

    /**
     * 是否享受税收优惠政策转换
     *
     * @param taxPre 是否享受税收优惠政策 0 -不 1-享受
     * @return 是否享受税收优惠政策 中文
     */
    private String getTaxPre(String taxPre) {
        return StringUtils.equals(taxPre, "1") ? "享受" : "不享受";
    }

    /**
     * 零税率标识转换
     *
     * @param zeroTax 零税率标识
     * @return 零税率标识中文
     */
    private String getZeroTaxMark(String zeroTax) {
        return StringUtils.equals(zeroTax, "0") ? "出口退税"
                : StringUtils.equals(zeroTax, "1") ? "免税"
                : StringUtils.equals(zeroTax, "2") ? "不征税"
                : StringUtils.equals(zeroTax, "3") ? "普通零税率"
                : "非零税率";
    }

    /**
     * 价格方式转换
     *
     * @param priceMethod 价格方式 1-含税 2-不含税
     * @return 价格方式中文
     */
    private String getPriceMethod(PriceMethod priceMethod) {
        return priceMethod == PriceMethod.WITH_TAX ? "含税" : "不含税";

    }
}
