package com.xforceplus.action.trail.thread.config;

import com.alibaba.ttl.spi.TtlWrapper;
import com.alibaba.ttl.threadpool.TtlExecutors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.MapUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnExpression;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.annotation.EnableAsync;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;

import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.ThreadPoolExecutor;


/**
 * @author duanhy
 */
@EnableAsync
@ConditionalOnExpression("${action.trail.enable:false}")
@Configuration(value = "globalTrailThreadPoolConfig")
@EnableConfigurationProperties({ThreadPoolConfigProps.class})
@Slf4j
public class TrailThreadPoolConfig implements ApplicationContextAware, DisposableBean {

    private final ThreadPoolConfigProps threadPoolConfigProps;
    private static final String THREAD_NAME_PREFIX = "async-thread-trail-";
    public static final String ASYNC_THREAD_POOL_EXECUTOR_TRAIL_NAME = "asyncThreadPoolExecutorTrail";

    private ApplicationContext applicationContext;

    public TrailThreadPoolConfig(ThreadPoolConfigProps threadPoolConfigProps) {
        log.info("TrailThreadPoolConfig-initialized,threadPoolConfigProps:{}", threadPoolConfigProps);
        this.threadPoolConfigProps = threadPoolConfigProps;
    }

    @Override
    public void setApplicationContext(@NotNull ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Bean(ASYNC_THREAD_POOL_EXECUTOR_TRAIL_NAME)
    public ThreadPoolExecutor getAsyncThreadPoolExecutor() {
        TtlWrapper<ThreadPoolExecutor> ttlWrapper = (TtlWrapper<ThreadPoolExecutor>) this.getAsyncExecutor();
        if (ttlWrapper == null) {
            throw new RuntimeException("ttlWrapper is null");
        }
        ThreadPoolExecutor executor = ttlWrapper.unwrap();
        return executor;
    }


    private Executor getAsyncExecutor() {
        ThreadPoolTaskExecutor executor = new ThreadPoolTaskExecutor();
        // 配置核心线程数
        executor.setCorePoolSize(threadPoolConfigProps.getCorePoolSize());
        // 配置最大线程数
        executor.setMaxPoolSize(threadPoolConfigProps.getMaxPoolSize());
        // 配置队列大小
        executor.setQueueCapacity(threadPoolConfigProps.getQueueCapacity());
        //配置线程池中的线程的名称前缀
        executor.setThreadNamePrefix(THREAD_NAME_PREFIX);
        // rejection-policy：当pool已经达到max size的时候，如何处理新任务
        // CALLER_RUNS：不在新线程中执行任务，而是有调用者所在的线程来执行
        executor.setRejectedExecutionHandler(new ThreadPoolExecutor.CallerRunsPolicy());
        //执行初始化
        executor.initialize();
        //TtlExecutors使用这个包装，这个在前置知识第四点的博客中可以明白其原理
        return TtlExecutors.getTtlExecutorService(executor.getThreadPoolExecutor());
    }

    @Override
    public void destroy() {
        Map<String, ThreadPoolExecutor> executorMap = applicationContext.getBeansOfType(ThreadPoolExecutor.class);
        if (MapUtils.isNotEmpty(executorMap)) {
            executorMap.forEach((key, executor) -> {
                try {
                    executor.shutdown();
                } catch (Exception e) {
                    log.warn("{}.TrailThreadPoolExecutor shutdown failed, reason:{}", key, e.toString());
                }
            });
        }
    }
}
