package com.xforceplus.action.trail.servlet.interceptor;

import com.xforceplus.action.trail.core.context.TrailInfoHolder;
import com.xforceplus.action.trail.core.domain.Operator;
import com.xforceplus.action.trail.core.domain.TrailInfo;
import com.xforceplus.action.trail.core.utils.IpUtils;
import com.xforceplus.action.trail.core.utils.Separator;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import com.xforceplus.tenant.security.core.context.ClientInfoHolder;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import io.geewit.utils.uuid.UUID;
import io.geewit.web.utils.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.resource.ResourceHttpRequestHandler;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.lang.reflect.Method;
import java.util.*;

import static com.xforceplus.action.trail.core.constants.ActionTrailConstants.TRACE_ID;


/**
 * @author duanhy
 */
@Slf4j
public class ActionTrailInterceptor implements HandlerInterceptor {

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception arg3) {
        TrailInfoHolder.clearContext();
        log.info("action.trail.infoHolder.clear...");
    }

    @Override
    public void postHandle(HttpServletRequest request, HttpServletResponse response, Object handler, ModelAndView arg3) {
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) {

        if (handler instanceof ResourceHttpRequestHandler) {
            log.debug("handler instanceof ResourceHttpRequestHandler");
            return true;
        }

        boolean isStaticPage = isStaticPage(request.getRequestURI());
        if (isStaticPage) {
            log.debug("isStaticPage: true, return");
            return true;
        }

        if (handler instanceof HandlerMethod) {
            String httpMethod = request.getMethod();
            String traceId = request.getHeader(TRACE_ID);
            String url = request.getRequestURL().toString();
            if (StringUtils.isEmpty(traceId)) {
                traceId = UUID.randomUUID().toString();
            }
            log.info("traceId = {}, url = {}, httpMethod = {}, ", traceId, url, httpMethod);
            response.setHeader(TRACE_ID, traceId);
            String resourceCode = "";
            HandlerMethod handlerMethod = (HandlerMethod) handler;
            Method method = handlerMethod.getMethod();
            AuthorizedDefinition authorizedDefinition = method.getAnnotation(AuthorizedDefinition.class);
            if (null != authorizedDefinition) {
                resourceCode = StringUtils.join(authorizedDefinition.resources(), ",");
            }
            TrailInfo trailInfo = new TrailInfo();
            trailInfo.setEventId(traceId);
            trailInfo.setIp(IpUtils.getIp(request));
            trailInfo.setUri(url);
            trailInfo.setHttpMethod(request.getMethod());
            trailInfo.setPath(request.getRequestURI());
            trailInfo.setResourceCode(resourceCode);
            IAuthorizedUser iAuthorizedUser = UserInfoHolder.get();
            if (null != iAuthorizedUser) {
                Operator operator = new Operator(iAuthorizedUser);
                trailInfo.setOperator(operator);
            }
            log.info("traceId = {},IAuthorizedUser = {}", traceId, JsonUtils.toJson(trailInfo));
            trailInfo.setClientId(ClientInfoHolder.get());
            TrailInfoHolder.put(trailInfo);
            return true;
        }
        return true;
    }

    private HttpServletRequest getRequest() {
        return ((ServletRequestAttributes) (RequestContextHolder.currentRequestAttributes())).getRequest();
    }

    private Map<String, String> getRequestHeaderMap() {
        HttpServletRequest request = getRequest();
        Enumeration<String> headerNames = request.getHeaderNames();
        Map<String, String> headerMap = new HashMap<>();
        while (headerNames.hasMoreElements()) {
            String name = headerNames.nextElement();
            headerMap.put(name, request.getHeader(name));
        }
        return headerMap;
    }


    public static boolean isStaticPage(String url) {
        List<String> suffixList = Arrays.asList(".html", ".css", ".js", ".css", ".png", ".ttf");
        List<String> passList = Arrays.asList("swagger-resources/configuration/ui", "swagger-resources", "v2/api-docs", "swagger-resources/configuration/security", "health", "xplat/health", "error");
        if (StringUtils.isBlank(url)) {
            return false;
        }
        String path = url.trim();
        if (StringUtils.startsWith(path, Separator.BACKSLASH)) {
            path = StringUtils.substring(path, 1);
        }
        boolean match = suffixList.stream().anyMatch(path::endsWith) || passList.stream().anyMatch(path::equals);
        log.debug("path=={},match=={}", url, match);
        return match;
    }
}
