package com.xforceplus.antc.snowflake;

import org.apache.commons.lang3.RandomUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.net.InetAddress;

@Component
public class AntcSnowflakeUtil {
  private static final Logger logger = LoggerFactory.getLogger(AntcSnowflakeUtil.class);

  @Autowired
  private Environment environment;
  @Autowired
  AntcSnowflakeConfig antcSnowflakeConfig;
  @Autowired
  AntcSnowflakeService snowflakeldCheckService;

  @PostConstruct
  public void init() {
    Integer datacenterId = antcSnowflakeConfig.getAppId();
    if (null == datacenterId) {
      datacenterId = RandomUtils.nextInt(1, 31);
      logger.warn("##### AntcSnowflake应用未配置${antc.snowflake.appId}; datacenterId={} (随机生成)", datacenterId);
    } else {
      logger.info("##### AntcSnowflake应用已配置${antc.snowflake.appId} datacenterId={}", datacenterId);
    }
    int workerId;
    String ip = null;
    try {
      InetAddress ia = InetAddress.getLocalHost();
      ip = ia.getHostAddress();
      String localip = StringUtils.substringAfterLast(ip, ".");
      workerId = Integer.parseInt(localip);
    } catch (Exception e) {
      workerId = RandomUtils.nextInt(1, 31);
    }
    logger.info("##### 初始化该节点分配的workerID={}；(workerID & 31)={}", workerId, (workerId & 31));
    long checkWorkerId = workerId & 31;
    long checkDatacenterId = datacenterId & 31;
    // 检测
    if (antcSnowflakeConfig.getRemoteCheckFlag() && StringUtils.isNotBlank(antcSnowflakeConfig.getRemoteCheckUrl())) {
      try {
        for (int i = 1; i <= 35; i++) {
          if (snowflakeldCheckService.check(checkDatacenterId, checkWorkerId, ip)) {
            snowflakeldCheckService.timer();
            break;
          }
          checkWorkerId++;
          checkWorkerId = (checkWorkerId & 31);
          logger.info("##### 分布式ID 数据检测[第{}次], checkDatacenterId:{};checkWorkerId:{};ip:{}", i, checkDatacenterId, checkWorkerId, ip);
        }
      } catch (Exception e) {
        logger.error("##### 分布式ID 数据检测异常：", e.getMessage());
      }
    } else {
      logger.info("##### AntcSnowflake 远程监测服务已关闭");
    }

    logger.info("##### 检测后的该节点分配的datacenterId={}；(workerID & 31)={}", checkDatacenterId, checkWorkerId);
    snowflakeIdWorker = new AntcSnowflakeWorker(checkWorkerId, checkDatacenterId);
    logger.info("##### snowflakeIdWorker IOC实例：{}", snowflakeIdWorker);
  }


  static AntcSnowflakeWorker snowflakeIdWorker;

  static {
    snowflakeIdWorker = new AntcSnowflakeWorker(RandomUtils.nextInt(1, 1000) & 31, RandomUtils.nextInt(1, 1000) & 31);
    logger.info("##### snowflakeIdWorker 静态实例：{}", snowflakeIdWorker);
  }

  /**
   * 单个ID 生产
   *
   * @return
   */
  public static long nextId() {
    return snowflakeIdWorker.nextId();
  }

  /**
   * 批量生产ID(上限：100000)
   *
   * @param size
   * @return
   */
  public static long[] nextId(int size) {
    return snowflakeIdWorker.nextId(size);
  }


}