package com.xforceplus.bi.commons.authority.encryptions.xplat.beans;

import com.xforceplus.bi.commons.http.OkHttpInstance;
import com.xforceplus.bi.commons.integration.platform.AuthSource;
import com.xforceplus.bi.commons.integration.user.beans.UserInfo;
import com.xforceplus.bi.commons.integration.user.beans.UserModelPermission;
import com.xforceplus.bi.commons.integration.user.xplat.XplatTokenBody;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.ExpiredJwtException;
import io.jsonwebtoken.Jws;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.impl.TextCodec;
import io.jsonwebtoken.impl.crypto.DefaultSignatureValidatorFactory;
import io.jsonwebtoken.impl.crypto.SignatureValidatorFactory;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.util.CollectionUtils;

import javax.crypto.spec.SecretKeySpec;
import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.util.*;
import java.util.stream.Stream;

import static io.jsonwebtoken.SignatureAlgorithm.HS256;
import static java.nio.charset.StandardCharsets.UTF_8;

@Slf4j
public class XplatParseToken {

    public static final String OPERATION_TOKEN = "X-Operation-Token";
    public static final String innerCompanyCodeRangeName = "CompanyCodeRange";
    public static final String outterTaxNoRangeName = "OutterTaxNoRange";
    public static final String innerExt1OptionsName = "InnerExt1";
    public static final String innerExt2OptionsName = "InnerExt2";
    public static final String innerExt3OptionsName = "InnerExt3";
    public static final String ACCESS_TOKEN_URL = "%s/security/access-token?corp_id=%s&corp_secret=%s";
    public static final String DATA_PERMISSION_URL = "%s/zapi-v1/dataset/getRangeList?access_token=";
    private final static String securityKey = "XplatSecret_#0303";
    public static final String TYPE = "typ";
    public static final String DISPLAY_NAME = "din";
    private String signingKey = TextCodec.BASE64.encode(securityKey);

    @Autowired
    private XplatAccessTokenProperties xplatAccessTokenProperties;

    @Value("${xforce.permissionCode:b0201004}")
    private String permissionCode;

    @Autowired
    private OkHttpInstance okHttpInstance;

    /**
     * 解析Token
     *
     * @param token
     * @return
     */
    public UserInfo<XplatTokenBody> validateToken(String token) throws IOException {
        Jws<Claims> jwt = null;
        try {
            jwt = Jwts.parser().setSigningKey(signingKey).parseClaimsJws(token);
        } catch (ExpiredJwtException e) {
            log.info("TOken过期");
            return null;
        } catch (Exception e) {
            log.info("TOken错误");
            return null;
        }
        Claims claims = jwt.getBody();
        XplatTokenBody xplatTokenBody = new XplatTokenBody(claims.getSubject(), claims.get(DISPLAY_NAME, String.class), claims.get(TYPE, String.class), claims.getExpiration(), claims);
        if (xplatTokenBody == null) {
            return null;
        }
        UserInfo userVo = new UserInfo();
        String athenaUser = xplatTokenBody.getSubject();
        userVo.setUsername(athenaUser);
        if (xplatTokenBody.getAdditionalProperties() != null) {
            String athenaTenant = (String) xplatTokenBody.getAdditionalProperties().get("TENANT_CODE");
            userVo.setTenantCode(athenaTenant);
            String mobile = (String) xplatTokenBody.getAdditionalProperties().get("MOBILE");
            userVo.setMobile(mobile);
            String name = (String) xplatTokenBody.getAdditionalProperties().get("din");
            userVo.setName(name);
            String fuc = (String) xplatTokenBody.getAdditionalProperties().get("fuc");
            userVo.setFuc(fuc);
            String email = (String) xplatTokenBody.getAdditionalProperties().get("email");
            userVo.setEmail(email);
        }
        userVo.setAuthSource(AuthSource.XPLAT);
        List<DataSetVo> dataPermission = getUserDataPermission(userVo.getUsername(), permissionCode, xplatAccessTokenProperties);
        Map pemission = new HashMap();
        pemission.put("default", commonXplatUserParams(dataPermission));
        userVo.setPermissionParamS(pemission);
        // 设置平台用户的原始值
        userVo.setOrigin(xplatTokenBody);
        return userVo;
    }

    /**
     * 获取数据权限
     *
     * @param subject
     * @param operationName
     * @param xplatAccessTokenProperties
     * @return
     * @throws IOException
     */
    private List<DataSetVo> getUserDataPermission(String subject, String operationName, XplatAccessTokenProperties xplatAccessTokenProperties) throws IOException {

        String openAPIAccessTokenURL = String.format(ACCESS_TOKEN_URL, xplatAccessTokenProperties.getHost(), xplatAccessTokenProperties.getCorpId(), xplatAccessTokenProperties.getCorpSecret());
        ResponseAccessToken accessToken = okHttpInstance.postBodyToClass(openAPIAccessTokenURL, "{}", null, ResponseAccessToken.class);
        if (accessToken == null || !"1".equals(accessToken.getCode())) {
            return null;
        }
        String permissionAPIURL = String.format(DATA_PERMISSION_URL, xplatAccessTokenProperties.getHost()).concat(accessToken.getResult());
        Map<String, String> params = new HashMap<String, String>();
        params.put("account", subject);
        params.put("nodeEname", operationName);
        DatasetGetRangeListResponse datasets = okHttpInstance.postBodyToClass(permissionAPIURL, params, null, DatasetGetRangeListResponse.class);
        if (datasets == null) {
            return null;
        }
        if (datasets.getCode() != 1) {
            return null;
        }
        return datasets.getListDataset();
    }

    /**
     * 数据权限转换
     *
     * @param dataSetVoList
     * @return
     */
    private List<UserModelPermission.DataPermission> commonXplatUserParams(List<DataSetVo> dataSetVoList) {
        List<UserModelPermission.DataPermission> permissionBeanList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(dataSetVoList)) {
            Set<String> innerCompanyCodeRange = new HashSet<>();
            Set<String> outterTaxNoRange = new HashSet<String>();
            Set<String> innerExt1Options = new HashSet<String>();
            Set<String> innerExt2Options = new HashSet<String>();
            Set<String> innerExt3Options = new HashSet<String>();

            for (DataSetVo dto : dataSetVoList) {

                if (StringUtils.isNotEmpty(dto.getInnerCompanyCodeRange())) {
                    innerCompanyCodeRange.add(dto.getInnerCompanyCodeRange());
                }

                if (StringUtils.isNotEmpty(dto.getOutterTaxNoRange())) {
                    outterTaxNoRange.add(dto.getOutterTaxNoRange());
                }

                if (StringUtils.isNotEmpty(dto.getInnerExt1Options())) {
                    innerExt1Options.add(dto.getInnerExt1Options());
                }

                if (StringUtils.isNotEmpty(dto.getInnerExt2Options())) {
                    innerExt2Options.add(dto.getInnerExt2Options());
                }

                if (StringUtils.isNotEmpty(dto.getInnerExt3Options())) {
                    innerExt3Options.add(dto.getInnerExt3Options());
                }
            }

            if (!innerCompanyCodeRange.isEmpty()) {
                String innerCompanyCodeRangeStr = StringUtils.join(innerCompanyCodeRange, ",");
                addPermissionBi(innerCompanyCodeRangeName, innerCompanyCodeRangeStr, permissionBeanList);
            }
            if (!outterTaxNoRange.isEmpty()) {
                String outterTaxNoRangeStr = StringUtils.join(outterTaxNoRange, ",");
                addPermissionBi(outterTaxNoRangeName, outterTaxNoRangeStr, permissionBeanList);
            }
            if (!innerExt1Options.isEmpty()) {
                String innerExt1OptionsStr = StringUtils.join(innerExt1Options, ",");
                addPermissionBi(innerExt1OptionsName, innerExt1OptionsStr, permissionBeanList);
            }
            if (!innerExt2Options.isEmpty()) {
                String innerExt2OptionsStr = StringUtils.join(innerExt2Options, ",");
                addPermissionBi(innerExt2OptionsName, innerExt2OptionsStr, permissionBeanList);
            }
            if (!innerExt3Options.isEmpty()) {
                String innerExt3OptionsStr = StringUtils.join(innerExt3Options, ",");
                addPermissionBi(innerExt3OptionsName, innerExt3OptionsStr, permissionBeanList);
            }
        }

        return permissionBeanList;
    }

    /**
     * 数据权限字段转换
     *
     * @param field
     * @param value
     * @param biPermissionBeans
     */
    private void addPermissionBi(String field, String value, List<UserModelPermission.DataPermission> biPermissionBeans) {
        UserModelPermission.DataPermission biPermissionBean = new UserModelPermission.DataPermission();
        biPermissionBean.setFieldType("string");
        // biPermissionBean.setSelectType(BiPermissionType.SELECT.getValue()); // 现在可能有多种筛选方式,需要放到值List中
        biPermissionBean.setFieldRealName(field);
        // biPermissionBean.setValueList(Arrays.asList(value)); // 现在可能有多种筛选方式,需要放到值List中
        // 现在可能有多种筛选方式,需要放到值List中
        UserModelPermission.DataPermission.DataPermissionValue dataPermissionValue = new UserModelPermission.DataPermission.DataPermissionValue();
        dataPermissionValue.setValueList(Arrays.asList(value));
        dataPermissionValue.setSelectType("standardSelect");
        biPermissionBean.setDataPermissionValues(Arrays.asList(dataPermissionValue));
        biPermissionBeans.add(biPermissionBean);
    }

    /**
     * 初步判断是否有访问权限
     *
     * @param request
     * @param userVo
     * @return
     */
    public boolean permissionAuth(HttpServletRequest request, UserInfo userVo) {
        String checkAuth = request.getParameter("checkAuth");
        String initDataPermission = request.getParameter("initDataPermission");
        String operationToken = getOperationToken(request);
        String path = request.getParameter("path");
        String dataAccessId = request.getParameter("dataAccessId");
        if ((!StringUtils.equalsIgnoreCase(checkAuth, "false") || !StringUtils.equalsIgnoreCase(initDataPermission, "false")) && !haveAccessAuthority(userVo, operationToken, path, dataAccessId)) {
            return false;
        }
        return true;
    }

    /**
     * 判断是否有访问权限
     *
     * @param tokenBody
     * @param operationToken
     * @param path
     * @param dataAccessId
     * @return
     */
    private boolean haveAccessAuthority(UserInfo tokenBody, String operationToken, String path, String dataAccessId) {
        String funcs = tokenBody.getFuc();
        if (StringUtils.isEmpty(operationToken) || funcs == null || funcs.isEmpty()) {
            return false;
        }
        String[] tokenParts = operationToken.split(":");
        if (tokenParts.length != 2) {
            return false;
        }
        String[] operations = funcs.split(",");
        String operationName = tokenParts[0];
        if (Stream.of(operations).noneMatch(opt -> operationName.equals(opt))) {
            return false;
        }
        String cdaName = StringUtils.substringAfterLast(path, "/");
        String apiName = StringUtils.replace(cdaName, "cda", dataAccessId);
        String[] apiSignatures = tokenParts[1].split("\\.");
        if (Stream.of(apiSignatures).noneMatch(sign -> validateOperationToken(operationName, apiName, tokenBody.getUsername(), sign))) {
            return false;
        }
        return true;
    }

    /**
     * 获取Operation Token
     *
     * @param request
     * @return
     */
    private String getOperationToken(HttpServletRequest request) {
        String operationToken = request.getHeader(OPERATION_TOKEN);
        if (operationToken == null) {
            operationToken = request.getParameter("operation_token");
        }
        return operationToken;
    }

    /**
     * 验证OperationToken
     *
     * @param operationName
     * @param apiName
     * @param operatorName
     * @param signature
     * @return
     */
    private boolean validateOperationToken(String operationName, String apiName, String operatorName, String signature) {
        try {
            /**
             * first validate the signature to avoid illegal expression execution
             */
            String[] signatureArray = signature.split("\\$\\$");
            String params = signatureArray.length == 2 ? signatureArray[0] : "";
            String realSignature = signatureArray.length == 2 ? signatureArray[1] : signatureArray[0];
            String decodedParams = StringUtils.isEmpty(params) ? "" : TextCodec.BASE64URL.decodeToString(params);
            String content = operationName + "-" + apiName + "$$" + decodedParams;
            String secret = securityKey + "." + operatorName;
            if (!validateHS256Signature(content, secret, realSignature)) {
                return false;
            }
        } catch (RuntimeException e) {
            return false;
        }

        return true;
    }

    /**
     * 验证签名
     *
     * @param content
     * @param secret
     * @param signature
     * @return
     */
    private boolean validateHS256Signature(String content, String secret, String signature) {
        SignatureValidatorFactory factory = DefaultSignatureValidatorFactory.INSTANCE;
        SecretKeySpec key = new SecretKeySpec(secret.getBytes(UTF_8), HS256.getJcaName());
        return factory.createSignatureValidator(HS256, key).isValid(content.getBytes(UTF_8), TextCodec.BASE64URL.decode(signature));
    }
}
