package com.xforceplus.bi.commons.authority.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.xforceplus.bi.commons.authority.service.TokenManagerService;
import com.xforceplus.bi.commons.integration.platform.AuthSource;
import com.xforceplus.bi.commons.integration.user.beans.UserInfo;
import com.xforceplus.bi.commons.integration.user.utils.RequestUserContext;
import com.xforceplus.tenantsecurity.domain.AuthorizedUser;
import com.xforceplus.tenantsecurity.domain.UserInfoHolder;
import com.xforceplus.tenantsecurity.utils.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RBucket;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.util.concurrent.TimeUnit;

@Slf4j
@Service
public class TokenManagerServiceImpl implements TokenManagerService {
    public static final String LOCK_KEY = "xforce_bi:access_token_lock:";
    public static final String TOKEN_KEY = "xforce_bi:access_token:";

    @Value("${xforce.auth.tokenTimeOut:3600}")
    private long tokenTimeOut;

    @Autowired
    private RedissonClient redissonClient;

    @Override
    public UserInfo getUser(String token) {
        RBucket bucket = redissonClient.getBucket(genAccessTokenPath(token));
        if (!bucket.isExists()) {
            return null;
        }
        UserInfo userInfo = JSONObject.parseObject((String) bucket.get(), UserInfo.class);
        RequestUserContext.set(userInfo);
        if (userInfo != null && userInfo.getAuthSource() == AuthSource.USER_CENTER) {
            AuthorizedUser authorizedUser = (AuthorizedUser) JsonUtils.fromJson(JSON.toJSONString(userInfo.getOrigin()), AuthorizedUser.class);
            if (authorizedUser != null) {
                log.info("存入AuthorizedUser到ThreadLocal");
                UserInfoHolder.put(authorizedUser);
            }else {
                log.info("AuthorizedUser is null");
            }
        }
        return userInfo;
    }

    /**
     * 用户在Redis,则刷新过期时间;用户不在Redis,则存放一份新的
     *
     * @param token
     * @param authorizedUserInfo
     */
    @Override
    public void putUserOrRefreshExpire(String token, UserInfo authorizedUserInfo) {
        putUserOrRefreshExpire(token, authorizedUserInfo,tokenTimeOut);
    }

    /**
     * 自定义过期时间存储用户信息
     *
     * @param token
     * @param authorizedUserInfo
     * @param timeSecond
     */
    @Override
    public void putUserOrRefreshExpire(String token, UserInfo authorizedUserInfo, long timeSecond) {
        RLock lock = redissonClient.getLock(LOCK_KEY + token);
        try {
            boolean locked = lock.tryLock(3, TimeUnit.SECONDS);
            RBucket bucket = redissonClient.getBucket(genAccessTokenPath(token));
            if (!bucket.isExists() && locked) {
                // 用户不存在于Redis,则存放一份新的
                String userStr = JSONObject.toJSONString(authorizedUserInfo);
                log.info("存Redis的用户信息:{}", userStr);
                bucket.set(userStr, timeSecond, TimeUnit.SECONDS);
                AuthSource authSource = authorizedUserInfo.getAuthSource();
                log.info("认证成功 => {}方式", authSource == null ? "未知" : authSource.getDescription());
            } else {
                // 用户存在于Redis,刷新过期时间
                bucket.expire(timeSecond, TimeUnit.SECONDS);
            }
        } catch (Exception e) {
            log.error("缓存token失败", e);
        } finally {
            lock.unlock();
        }
    }

    private String genAccessTokenPath(String accessToken) {
        return TOKEN_KEY.concat(accessToken);
    }

}
