package com.xforceplus.bi.commons.authority;

import com.google.common.collect.Maps;
import com.xforceplus.bi.commons.authority.anno.HasMenu;
import com.xforceplus.bi.commons.authority.anno.WithoutAuth;
import com.xforceplus.bi.commons.authority.encryptions.AuthEncryptionInterface;
import com.xforceplus.bi.commons.authority.service.TokenManagerService;
import com.xforceplus.bi.commons.authority.util.SpringContextUtil;
import com.xforceplus.bi.commons.integration.user.beans.UserInfo;
import com.xforceplus.bi.commons.integration.user.utils.BiTokenKey;
import com.xforceplus.bi.commons.integration.user.utils.RequestUserContext;
import com.xforceplus.bi.commons.jdk.util.UUIDUtil;
import com.xforceplus.bi.commons.webutils.RequestUrlUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.util.CollectionUtils;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.HandlerInterceptor;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

@Slf4j
@EnableConfigurationProperties(value = AuthExcludeConfig.class)
public class AuthInterceptor implements HandlerInterceptor, InitializingBean {

    private static List<AuthEncryptionInterface> authEncryptionInterfaces;

    @Autowired
    private SpringContextUtil springContextUtil;

    @Autowired
    private AuthExcludeConfig authExcludeConfig;

    @Autowired
    private TokenManagerService tokenManagerService;

    @Override
    public void afterPropertiesSet() throws Exception {
        Map<String, AuthEncryptionInterface> authEncryptionInterfaceMap =
                this.springContextUtil.getApplicationContext().getBeansOfType(AuthEncryptionInterface.class);
        authEncryptionInterfaces = new ArrayList<>(authEncryptionInterfaceMap.values());
        log.info("发现{}种认证方式", authEncryptionInterfaces.size());
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
        // 请求路径
        String requestURI = request.getRequestURI();

        // 静态资源不拦截
        if (RequestUrlUtils.isStaticPage(requestURI)) {
            return true;
        }

        // 例外路径不拦截(配置文件中配置规则)
        if (RequestUrlUtils.isOpenApi(requestURI, authExcludeConfig.getExclude())) {
            return true;
        }

        // 此路由对应的controller方法,有WithoutAuth注解,不拦截
        if (withoutAuth(handler)) {
            return true;
        }

//        debugCookies(request);

        // 获取用户信息
        UserInfo authorizedUserInfo = registerUserInfo(request);
        if (authorizedUserInfo == null) {
            noAuth(response, requestURI);
            return false;
        }

        //判断是否有菜单限制
        HasMenu hasMenu = ((HandlerMethod) handler).getMethodAnnotation(HasMenu.class);
        if (!hasMenu(hasMenu, authorizedUserInfo)) {
            noAuth(response, requestURI);
            return false;
        }

        return true;
    }

    private void debugCookies(HttpServletRequest request) {
        Cookie[] cookies = request.getCookies();
        if (ArrayUtils.isEmpty(cookies)) {
            log.info("无cookie");
            return;
        }

        String cookieNamesStr = "";
        log.info("----------------Cookie debug begin----------------");
        for (Cookie cookie : cookies) {
            cookieNamesStr += cookie.getName();
            cookieNamesStr += ",";
        }
        log.info(cookieNamesStr);
        log.info("----------------Cookie debug   end----------------");
    }

    /**
     * 注册用户
     *
     * @param request
     * @return
     */
    private UserInfo registerUserInfo(HttpServletRequest request) {
        UserInfo authorizedUserInfo = null;
        String token = null;
        String tokenKey = null;
        // 搜集所有认证方式的异常,都失败的情况下记录日志
        Map<String, Exception> exceptions = Maps.newHashMap();
        String batchId = UUIDUtil.gen();
        log.info("{}:获取用户信息------------------------start", batchId);
        for (AuthEncryptionInterface authEncryptionInterface : authEncryptionInterfaces) {
            // 先看请求头是否有token
            log.info("{}:获取用户信息方式:{}", batchId, authEncryptionInterface.getClass().getSimpleName());
            tokenKey = authEncryptionInterface.tokenKey();
            log.info("{}:获取用户信息Key:{}", batchId, tokenKey);
            token = authEncryptionInterface.token(request);
            if (StringUtils.isEmpty(token)) {
                // 取不到token,轮询一下个认证
                log.error("{}:{}获取用户信息Token为空", batchId, authEncryptionInterface.getClass().getSimpleName());
                continue;
            }

            // 再看Redis中是否有token
            authorizedUserInfo = tokenManagerService.getUser(token);
            if (authorizedUserInfo != null) {
                log.info("{}:{}获取用户信息缓存", batchId, authEncryptionInterface.getClass().getSimpleName());
                break;
            }

            //  缓存中没有,则是第一次授权,解析token
            try {
                authorizedUserInfo = authEncryptionInterface.decode(request);
                if (authorizedUserInfo != null) {
                    log.info("{}:认证成功,认证方式为:{}", batchId, authEncryptionInterface.getClass());
                    break;
                }
            } catch (Exception e) {
                log.info("{}:{}认证失败,认证方式为:{}", batchId, authEncryptionInterface.getClass(), e.getMessage());
                exceptions.put(authEncryptionInterface.getClass().getName(), e);
            }
        }
        log.info("{}:获取用户信息------------------------end", batchId);
        if (StringUtils.isEmpty(token) || authorizedUserInfo == null) {
            log.error("所有认证方式都失败,详细日志 ↓↓↓↓↓↓");
            for (Map.Entry<String, Exception> error : exceptions.entrySet()) {
                log.error(error.getKey(), error.getValue());
            }
            return null;
        }

        // 设置token的key名称
        BiTokenKey.set(request, tokenKey);
        log.info("{}:获取用户信息为:{}", batchId, authorizedUserInfo);
        // 用户存redis
        tokenManagerService.putUserOrRefreshExpire(token, authorizedUserInfo);
        authorizedUserInfo.setAccessToken(token);
        // 用户设到当前线程
        RequestUserContext.set(authorizedUserInfo);
        return authorizedUserInfo;
    }

    /**
     * 判断用户是否有菜单
     *
     * @param hasMenu
     * @param authorizedUserInfo
     * @return
     */
    private boolean hasMenu(HasMenu hasMenu, UserInfo authorizedUserInfo) {
        // 没加菜单验证,返回true
        if (hasMenu == null || ArrayUtils.isEmpty(hasMenu.value())) {
            return true;
        }

        // 用户没有菜单,则没有权限
        if (CollectionUtils.isEmpty(authorizedUserInfo.getMenus())) {
            return false;
        }

        // 用户菜单必须包含注解中所有的菜单
        for (int i = hasMenu.value().length - 1; i >= 0; i--) {
            String needPermission = hasMenu.value()[i];
            if (!authorizedUserInfo.getMenus().contains(needPermission)) {
                return false;
            }
        }

        return true;
    }

    /**
     * 验证是否不需要授权
     *
     * @param handler
     * @return
     */
    private boolean withoutAuth(Object handler) {
        HandlerMethod handlerMethod;
        if (handler instanceof HandlerMethod) {
            handlerMethod = (HandlerMethod) handler;
        } else {
            return true;
        }

        WithoutAuth withoutAuth = handlerMethod.getMethodAnnotation(WithoutAuth.class);
        if (withoutAuth != null) {
            return true;
        }

        return false;
    }

    private void noAuth(HttpServletResponse httpServletResponse, String requestURI) {
        log.warn("无访问权限, 访问路径:{}", requestURI);
        httpServletResponse.setStatus(HttpServletResponse.SC_UNAUTHORIZED);
    }

}
