package com.xforceplus.bi.commons.authority.encryptions.usercenter;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Sets;
import com.xforceplus.bi.commons.authority.encryptions.AuthEncryptionInterface;
import com.xforceplus.bi.commons.authority.usercenter.feign.client.UserExtraInfoClientService;
import com.xforceplus.bi.commons.authority.usercenter.feign.request.MsGetUserExtraInfoRequest;
import com.xforceplus.bi.commons.authority.usercenter.feign.response.ExtraInfoModel;
import com.xforceplus.bi.commons.authority.usercenter.feign.response.MsGetUserExtraInfoResponse;
import com.xforceplus.bi.commons.integration.platform.AuthSource;
import com.xforceplus.bi.commons.integration.user.beans.UserInfo;
import com.xforceplus.tenantsecurity.domain.AuthorizedUser;
import com.xforceplus.tenantsecurity.domain.Role;
import com.xforceplus.tenantsecurity.domain.UserType;
import com.xforceplus.tenantsecurity.jwt.JwtUtils;
import com.xforceplus.tenantsecurity.utils.CompressionUtils;
import com.xforceplus.tenantsecurity.utils.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.web.servlet.HandlerMapping;
import org.springframework.web.util.WebUtils;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.util.Map;
import java.util.Set;

import static com.xforceplus.bi.commons.authority.util.DetermineAuthUtil.containsXAccessTokenInHeaders;
import static com.xforceplus.bi.commons.authority.util.DetermineAuthUtil.isFromAthena;

@Slf4j
public class UserCenterAuthEncryption implements AuthEncryptionInterface {

    @Value("${xforce.platforms.usercenter.accessTokenKey:xforce-saas-token}")
    private String accessTokenKey = "";

    @Value("${xforce.platforms.usercenter.secret:my_sessionjw_tsecret_xdfdffdsdfdfs}")
    private String secret;

    @Value("${xforce.platforms.usercenter.appid:100}")
    private String appid;

    @Value("${xforce.platforms.usercenter.extra:true}")
    private boolean needExtraAuth = true;

    @Autowired
    private UserExtraInfoClientService userExtraInfoClientService;

    @Override
    public String tokenKey() {
        return accessTokenKey;
    }

    /**
     * 用户中心取token的逻辑(先从cookie取,没有就去header取,再没有就去参数里取)
     *
     * @param request
     * @return
     */
    @Override
    public String token(HttpServletRequest request) {
        if (isFromAthena(request) || containsXAccessTokenInHeaders(request)) {
            if (isFromAthena(request)){
                log.error("Cookie中包含Token");
            }else {
                log.error("Header中包含AccessToken");
            }
            return null;
        }
        Cookie tokenCookie = WebUtils.getCookie(request, UserType.USER.tokenKey());
        if (tokenCookie != null) {
            return tokenCookie.getValue();
        }
        String token = request.getHeader(accessTokenKey);
        if (StringUtils.isNotEmpty(token)) {
            return token;
        }
        return WebUtils.findParameterValue(request, UserType.USER.tokenKey());
    }

    @Override
    public UserInfo decode(HttpServletRequest request) throws Exception {
        // 1.获取token
        String token = token(request);
        // 2.校验和解析token
        Map<String, String> claims = JwtUtils.verifyAndDecodeToken(this.secret, token);
        Assert.notNull(claims, "claims == null, 访问失败，没有登录");

        // 3.实例化用户中心的UserInfo
        AuthorizedUser userOfUserCenter = getUserOfUserCenter(request, claims, token);

        // 4.转成我们BI的用户
        UserInfo retUser = transferToBIUser(userOfUserCenter);
        return retUser;
    }

    private UserInfo<AuthorizedUser> transferToBIUser(AuthorizedUser userOfUserCenter) {
        UserInfo userInfo = new UserInfo();
        userInfo.setAuthSource(AuthSource.USER_CENTER);
        userInfo.setUsername(userOfUserCenter.getUsername());
        userInfo.setTenantId(String.valueOf(userOfUserCenter.getTenantId()));
        userInfo.setTenantCode(userOfUserCenter.getTenantCode());
        userInfo.setMobile(userOfUserCenter.getMobile());
        userInfo.setEmail(userOfUserCenter.getEmail());
        // 设置角色
        Set<String> roles = Sets.newHashSet();
        if (!CollectionUtils.isEmpty(userOfUserCenter.getRoles())) {
            for (Role role : userOfUserCenter.getRoles()) {
                roles.add(String.valueOf(role.getId()));
            }
        } else {
            log.warn("该用户({})的角色为空", userOfUserCenter.getUsername());
        }
        userInfo.setRoles(roles);
        // 设置平台用户的原始值
        userInfo.setOrigin(userOfUserCenter);

        // 设置资源码
        userInfo.setResources(userOfUserCenter.getResourceCodes());
        return userInfo;
    }

    private AuthorizedUser getUserOfUserCenter(HttpServletRequest request, Map<String, String> claims, String token) throws IOException {
        String userinfo = CompressionUtils.decode(claims.get("userinfo"));
        Assert.hasText(userinfo, "访问失败，无效令牌");

        // token解析出来的只包含基础信息
        AuthorizedUser userInfo = JsonUtils.fromJson(userinfo, AuthorizedUser.class);
        Assert.notNull(userInfo, "userinfo is null");
        userInfo.setToken(token);

        // 设置集团ID
        Long tenantId = getTenantId(request);
        if (tenantId != null) {
            userInfo.setTenantId(tenantId);
        }

        // 设置额外信息(资源码)
        if (needExtraAuth){
            setExtraInfo(userInfo);
        }
        return userInfo;
    }

    /**
     * 设置用户信息
     *
     * @param userInfo 用户信息
     * @throws IOException
     */
    private void setExtraInfo(AuthorizedUser userInfo) throws IOException {
        MsGetUserExtraInfoRequest msGetUserExtraInfoRequest = new MsGetUserExtraInfoRequest();
        int appId = Integer.parseInt(this.appid);
        msGetUserExtraInfoRequest.setAppId(appId);
        msGetUserExtraInfoRequest.setResources(true);

        MsGetUserExtraInfoResponse msGetUserExtraInfoResponse = userExtraInfoClientService.userExtraInfo(userInfo.token(), msGetUserExtraInfoRequest);

        if (msGetUserExtraInfoResponse.getCode() != 1) {
            log.error("获取用户信息接口内部错误:" + msGetUserExtraInfoResponse.getMessage());
        }

        String infoJson = msGetUserExtraInfoResponse.getInfoJson();
        if (StringUtils.isEmpty(infoJson)) {
            log.info("获取用户额外信息返回为空");
            userInfo.setResourceCodes(Sets.newHashSet());
            return;
        }
        ExtraInfoModel extraInfoModel = JsonUtils.fromJson(infoJson, ExtraInfoModel.class);
        Set<String> resourceCodes = extraInfoModel.getResourceCodes();
        if (CollectionUtils.isEmpty(resourceCodes)){
            log.warn("infoJson:" + JSON.toJSONString(msGetUserExtraInfoResponse));
        }
        log.info("获取到的权限为: {}", resourceCodes);
        userInfo.setResourceCodes(resourceCodes);
    }

    /**
     * 用户中心取集团ID的逻辑
     *
     * @param request
     * @return
     */
    private Long getTenantId(HttpServletRequest request) {
        String tenantIdStr = request.getHeader("tenantId");
        if (tenantIdStr == null) {
            Map<String, String> pathVariables = (Map) request.getAttribute(HandlerMapping.URI_TEMPLATE_VARIABLES_ATTRIBUTE);
            if (!CollectionUtils.isEmpty(pathVariables)) {
                tenantIdStr = pathVariables.get("tenantId");
            }
        }

        Long tenantId;
        try {
            tenantId = Long.parseLong(tenantIdStr);
        } catch (NumberFormatException e) {
            log.warn(e.getMessage() + ", tenantId = " + tenantIdStr);
            tenantId = null;
        }
        return tenantId;
    }

}
