package com.xforceplus.bi.commons.authority.encryptions.usercenter;

import com.google.common.collect.Sets;
import com.xforceplus.bi.commons.authority.encryptions.AuthEncryptionInterface;
import com.xforceplus.bi.commons.authority.usercenter.feign.decoder.UserContextDecoder;
import com.xforceplus.bi.commons.authority.usercenter.feign.resourcecode.ResourceCodeClient;
import com.xforceplus.bi.commons.authority.usercenter.feign.resourcecode.ResourceCodeClientProvider;
import com.xforceplus.bi.commons.integration.platform.AuthSource;
import com.xforceplus.bi.commons.integration.user.beans.UserInfo;
import com.xforceplus.bi.commons.jdk.net.URLUtils;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.AuthorizedUser;
import com.xforceplus.tenant.security.token.domain.IRole;
import com.xforceplus.tenant.security.token.domain.UserType;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.util.CollectionUtils;
import org.springframework.web.servlet.HandlerMapping;
import org.springframework.web.util.WebUtils;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import java.util.Map;
import java.util.Set;

@Slf4j
public class UserCenterAuthEncryption implements AuthEncryptionInterface {
    @Autowired
    private ResourceCodeClientProvider resourceCodeClientProvider;

    /**
     * 资源码版本配置
     * XFORCE_PLATFORMS_USERCENTER_RESOURCECODE_VERSION: v1
     */
    @Value("${xforce.platforms.usercenter.resourcecode.version:v2}")
    private String resourceCodeVersion;

    @Value("${xforce.platforms.usercenter.accessTokenKey:xforce-saas-token}")
    private String accessTokenKey = "";

    @Autowired
    private UserContextDecoder userContextDecoder;

    @Override
    public String tokenKey() {
        return accessTokenKey;
    }

    /**
     * 用户中心取token的逻辑(先从cookie取,没有就去header取,再没有就去参数里取)
     *
     * @param request
     * @return
     */
    @Override
    public String token(HttpServletRequest request) {
//        if (isFromAthena(request) || containsXAccessTokenInHeaders(request)) {
//            if (isFromAthena(request)){
//                log.error("Cookie中包含Token");
//            }else {
//                log.error("Header中包含AccessToken");
//            }
//            return null;
//        }
        // Cookie
        Cookie tokenCookie = WebUtils.getCookie(request, UserType.USER.tokenKey());
        if (tokenCookie != null) {
            return tokenCookie.getValue();
        }
        // Header
        String tokenHeader = request.getHeader(accessTokenKey);
        if (StringUtils.isNotEmpty(tokenHeader)) {
            return tokenHeader;
        }
        // param
        String tokenParam = WebUtils.findParameterValue(request, UserType.USER.tokenKey());
        if (StringUtils.isNotEmpty(tokenParam)) {
            return tokenParam;
        }
        // referer
        try {
            Map<String, String> referer = URLUtils.params(request.getHeader("referer"));
            String token = referer.get("token");
            log.info("后端从head中的referer取到token");
            return token;
        } catch (Exception e) {
            log.error("分析用户中心referer token失败", e);
        }
        return null;
    }

    @Override
    public UserInfo decode(HttpServletRequest request) throws Exception {
        // 1.获取token
        String token = token(request);
        // 2.解析user
        AuthorizedUser authorizedUser = userContextDecoder.decode(token, getTenantId(request));
        ResourceCodeClient resourceCodeClient = resourceCodeClientProvider.getInstance(resourceCodeVersion);
        Set<String> resources = resourceCodeClient.fetchResources(String.valueOf(authorizedUser.getId()), token);
        log.debug("用户({})获取到资源码:{}", authorizedUser.getUsername(), resources);
        authorizedUser.setResourceCodes(resources);
        UserInfoHolder.put(authorizedUser);
        // 4.转成我们BI的用户
        UserInfo retUser = transferToBIUser(authorizedUser);
        return retUser;
    }

    private UserInfo<AuthorizedUser> transferToBIUser(AuthorizedUser userOfUserCenter) {
        UserInfo userInfo = new UserInfo();
        userInfo.setId(String.valueOf(userOfUserCenter.getId()));
        userInfo.setAuthSource(AuthSource.USER_CENTER);
        userInfo.setUsername(userOfUserCenter.getUsername());
        userInfo.setTenantId(String.valueOf(userOfUserCenter.getTenantId()));
        userInfo.setTenantCode(userOfUserCenter.getTenantCode());
        userInfo.setMobile(userOfUserCenter.getMobile());
        userInfo.setEmail(userOfUserCenter.getEmail());
        userInfo.setName(userOfUserCenter.getUsername());
        // 设置角色
        Set<String> roles = Sets.newHashSet();
        if (!CollectionUtils.isEmpty(userOfUserCenter.getRoles())) {
            for (Object role : userOfUserCenter.getRoles()) {
                roles.add(String.valueOf(((IRole) role).getId()));
            }
        } else {
            log.warn("该用户({})的角色为空", userOfUserCenter.getUsername());
        }
        userInfo.setRoles(roles);
        // 设置平台用户的原始值
        userInfo.setOrigin(userOfUserCenter);

        // 设置资源码
        userInfo.setResources(userOfUserCenter.getResourceCodes());
        return userInfo;
    }

    /**
     * 用户中心取集团ID的逻辑
     *
     * @param request
     * @return
     */
    private Long getTenantId(HttpServletRequest request) {
        String tenantIdStr = request.getHeader("tenantId");
        if (tenantIdStr == null) {
            Map<String, String> pathVariables = (Map) request.getAttribute(HandlerMapping.URI_TEMPLATE_VARIABLES_ATTRIBUTE);
            if (!CollectionUtils.isEmpty(pathVariables)) {
                tenantIdStr = pathVariables.get("tenantId");
            }
        }

        Long tenantId;
        try {
            tenantId = Long.parseLong(tenantIdStr);
        } catch (NumberFormatException e) {
            log.warn(e.getMessage() + ", tenantId = " + tenantIdStr);
            tenantId = null;
        }
        return tenantId;
    }

}
