package com.xforceplus.bi.commons.sql.generator;

import com.google.common.collect.Lists;
import com.xforceplus.bi.commons.sql.generator.fields.FieldMeta;
import com.xforceplus.bi.commons.sql.generator.fields.FieldType;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.jooq.Constraint;
import org.jooq.CreateTableConstraintStep;
import org.jooq.DataType;
import org.jooq.Field;
import org.jooq.impl.SQLDataType;
import org.springframework.stereotype.Service;

import java.util.List;

import static com.xforceplus.bi.commons.sql.core.JooqUtils.getDSLContext;
import static org.jooq.impl.DSL.constraint;
import static org.jooq.impl.DSL.field;

@Slf4j
@Service
public class SQLGenDDLServiceImpl implements SQLGenDDLService {

    @Override
    public String tableCreateSQL(String dialect,
                                 String tableName,
                                 List<FieldMeta> fieldMetas,
                                 List<String[]> uniqueKeyGroups) {
        // 定义列
        CreateTableConstraintStep finalStep = getDSLContext(dialect)
                .createTable(tableName)
                .columns(fields(fieldMetas));

        // 定义约束
        List<Constraint> constraints = Lists.newArrayList();

        // 主键约束
        String[] primaryKeys = getPrimaryKeys(fieldMetas);
        if (ArrayUtils.isNotEmpty(primaryKeys)) {
            constraints.add(constraint("pk_" + tableName).primaryKey(primaryKeys));
        }

        // 唯一性约束
        if (CollectionUtils.isNotEmpty(uniqueKeyGroups)) {
            for (String[] uniqueKeyGroup : uniqueKeyGroups) {
                if (ArrayUtils.isEmpty(uniqueKeyGroup)) {
                    continue;
                }
                String uniqueKey = genUniqueKeyName(tableName, uniqueKeyGroup);
                constraints.add(constraint(uniqueKey).unique(uniqueKeyGroup));
            }
        }

        // 有约束则追加
        if (CollectionUtils.isNotEmpty(constraints)) {
            finalStep = finalStep.constraints(constraints);
        }

        return finalStep.getSQL();
    }

    /**
     * 生成唯一键的名称
     *
     * @param tableName
     * @param uniqueKeyGroup
     * @return
     */
    private String genUniqueKeyName(String tableName, String[] uniqueKeyGroup) {
        String uniqueKey = "unique_" + tableName + "_";
        for (String realName : uniqueKeyGroup) {
            uniqueKey += realName + "_";
        }
        uniqueKey = uniqueKey.substring(0, uniqueKey.length() - 1);
        return uniqueKey;
    }

    /**
     * 获取主键字段数组(支持联合主键)
     *
     * @param fieldMetas
     * @return
     */
    private String[] getPrimaryKeys(List<FieldMeta> fieldMetas) {
        List<String> primaryKeys = Lists.newArrayList();
        for (FieldMeta field : fieldMetas) {
            if (field.isPrimaryFlag()) {
                primaryKeys.add(field.getFieldName());
            }
        }
        return primaryKeys.toArray(new String[primaryKeys.size()]);
    }

    /**
     * 定义字段
     *
     * @param fieldMetas
     * @return
     */
    private List<Field<String>> fields(List<FieldMeta> fieldMetas) {
        List<Field<String>> retFields = Lists.newArrayList();
        for (FieldMeta fieldBuilder : fieldMetas) {
            retFields.add(getJooqField(fieldBuilder));
        }
        return retFields;
    }

    /**
     * 获取Jooq字段
     *
     * @param fieldMeta
     * @return
     */
    private Field getJooqField(FieldMeta fieldMeta) {
        DataType dataType = getJooqDataType(fieldMeta.getFieldType(), fieldMeta.getDataFormat());

        // 设置默认值
        if (fieldMeta.getDefaultValue() != null) {
            dataType = dataType.defaultValue(fieldMeta.getDefaultValue());
        }

        // 自增长
        if (fieldMeta.isAutoIncrement()) {
            dataType = dataType.identity(true);
        }

        return field(fieldMeta.getFieldName(), dataType);
    }

    /**
     * 获取字段类型
     *
     * @param fieldType
     * @param dataFormat
     * @return
     */
    private DataType getJooqDataType(FieldType fieldType, String dataFormat) {
        switch (fieldType) {
            case LONG:
                return SQLDataType.BIGINT;
            case NUMERIC:
                return SQLDataType.NUMERIC(18, Integer.parseInt(dataFormat));
            case TIMESTAMP:
                return SQLDataType.TIMESTAMP; // 3  6
            case STRING_ARRAY:
                return SQLDataType.CLOB.getArrayDataType();
            case BOOLEAN:
                return SQLDataType.BOOLEAN;
            default:
                int varcharLength;
                try {
                    varcharLength = Integer.parseInt(dataFormat);
                } catch (Exception e) {
                    varcharLength = 255;
                    log.warn("数据格式为空, 转为字符串默认长度为{}", varcharLength, e);
                }
                return SQLDataType.VARCHAR.length(varcharLength);
        }
//
//        if (fieldType.equals("number")) {
//            // 如果是number类型,那么数据格式就表示精度
//            return StringUtils.isEmpty(dataFormat) || "0".equals(dataFormat) ?
//                    SQLDataType.BIGINT :
//                    SQLDataType.NUMERIC(18, Integer.parseInt(dataFormat));
//        } else if (fieldType.equals("date")) {
//            // 日期类型都用时间戳存储
//            return SQLDataType.TIMESTAMP; // 3  6
//        }
//        // 如果是string类型,那么数据格式就表示字符串长度
//        int varcharLength = Integer.parseInt(dataFormat);
//        return SQLDataType.VARCHAR.length(varcharLength);
    }

    @Override
    public String columnAddSQL(String dialect, String tableName, FieldMeta fieldMeta) {
        return getDSLContext(dialect)
                .alterTable(tableName)
                .addColumn(getJooqField(fieldMeta))
                .getSQL();
    }

    @Override
    public String columnDropSQL(String dialect, String tableName, String columnName) {
        return getDSLContext(dialect)
                .alterTable(tableName)
                .dropColumn(columnName)
                .getSQL();
    }

    @Override
    public String columnNameModifySQL(String dialect, String tableName, String oldColumnName, String newColumnName) {
        return getDSLContext(dialect)
                .alterTable(tableName)
                .renameColumn(oldColumnName)
                .to(newColumnName)
                .getSQL();
    }

    @Override
    public String columnModifySQL(String dialect, String tableName, FieldMeta fieldMeta) {
        return getDSLContext(dialect)
                .alterTable(tableName)
                .alterColumn(fieldMeta.getFieldName())
                .set(getJooqDataType(fieldMeta.getFieldType(), fieldMeta.getDataFormat()))
                .getSQL();
    }

    @Override
    public String tableDropSQL(String dialect, String tableName) {
        return getDSLContext(dialect)
                .dropTable(tableName)
                .getSQL();
    }

    @Override
    public String tableCommentSQL(String dialect, String tableName, String tableComment) {
        return getDSLContext(dialect)
                .commentOnTable(tableName)
                .is(tableComment)
                .getSQL();
    }

    @Override
    public String columnCommentSQL(String dialect, String tableName, String fieldName, String fieldComment) {
        return getDSLContext(dialect)
                .commentOnColumn(field(tableName + "." + fieldName))
                .is(fieldComment)
                .getSQL();
    }
}
