package com.xforceplus.bi.commons.sql.generator;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.xforceplus.bi.commons.sql.generator.conditions.SQLCondition;
import com.xforceplus.bi.commons.sql.generator.conditions.SQLConditionOperator;
import com.xforceplus.bi.commons.sql.generator.fields.FieldValue;
import org.jooq.*;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Map;

import static com.xforceplus.bi.commons.sql.core.JooqUtils.getDSLContext;
import static org.jooq.impl.DSL.field;
import static org.jooq.impl.DSL.table;

@Service
public class SQLGenDMLServiceImpl implements SQLGenDMLService {
    @Override
    public String recordExistsSQL(String dialect, String tableName, List<SQLCondition> sqlConditions) {
        SelectJoinStep<Record1<Object>> selectJoinStep = getDSLContext(dialect)
                .select(field("1"))
                .from(tableName);

        if (CollectionUtils.isEmpty(sqlConditions)) {
            return selectJoinStep.getSQL();
        }

        // 条件
        List<org.jooq.Condition> conditions = Lists.newArrayList();
        for (SQLCondition sqlCondition : sqlConditions) {
            conditions.add(getCondition(sqlCondition));
        }

        return selectJoinStep
                .where(conditions)
                .getSQL();
    }

    @Override
    public String recordInsertSQL(String dialect, String tableName, List<FieldValue> fieldValues) {
        Assert.notEmpty(fieldValues, "要更新的字段不能为空");

        // 字段
        List<Field<Object>> columns = Lists.newArrayList();
        // 值
        List<Object> values = Lists.newArrayList();

        for (FieldValue fieldValueBuilder : fieldValues) {
            columns.add(field(fieldValueBuilder.getFieldName()));
            values.add(fieldValueBuilder.getValue());
        }

        return getDSLContext(dialect)
                .insertInto(table(tableName))
                .columns(columns)
                .values(values)
                .getSQL();
    }

    @Override
    public String recordUpdateSQL(String dialect, String tableName, List<FieldValue> fieldValues, List<SQLCondition> sqlConditions) {
        Assert.notEmpty(fieldValues, "要更新的字段不能为空");

        Map<String, Object> valMap = Maps.newHashMap();
        for (FieldValue fieldValue : fieldValues) {
            valMap.put(fieldValue.getFieldName(), fieldValue.getValue());
        }

        // 条件
        List<org.jooq.Condition> conditions = Lists.newArrayList();
        for (SQLCondition sqlCondition : sqlConditions) {
            conditions.add(getCondition(sqlCondition));
        }

        UpdateSetFirstStep<Record> updateSetFirstStep = getDSLContext(dialect).update(table(tableName));
        UpdateSetMoreStep<Record> updateSetMoreStep = null;
        for (FieldValue fieldValue : fieldValues) {
            updateSetMoreStep = updateSetFirstStep.set(field(fieldValue.getFieldName()), fieldValue.getValue());
        }

        return updateSetMoreStep
                .where(conditions)
                .getSQL();
    }

    private org.jooq.Condition getCondition(SQLCondition conditionBuilder) {
        Field<Object> field = field(conditionBuilder.getFieldName());

        if (SQLConditionOperator.equal.equals(conditionBuilder.getConditionOperator())) {
            return field.eq(conditionBuilder.getFieldValue());
        }
        if (SQLConditionOperator.notEqual.equals(conditionBuilder.getConditionOperator())) {
            return field.notEqual(conditionBuilder.getFieldValue());
        }
        throw new RuntimeException("暂不支持的类型:" + conditionBuilder.getConditionOperator());
    }
}
