package com.xforceplus.bi.commons.sql.parser.service.impl;

import com.google.common.collect.Lists;
import com.xforceplus.bi.commons.sql.parser.beans.*;
import com.xforceplus.bi.commons.sql.parser.service.SqlBuilderService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.jooq.*;
import org.jooq.conf.ParamType;
import org.jooq.conf.Settings;
import org.jooq.conf.StatementType;
import org.jooq.impl.DSL;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Objects;

import static org.jooq.impl.DSL.field;

@Service
@Slf4j
public class SqlBuilderServiceImpl implements SqlBuilderService {

    private static final String COUNT_SQL = "SELECT COUNT(1) FROM (%s) AS FOO";

    private SelectLimitStep getSelectOptionStep(DataBaseBean dataBaseBean,
                                                String from,
                                                List<QuerySelectBean> querySelectBeans,
                                                List<QueryJoinBean> queryJoinBeans,
                                                List<String> queryGroupBeans,
                                                List<QueryFilterGroupBean> queryFilterBeans,
                                                List<String> notNullFields,
                                                List<QueryOrderBean> queryOrderBeans) {
        SelectSelectStep selectSelectStep = null;
        // 添加Select语句
        selectSelectStep = selectFields(dataBaseBean, querySelectBeans);
        // 添加From语句
        SelectJoinStep selectJoinStep = selectSelectStep.from(from);
        // 添加Join语句
        SelectWhereStep selectWhereStep = selectJoinStep(selectJoinStep, queryJoinBeans);
        // 添加Where语句
        SelectGroupByStep selectGroupByStep = selectWhereSetp(selectWhereStep, queryFilterBeans, notNullFields);
        // 添加Group语句
        SelectOrderByStep selectOrderByStep = selectGroupSetp(selectGroupByStep, queryGroupBeans);
        // 添加Order语句
        return selectOrderSetp(selectOrderByStep, queryOrderBeans);
    }

    @Override
    public String sqlBuilder(String from,
                             List<QuerySelectBean> querySelectBeans,
                             List<QueryJoinBean> queryJoinBeans,
                             List<String> queryGroupBeans,
                             List<QueryFilterGroupBean> queryFilterBeans,
                             List<String> notNullFields,
                             List<QueryOrderBean> queryOrderBeans,
                             QueryPageBean queryPageBean, DataBaseBean dataBaseBean) {
        SelectLimitStep selectLimitStep =
                getSelectOptionStep(dataBaseBean,
                        from,
                        querySelectBeans,
                        queryJoinBeans,
                        queryGroupBeans,
                        queryFilterBeans,
                        notNullFields,
                        queryOrderBeans);
        // 添加LImit
        SelectOptionStep selectOptionStep = selectLimitStep(selectLimitStep, queryPageBean);
        return selectOptionStep.getSQL(ParamType.INLINED);
    }

    private SelectOptionStep selectLimitStep(
            SelectLimitStep selectLimitStep, QueryPageBean queryPageBean) {
        if (queryPageBean != null) {
            int offset = queryPageBean.getPageSize() * queryPageBean.getPageNum() - queryPageBean.getPageSize();
            offset = offset > 0 ? offset : 0;
            return selectLimitStep.limit(queryPageBean.getPageSize()).offset(offset);
        }
        return selectLimitStep;
    }

    private SelectLimitStep selectOrderSetp(
            SelectOrderByStep selectOrderByStep, List<QueryOrderBean> queryOrderBeans) {
        if (CollectionUtils.isEmpty(queryOrderBeans)) {
            return selectOrderByStep;
        }
        List<SortField> orderFields = Lists.newArrayList();
        for (QueryOrderBean queryOrderBean : queryOrderBeans) {
            SortField sortField =
                    "asc".equalsIgnoreCase(queryOrderBean.getType())
                            ? field(queryOrderBean.getIndex()).asc()
                            : field(queryOrderBean.getIndex()).desc();
            orderFields.add(sortField);
        }
        return selectOrderByStep.orderBy(orderFields);
    }

    private SelectOrderByStep selectGroupSetp(
            SelectGroupByStep selectGroupByStep, List<String> queryGroupBeans) {
        if (CollectionUtils.isEmpty(queryGroupBeans)) {
            return selectGroupByStep;
        }
        List<Field> groupField = Lists.newArrayList();
        for (String queryGroupBean : queryGroupBeans) {
            groupField.add(field(queryGroupBean));
        }
        return selectGroupByStep.groupBy(groupField);
    }

    private SelectGroupByStep selectWhereSetp(
            SelectWhereStep selectWhereStep,
            List<QueryFilterGroupBean> queryFilterBeans,
            List<String> notNullFields) {

        List<Condition> conditionList = getAllCondition(queryFilterBeans);
        // 排除为空字段
        if (!CollectionUtils.isEmpty(notNullFields)) {
            notNullFields.forEach(item -> {
                conditionList.add(field(item).isNotNull());
            });
        }
        if (CollectionUtils.isEmpty(conditionList)) {
            return selectWhereStep;
        }
        return selectWhereStep.where(conditionList);
    }

    private List<Condition> getAllCondition(List<QueryFilterGroupBean> queryFilterBeans) {
        List<Condition> conditionList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(queryFilterBeans)) {
            return conditionList;
        }
        for (QueryFilterGroupBean filterGroupBean : queryFilterBeans) {
            List<QueryFilterBean> queryFilterBeans1 = filterGroupBean.getQueryFilterBeans();
            if (CollectionUtils.isEmpty(queryFilterBeans1)) {
                continue;
            }
            Condition condition = getCondition(queryFilterBeans1);
            conditionList.add(condition);
        }
        return conditionList;
    }

    private Condition getCondition(List<QueryFilterBean> queryFilterBeans) {
        Condition condition = null;
        for (QueryFilterBean queryFilterBean : queryFilterBeans) {
            Field field = field(queryFilterBean.getField());
            Condition conditionTemp = null;
            if (!CollectionUtils.isEmpty(queryFilterBean.getValues())) {
                conditionTemp =
                        queryFilterBean.getValues().size() == 1
                                ? field.equal(queryFilterBean.getValues().get(0))
                                : field.in(queryFilterBean.getValues());
            }
            if (!CollectionUtils.isEmpty(queryFilterBean.getRanges())) {

                for (QueryFilterRangeBean item : queryFilterBean.getRanges()) {
                    Condition filterCondition = null;
                    // 有开始
                    if (notEmpty(item.getStart())) {
                        filterCondition = field.greaterOrEqual(item.getStart());
                    }
                    // 有结束
                    if (notEmpty(item.getEnd())) {
                        filterCondition = (filterCondition == null) ?
                                field.lessOrEqual(item.getEnd()) :
                                filterCondition.and(field.lessOrEqual(item.getEnd()));
                    }
                    if (conditionTemp == null) {
                        conditionTemp = filterCondition;
                    } else if (filterCondition != null) {
                        // 多个条件用OR连接
                        conditionTemp = conditionTemp.or(filterCondition);
                    }
                }
            }
            if (StringUtils.isNotBlank(queryFilterBean.getLikeValue())){
                Condition filterCondition = field.like(queryFilterBean.getLikeValue());
                if (conditionTemp == null) {
                    conditionTemp = filterCondition;
                } else if (filterCondition != null) {
                    // 多个条件用OR连接
                    conditionTemp = conditionTemp.or(filterCondition);
                }
            }
            if (CollectionUtils.isEmpty(queryFilterBean.getValues()) && CollectionUtils.isEmpty(queryFilterBean.getRanges())&&StringUtils.isBlank(queryFilterBean.getLikeValue()) ) {
                conditionTemp = field.isNull();
            }
            if (condition == null) {
                condition = conditionTemp;
            } else {
                condition = condition.or(conditionTemp);
            }
        }
        return condition;
    }

    private boolean notEmpty(Object obj) {
        return obj != null && !"".equals(obj);
    }

    private SelectWhereStep selectJoinStep(SelectJoinStep selectJoinStep, List<QueryJoinBean> list) {
        if (CollectionUtils.isEmpty(list)) {
            return selectJoinStep;
        }
        for (QueryJoinBean queryJoinBean : list) {
            switch (queryJoinBean.getType().toLowerCase()) {
                case "left join": {
                    selectJoinStep =
                            selectJoinStep
                                    .leftJoin(queryJoinBean.getTable())
                                    .on(condition(queryJoinBean.getConditions()));
                }
                break;
                case "right join": {
                    selectJoinStep =
                            selectJoinStep
                                    .rightJoin(queryJoinBean.getTable())
                                    .on(condition(queryJoinBean.getConditions()));
                }
                break;
                case "inner join": {
                    selectJoinStep =
                            selectJoinStep
                                    .innerJoin(queryJoinBean.getTable())
                                    .on(condition(queryJoinBean.getConditions()));
                }
                break;
            }
        }
        return selectJoinStep;
    }

    private Condition condition(List<QueryJoinBean.Condition> conditionList) {
        Condition condition = null;
        for (QueryJoinBean.Condition condition1 : conditionList) {
            Condition conditionTemp =
                    field(condition1.getJoinField()).eq(field(condition1.getFromField()));
            condition = condition == null ? conditionTemp : condition.and(conditionTemp);
        }
        return condition;
    }

    private SelectSelectStep<Record> selectFields(DataBaseBean dataBaseBean, List<QuerySelectBean> querySelectBeans) {
        List<SelectField> selectFields = Lists.newArrayList();
        for (QuerySelectBean item : querySelectBeans) {
            Field field = field(item.getField());
            if (StringUtils.isNotBlank(item.getAlias())) {
                field = field.as(item.getAlias());
            }
            selectFields.add(field);
        }
        DSLContext dslContext = getDslContext(dataBaseBean);
        return dslContext.select(selectFields);
    }

    private DSLContext getDslContext(DataBaseBean dataBaseBean) {
        Settings settings = new Settings();
        settings.setParamType(ParamType.INLINED);
        settings.setStatementType(StatementType.STATIC_STATEMENT);

        String jdbcDbDialect = dataBaseBean.getJdbcDbDialect();

        SQLDialect sqlDialect = null;
        if (StringUtils.isNotEmpty(jdbcDbDialect)) {
            try {
                sqlDialect = SQLDialect.valueOf(jdbcDbDialect);
            } catch (Exception e) {
                log.warn("未知的方言({}), 使用默认方言", jdbcDbDialect);
            }
        }
        if (sqlDialect == null) {
            sqlDialect = SQLDialect.DEFAULT;
        }
        return DSL.using(sqlDialect, settings);
    }

    @Override
    public String dataSql(
            String from,
            List<QuerySelectBean> querySelectBeans,
            List<QueryJoinBean> queryJoinBeans,
            List<String> queryGroupBeans,
            List<QueryFilterGroupBean> queryFilterBeans,
            List<String> notNullFields,
            List<QueryOrderBean> queryOrderBeans,
            QueryPageBean queryPageBean,
            DataBaseBean dataBaseBean) {
        return sqlBuilder(from, querySelectBeans, queryJoinBeans, queryGroupBeans, queryFilterBeans, notNullFields, queryOrderBeans, queryPageBean, dataBaseBean);
    }

    @Override
    public String countSql(
            String from,
            List<QuerySelectBean> querySelectBeans,
            List<QueryJoinBean> queryJoinBeans,
            List<String> queryGroupBeans,
            List<QueryFilterGroupBean> queryFilterBeans,
            List<String> notNullFields,
            List<QueryOrderBean> queryOrderBeans,
            DataBaseBean dataBaseBean) {
        SelectLimitStep selectLimitStep =
                getSelectOptionStep(dataBaseBean,
                        from,
                        querySelectBeans,
                        queryJoinBeans,
                        queryGroupBeans,
                        queryFilterBeans,
                        notNullFields,
                        queryOrderBeans);
        return String.format(COUNT_SQL, selectLimitStep.getSQL());
    }

//    @Override
//    public String countSql(
//            String from,
//            List<QuerySelectBean> querySelectBeans,
//            List<QueryJoinBean> queryJoinBeans,
//            List<String> queryGroupBeans,
//            List<QueryFilterGroupBean> queryFilterBeans,
//            List<String> notNullFields,
//            List<QueryOrderBean> queryOrderBeans,
//            DataBaseBean dataBaseBean) {
//        // 添加Select语句 (SELECT COUNT)
//        SelectSelectStep selectSelectStep = getDslContext(dataBaseBean).select(field("1"));
//        // 添加From语句
//        SelectJoinStep selectJoinStep = selectSelectStep.from(from);
//        // 添加Join语句
//        SelectWhereStep selectWhereStep = selectJoinStep(selectJoinStep, queryJoinBeans);
//        // 添加Where语句
//        SelectGroupByStep selectGroupByStep = selectWhereSetp(selectWhereStep, queryFilterBeans, notNullFields);
//        // 添加Group语句
//        SelectOrderByStep selectOrderByStep = selectGroupSetp(selectGroupByStep, queryGroupBeans);
//        String sql = selectOrderByStep.getSQL();
//        return String.format(COUNT_SQL, sql);
//    }

    @Override
    public String summarySql(
            String from,
            List<QuerySelectBean> querySelectBeans,
            List<QueryJoinBean> queryJoinBeans,
            List<String> queryGroupBeans,
            List<QueryFilterGroupBean> queryFilterBeans,
            List<String> notNullFields,
            List<QueryOrderBean> queryOrderBeans,
            DataBaseBean dataBaseBean) {
        return sqlBuilder(from, querySelectBeans, queryJoinBeans, queryGroupBeans, queryFilterBeans, notNullFields, queryOrderBeans, null, dataBaseBean);
    }

}
