package com.xforceplus.delivery.cloud.monomer.limiter;

import cn.hutool.core.collection.CollectionUtil;
import com.google.common.base.Joiner;
import com.google.common.util.concurrent.RateLimiter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;
import org.springframework.web.servlet.mvc.condition.PatternsRequestCondition;
import org.springframework.web.servlet.mvc.method.RequestMappingInfo;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;
import org.springframework.web.util.UrlPathHelper;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

@Slf4j
@Component
@ConfigurationProperties("delivery.cloud.monomer.rate-limiter")
@ConditionalOnProperty(prefix = "delivery.cloud.monomer.rate-limiter", name = "enabled", havingValue = "true", matchIfMissing = true)
public class SmoothBurstyLimiterInterceptor extends HandlerInterceptorAdapter implements BeanPostProcessor {

    /**
     * QPS
     */
    @Setter
    private Integer permitsPerSecond = 100;

    /**
     * 限流的 URL与QPS的K/V 值
     * <p>
     * /creditcloud/test=100
     */
    @Setter
    private Map<String, Integer> urlPatternMap;

    private UrlPathHelper urlPathHelper = new UrlPathHelper();

    private Map<PatternsRequestCondition, RateLimiter> urlRateMap;

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
        if (this.urlRateMap != null && !this.urlRateMap.isEmpty()) {
            String lookupPath = this.urlPathHelper.getLookupPathForRequest(request);
            for (PatternsRequestCondition patternsRequestCondition : this.urlRateMap.keySet()) {
                //使用spring DispatcherServlet的匹配器PatternsRequestCondition进行匹配
                List<String> matchingPatterns = patternsRequestCondition.getMatchingPatterns(lookupPath);
                if (CollectionUtil.isEmpty(matchingPatterns)) {
                    continue;
                }
                log.trace("请求{}执行tryAcquire({})", lookupPath, Joiner.on(",").join(patternsRequestCondition.getPatterns()));
                if (this.urlRateMap.get(patternsRequestCondition).tryAcquire()) {
                    log.trace("请求{}匹配到限流器令牌", lookupPath);
                } else {
                    log.trace("请求{}超过限流器速率", lookupPath);
                    response.setStatus(HttpStatus.ACCEPTED.value());
                    return false;
                }
            }
        }
        return true;
    }

    @Override
    public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
        return bean;
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        if (RequestMappingHandlerMapping.class.isAssignableFrom(bean.getClass())) {
            if (this.urlRateMap == null) {
                this.urlRateMap = new ConcurrentHashMap<>();
            }
            log.info("we get all the controllers's methods and assign it to urlRateMap");
            RequestMappingHandlerMapping requestMappingHandlerMapping = (RequestMappingHandlerMapping) bean;
            Map<RequestMappingInfo, HandlerMethod> handlerMethods = requestMappingHandlerMapping.getHandlerMethods();
            for (RequestMappingInfo rmi : handlerMethods.keySet()) {
                // 默认的 url 限流方案设定
                this.urlRateMap.put(rmi.getPatternsCondition(), RateLimiter.create(this.permitsPerSecond));
            }
            // 自定义的限流方案设定
            if (this.urlPatternMap != null) {
                this.urlPatternMap.forEach((urlPattern, permitsPerSecond) ->
                        this.urlRateMap.put(new PatternsRequestCondition(urlPattern), RateLimiter.create(permitsPerSecond))
                );
            }
        }
        return bean;
    }

}
