package com.xforceplus.delivery.cloud.tax.pur.verify.service.impl;

import cn.hutool.core.lang.UUID;
import com.xforceplus.apollo.utils.JacksonUtil;
import com.xforceplus.core.common.constan.QueueNames;
import com.xforceplus.core.common.constan.VerifyStatusEnum;
import com.xforceplus.core.common.domain.JsonResult;
import com.xforceplus.core.common.domain.SealedRecMessage;
import com.xforceplus.core.remote.XPurchaserInvoiceService;
import com.xforceplus.core.remote.domain.VerifyRequest;
import com.xforceplus.delivery.cloud.common.api.AjaxResult;
import com.xforceplus.delivery.cloud.common.api.ViewResult;
import com.xforceplus.delivery.cloud.common.util.*;
import com.xforceplus.delivery.cloud.gen.purchaser.entity.PurchaserInvoiceDetailsEntity;
import com.xforceplus.delivery.cloud.gen.purchaser.entity.PurchaserInvoiceMainEntity;
import com.xforceplus.delivery.cloud.gen.purchaser.entity.PurchaserVerifyRecordEntity;
import com.xforceplus.delivery.cloud.gen.purchaser.service.IPurchaserInvoiceDetailsService;
import com.xforceplus.delivery.cloud.gen.purchaser.service.IPurchaserInvoiceMainService;
import com.xforceplus.delivery.cloud.gen.purchaser.service.IPurchaserVerifyRecordService;
import com.xforceplus.delivery.cloud.secure.component.OAuth2UserContext;
import com.xforceplus.delivery.cloud.tax.api.janus.JanusCoreReceiveMsg;
import com.xforceplus.delivery.cloud.tax.api.janus.SealedRecMessageHandler;
import com.xforceplus.delivery.cloud.tax.pur.verify.domain.VerifyFeedback;
import com.xforceplus.delivery.cloud.tax.pur.verify.domain.VerifyMain;
import com.xforceplus.delivery.cloud.tax.pur.verify.domain.VerifyParamVo;
import com.xforceplus.delivery.cloud.tax.pur.verify.domain.VerifyQueryResult;
import com.xforceplus.delivery.cloud.tax.pur.verify.service.IPurchaserInvoiceVerifyService;
import io.jsonwebtoken.lang.Collections;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * <p>
 * 服务实现类
 * </p>
 *
 * @author Hanyongjie
 * @since 2020-06-19
 */
@Slf4j
@Service
@com.xforceplus.delivery.cloud.tax.api.janus.SealedRecMessage(QueueNames.VERIFY_RESULT_BACK)
public class PurchaserInvoiceVerifyServiceImpl implements IPurchaserInvoiceVerifyService, SealedRecMessageHandler<VerifyFeedback> {

    @Autowired
    private XPurchaserInvoiceService xPurchaserInvoiceService;

    @Autowired
    private IPurchaserInvoiceMainService iPurchaserInvoiceMainService;

    @Autowired
    private IPurchaserInvoiceDetailsService iPurchaserInvoiceDetailsService;

    @Autowired
    private IPurchaserVerifyRecordService iPurchaserVerifyRecordService;

    @Override
    public ViewResult<Object> verify(VerifyParamVo verifyParam) {
        if (StringUtils.isBlank(verifyParam.getSerialNo())) {
            verifyParam.setSerialNo(UUID.randomUUID().toString());
        }
        LocalDate paperDrewDate = DateUtils.toLocalDate(verifyParam.getPaperDrewDate());
        verifyParam.setPaperDrewDate(DateUtils.format(paperDrewDate, "yyyyMMdd"));
        PurchaserVerifyRecordEntity target = new PurchaserVerifyRecordEntity();
        BeanUtils.copyObject(verifyParam, target);
        target.setPaperDrewDate(paperDrewDate);
        target.setUserName(OAuth2UserContext.getPrincipal().getUsername());
        try {
            JsonResult<?> jResult = xPurchaserInvoiceService.invoiceVerify(verifyParam);
            if (jResult.isSuccess()) {
                target.setTaskId(jResult.getData().toString());
                target.setVerifyStatus(VerifyStatusEnum.VERIFY_ING.getCode());
                return ViewResult.success();
            } else {
                target.setVerifyStatus(VerifyStatusEnum.VERIFY_FAIL.getCode());
                target.setVerifyMessage(jResult.getMessage());
                return ViewResult.failed(jResult.getMessage());
            }
        } catch (Exception e) {
            log.error("", e);
            target.setVerifyStatus(VerifyStatusEnum.VERIFY_FAIL.getCode());
            target.setVerifyMessage("系统异常，验真失败，原因:" + e.getMessage());
            return ViewResult.failed(target.getVerifyMessage());
        } finally {
            this.iPurchaserVerifyRecordService.saveOrUpdate(target);
        }
    }

    @Override
    public ViewResult<Object> verify4(VerifyRequest verifyRequest) {
        if (StringUtils.isBlank(verifyRequest.getSerialNo())) {
            verifyRequest.setSerialNo(UUIDUtils.uuid36());
        }
        LocalDate paperDrewDate = DateUtils.toLocalDate(verifyRequest.getPaperDrewDate());
        verifyRequest.setPaperDrewDate(DateUtils.format(paperDrewDate, "yyyyMMdd"));
        PurchaserVerifyRecordEntity target = new PurchaserVerifyRecordEntity();
        BeanUtils.copyObject(verifyRequest, target);
        target.setPaperDrewDate(paperDrewDate);
        target.setUserName(OAuth2UserContext.getPrincipal().getUsername());
        final Double amountWithoutTax = verifyRequest.getAmountWithoutTax();
        if (amountWithoutTax != null) {
            target.setAmount(amountWithoutTax.toString());
        }
        try {
            JsonResult<?> jResult = xPurchaserInvoiceService.invoiceVerify(verifyRequest);
            if (jResult.isSuccess()) {
                target.setTaskId(jResult.getData().toString());
                target.setVerifyStatus(VerifyStatusEnum.VERIFY_ING.getCode());
                return ViewResult.success();
            } else {
                target.setVerifyStatus(VerifyStatusEnum.VERIFY_FAIL.getCode());
                target.setVerifyMessage(jResult.getMessage());
                return ViewResult.failed(jResult.getMessage());
            }
        } catch (Exception e) {
            log.error("", e);
            target.setVerifyStatus(VerifyStatusEnum.VERIFY_FAIL.getCode());
            target.setVerifyMessage("系统异常，验真失败，原因:" + e.getMessage());
            return ViewResult.failed(target.getVerifyMessage());
        } finally {
            this.iPurchaserVerifyRecordService.saveOrUpdate(target);
        }
    }

    @Override
    public ViewResult<VerifyMain> queryResult(String taskId) {
        JsonResult<?> jrs = xPurchaserInvoiceService.queryVerifyResult(taskId);
        VerifyMain main = new VerifyMain();
        if (jrs.isSuccess()) {
            main = parseQueryResult(jrs.getData().toString());
        } else {
            main.setVeriStatus(VerifyStatusEnum.VERIFY_FAIL.getCode());
            main.setVeriRemark(jrs.getMessage());
        }
        return ViewResult.<VerifyMain>success().data(main);
    }

    private VerifyMain parseQueryResult(String result) {
        VerifyQueryResult verifyRsp = JacksonUtil.getInstance().fromJson(result, VerifyQueryResult.class);
        VerifyMain main = verifyRsp.getResult() != null ? verifyRsp.getResult().getInvoiceMain() : new VerifyMain();
        if (1 == verifyRsp.getCode()) {//验真成功
            main.setAuthRemark(main.getAuthRemark());
            // 0-验真中;1--验真成功;2--验真失败;3-验真请求发送失败
            if (main.getVeriStatus() == 0) {
                main.setVeriStatus(VerifyStatusEnum.VERIFY_ING.getCode());
            } else if (main.getVeriStatus() == 1) {
                main.setCipherText(main.getCipherText());
                main.setDetails(verifyRsp.getResult().getInvoiceDetails());
                verifyRsp.getResult().setInvoiceDetails(null);
                main.setVeriStatus(VerifyStatusEnum.VERIFY_SUCC.getCode());
            } else if (main.getVeriStatus() == 2) {
                main.setVeriRemark(main.getVeriRemark());
                main.setVeriStatus(VerifyStatusEnum.VERIFY_FAIL.getCode());
            } else if (main.getVeriStatus() == 3) {
                main.setVeriStatus(VerifyStatusEnum.VERIFY_FAIL.getCode());
                main.setVeriRemark(main.getVeriRemark());
            }
        } else if (-1 == verifyRsp.getCode()) {
            main.setVeriStatus(VerifyStatusEnum.VERIFY_FAIL.getCode());
            main.setVeriRemark(verifyRsp.getMessage());
        } else {
            main.setVeriStatus(VerifyStatusEnum.VERIFY_FAIL.getCode());
            main.setVeriRemark(verifyRsp.getMessage());
        }
        return main;
    }

    @Transactional
    public void saveOrUpdateInvoiceInfo(String taskId, VerifyMain verifyMain) {
        final List<PurchaserInvoiceMainEntity> purchaserInvoiceMains = this.iPurchaserInvoiceMainService.lambdaQuery()
                .eq(PurchaserInvoiceMainEntity::getInvoiceCode, verifyMain.getInvoiceCode())
                .eq(PurchaserInvoiceMainEntity::getInvoiceNo, verifyMain.getInvoiceNo())
                .list();
        PurchaserInvoiceMainEntity invoiceMain = new PurchaserInvoiceMainEntity();
        BeanUtils.copyObject(verifyMain, invoiceMain);
        if (purchaserInvoiceMains.isEmpty()) {
            this.iPurchaserInvoiceMainService.save(invoiceMain);
            if (!Collections.isEmpty(verifyMain.getDetails())) {
                List<PurchaserInvoiceDetailsEntity> details = new ArrayList<>();
                verifyMain.getDetails().forEach(item -> {
                    item.setInvoiceId(invoiceMain.getId());
                    BeanUtils.copyObject(item, PurchaserInvoiceDetailsEntity.class).ifPresent(details::add);
                });
                this.iPurchaserInvoiceDetailsService.saveBatch(details);
            }
        } else {
            invoiceMain.setId(purchaserInvoiceMains.get(0).getId());
            this.iPurchaserInvoiceMainService.updateById(invoiceMain);
        }
        //
        final List<PurchaserVerifyRecordEntity> verifyRecords = this.iPurchaserVerifyRecordService.lambdaQuery()
                .eq(PurchaserVerifyRecordEntity::getInvoiceCode, verifyMain.getInvoiceCode())
                .eq(PurchaserVerifyRecordEntity::getInvoiceNo, verifyMain.getInvoiceNo())
                .eq(PurchaserVerifyRecordEntity::getTaskId, taskId)
                .orderByDesc(PurchaserVerifyRecordEntity::getCreateTime)
                .list();
        if (!verifyRecords.isEmpty()) {
            PurchaserVerifyRecordEntity para = new PurchaserVerifyRecordEntity();
            BeanUtils.copyObject(verifyMain, para);
            para.setTaskId(taskId);
            para.setAmount(verifyMain.getAmountWithoutTax().toString());
            this.iPurchaserVerifyRecordService.updateById(para);
        }
    }

    /**
     * 接收发票验真结果接口
     *
     * @param janusCoreReceiveMsg
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public AjaxResult process(JanusCoreReceiveMsg<VerifyFeedback> janusCoreReceiveMsg) {
        VerifyFeedback rst = janusCoreReceiveMsg.getPayload();
        VerifyMain main = rst.getResult();
        if (1 == rst.getCode()) {//验真成功
            // 0-验真中;1--验真成功;2--验真失败;3-验真请求发送失败
            main.setAuthRemark(main.getAuthRemark());
            if (main.getVeriStatus() == 0) {
                main.setVeriStatus(VerifyStatusEnum.VERIFY_ING.getCode());
            } else if (main.getVeriStatus() == 1) {
                main.setCipherText(main.getCipherText());
                main.setVeriStatus(VerifyStatusEnum.VERIFY_SUCC.getCode());
            } else if (main.getVeriStatus() == 2) {
                main.setVeriRemark(main.getVeriRemark());
                main.setVeriStatus(VerifyStatusEnum.VERIFY_FAIL.getCode());
            } else if (main.getVeriStatus() == 3) {
                main.setVeriStatus(VerifyStatusEnum.VERIFY_FAIL.getCode());
                main.setVeriRemark(main.getVeriRemark());
            }
        } else if (-1 == rst.getCode()) {
            main.setVeriStatus(VerifyStatusEnum.VERIFY_FAIL.getCode());
        } else {
            main.setVeriStatus(VerifyStatusEnum.VERIFY_FAIL.getCode());
            main.setVeriRemark(rst.getMessage());
        }
        String taskId = this.getTaskId(janusCoreReceiveMsg.getSealedRecMessage());
        this.saveOrUpdateInvoiceInfo(taskId, main);
        return ViewResult.success();
    }

    public String getTaskId(SealedRecMessage sealedRecMessage) {
        Map<String, String> obj = JsonUtils.fromJson(sealedRecMessage.getPayload().getObj().toString(), Map.class);
        if ("1".equals(obj.get("code"))) {//验真成功
            Map<String, String> result = JsonUtils.fromJson(obj.get("result"), Map.class);
            return result.get("veriInvoiceId");
        } else {
            return sealedRecMessage.getHeader().getOthers().get("businessNo");
        }
    }

}
