package com.xforceplus.elephant.image.common;

import static com.xforceplus.elephant.image.core.business.consts.Constants.TENANT_CODE;

import com.alibaba.fastjson.JSONObject;
import com.xforceplus.elephant.basecommon.baseconst.ReleationConstants;
import com.xforceplus.elephant.basecommon.dispatch.BeanDispatcher;
import com.xforceplus.elephant.basecommon.enums.EntityConstant;
import com.xforceplus.elephant.basecommon.enums.common.WebsocketNoticeTypeEnum;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.core.business.config.queue.MQUtils;
import com.xforceplus.elephant.image.core.business.enums.MQEnum;
import com.xforceplus.elephant.image.core.domain.bill.BaseBillService;
import com.xforceplus.elephant.image.core.domain.common.TicketSupport;
import com.xforceplus.elephant.image.core.domain.image.ImageService;
import com.xforceplus.elephant.image.core.expand.BillImageTicketService;
import com.xforceplus.elephant.image.core.expand.bean.BillRecogBean;
import com.xforceplus.elephant.image.core.expand.bean.TicketRecogBean;
import com.xforceplus.elephant.image.core.expand.impl.billimageticket.DefaultBillImageTicketServiceImpl;
import com.xforceplus.elephant.image.core.facade.application.collect.ticket.TicketFacade;
import com.xforceplus.elephant.image.core.util.RequestBuilder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.BillDataStatus;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.ImageType;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.YesNo;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.BaseBill;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.Image;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.meta.EntityMeta;
import com.xforceplus.ultraman.metadata.domain.vo.dto.ConditionOp;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

@Slf4j
@Component
@RequiredArgsConstructor
public class ScanCommonService {

    private final TicketSupport ticketSupport;
    private final TicketFacade ticketFacade;
    private final ImageService imageService;
    private final BillImageTicketService billImageTicketService;
    private final BaseBillService baseBillService;
    private final BeanDispatcher beanDispatcher;
    private final MQUtils mqUtils;

    /**
     * 混扫通用逻辑
     *
     * @param images 影像列表
     * @return 影像id列表
     */
    public List<Long> saveImage(final List<JSONObject> images) {
        final List<Long> imageIds = imageService.saveForMap(images);
        //创建单据
        toBill(images);
        return imageIds;
    }

    /**
     * 创建单据
     *
     * @param images 影像列表
     */
    private void toBill(final List<JSONObject> images) {
        final List<JSONObject> billImages = images.stream()
            .filter(image -> ImageType._1.getCode().equals(MapUtils.getString(image, EntityMeta.Image.IMAGE_TYPE.code())))
            .collect(Collectors.toList());
        if (ValidatorUtil.isEmpty(billImages)) {
            return;
        }
        final Map<String, List<JSONObject>> billImageMap = images.stream().collect(Collectors.groupingBy(image -> image.getString(EntityMeta.Image.BILL_CODE.code())));
        billImages.forEach(image -> {
            //单据保存
            final BillRecogBean bean = new BillRecogBean();
            bean.setImageId(image.getLong(EntityMeta.Image.ID.code()));
            bean.setTenantId(image.getLong(EntityMeta.Image.TENANT_ID.code()));
            bean.setBatchNo(image.getString(EntityMeta.Image.BATCH_NO.code()));
            bean.setIsPublic(image.getString(EntityMeta.Image.IS_PUBLIC.code()));
            bean.setScanUserId(image.getLong(EntityMeta.Image.CREATE_USER_ID.code()));
            bean.setScanUserName(image.getString(EntityMeta.Image.CREATE_USER_NAME.code()));
            bean.setBillTypeCode(image.getString(EntityMeta.Image.BILL_ENTITY_CODE.code()));
            final JSONObject discern = wrapperRecogInfo(image);
            if (billImageMap.containsKey(image.getString(EntityMeta.Image.BILL_CODE.code()))) {
                discern.put("image_count", billImageMap.get(image.getString(EntityMeta.Image.BILL_CODE.code())).size());
            }
            discern.put(ReleationConstants.BILL_OTO_IMAGE_ID, image.getLong(EntityMeta.Image.ID.code()));
            bean.setRecogInfo(discern);
            log.info("发送单据，imageId:【{}】,bill_code：【{}】", image.getLong(EntityMeta.Image.ID.code()), image.getString(EntityMeta.Image.BILL_CODE.code()));
            final Long billId = billImageTicketService.insertOrUpdateBillRecogInfo(bean);
            if (billId != null && billId > 0) {
                final RequestBuilder builder = new RequestBuilder()
                    .field("tenant_id", ConditionOp.eq, image.getLong(EntityMeta.Image.TENANT_ID.code()))
                    .field("bill_code", ConditionOp.eq, image.getString(EntityMeta.Image.BILL_CODE.code()))
                    //历史影像无需重复更新
                    .field(ReleationConstants.BILL_OTM_IMAGE_ID, ConditionOp.nil);
                final JSONObject update = new JSONObject();
                update.put(ReleationConstants.BILL_OTM_IMAGE_ID, billId);
                imageService.updateByCondition(builder, update);
            }
            //重算层级
            billImageTicketService.resetLevel(image.getLong(EntityMeta.Image.TENANT_ID.code()), image.getString(EntityMeta.Image.BILL_CODE.code()));
            //发送通知
            final JSONObject mqMessage = new JSONObject();
            mqMessage.put("noticeType", WebsocketNoticeTypeEnum.IMAGE_DISCERN.getCode());
            mqMessage.put("tenantId", image.getLong(EntityMeta.Image.TENANT_ID.code()));
            mqMessage.put("source", "imageSaveBill");
            mqMessage.put("createUserId", image.getLong(EntityMeta.Image.CREATE_USER_ID.code()));
            mqMessage.put("billCode", image.getString(EntityMeta.Image.BILL_CODE.code()));
            mqMessage.put("isBill", true);
            final Map<String, Object> headers = new HashMap<>();
            headers.put(TENANT_CODE, image.getString(EntityMeta.Image.TENANT_CODE.code()));
            mqUtils.sendByTopicExchange(MQEnum.WEBSOCKET_NOTICE_QUEUE, mqMessage, headers);
        });
    }

    private JSONObject wrapperRecogInfo(final JSONObject image) {
        final JSONObject discern = new JSONObject();
        discern.put("bill_code", image.getString(EntityMeta.Image.BILL_CODE.code()));
        discern.put("create_user_code", image.getString(EntityMeta.Image.CREATE_USER_CODE.code()));
        discern.put("system_source", image.getString(EntityMeta.Image.IMAGE_SOURCE.code()));
        return discern;
    }

    /**
     * 影像补扫通用逻辑
     *
     * @param user        用户
     * @param bill        单据
     * @param parentImage 插入位置影像
     * @param images      补扫列表
     * @return 影像id列表
     */
    public List<Long> insertBillScan(IAuthorizedUser user, BaseBill bill, Image parentImage, List<JSONObject> images) {
        final List<JSONObject> billImages = new ArrayList<>();
        final List<Long> imageIds = new ArrayList<>();
        if (ValidatorUtil.isEmpty(parentImage)) {
            images.forEach(item -> {
                final List<Long> itemIds = imageService.saveForMap(Arrays.asList(item));
                if (ValidatorUtil.isEmpty(itemIds)) {
                    return;
                }
                if (ticketSupport.isAttachSalesImage(item.getString("image_type"))) {
                    billImages.add(item);
                }
                ticketFacade.saveTicketVirtual(Image.fromOQSMap(item), itemIds.get(0));
                imageIds.addAll(itemIds);
            });
        } else {
            images.forEach(item -> {
                final List<Long> itemIds = imageService.insertImagesForMap(parentImage, Arrays.asList(item));
                if (ValidatorUtil.isEmpty(itemIds)) {
                    return;
                }
                if (ticketSupport.isAttachSalesImage(item.getString("image_type"))) {
                    billImages.add(item);
                }
                ticketFacade.saveTicketVirtual(Image.fromOQSMap(item), itemIds.get(0));
                imageIds.addAll(itemIds);
            });
        }
        if (ValidatorUtil.isNotEmpty(billImages)) {
            //创建附件
            toAnnex(billImages);
            //重算层级
            billImageTicketService.resetLevel(user.getTenantId(), bill.getBillCode());
            //发送通知
            final JSONObject mqMessage = new JSONObject();
            mqMessage.put("noticeType", WebsocketNoticeTypeEnum.IMAGE_DISCERN.getCode());
            mqMessage.put("tenantId", user.getTenantId());
            mqMessage.put("source", "imageInsertAttach");
            mqMessage.put("createUserId", user.getId());
            mqMessage.put("billCode", bill.getBillCode());
            final Map<String, Object> headers = new HashMap<>();
            headers.put(TENANT_CODE, user.getTenantCode());
            mqUtils.sendByTopicExchange(MQEnum.WEBSOCKET_NOTICE_QUEUE, mqMessage, headers);
        }
        //上传成功，则修改单据状态为待处理
        if (BillDataStatus.fromCode(bill.getBillDataStatus()) == BillDataStatus._6) {
            final JSONObject updater = new JSONObject();
            updater.put(EntityMeta.BaseBill.BILL_DATA_STATUS.code(), BillDataStatus._0.getCode());
            baseBillService.updateByBillIdSelective(bill.getId(), updater);
        }
        return imageIds;
    }

    private void toAnnex(final List<JSONObject> images) {
        if (ValidatorUtil.isEmpty(images)) {
            return;
        }
        images.forEach(image -> {
            //附件保存
            final Long tenantId = image.getLong(EntityMeta.Image.TENANT_ID.code());
            final TicketRecogBean bean = new TicketRecogBean();
            bean.setImageId(image.getLong(EntityMeta.Image.ID.code()));
            bean.setTenantId(tenantId);
            bean.setBatchNo(image.getString(EntityMeta.Image.BATCH_NO.code()));
            bean.setIsPublic(image.getString(EntityMeta.Image.IS_PUBLIC.code()));
            bean.setScanUserId(image.getLong(EntityMeta.Image.CREATE_USER_ID.code()));
            bean.setScanUserName(image.getString(EntityMeta.Image.CREATE_USER_NAME.code()));
            final JSONObject discern = this.imageToTicket(image);
            final Map<String, Object> data = new HashMap<>();
            final String entityCode = StringUtils.firstNonBlank(image.getString(EntityMeta.Image.TICKET_CODE.code()), EntityMeta.TicketAttachment.code());
            data.put("entityCode", entityCode);
            data.put("recogJson", discern);
            bean.setRecogList(Arrays.asList(data));
            log.info("发送无需识别数据，imageId:【{}】,单据号：【{}】，所属单据：【{}】", image.getLong(EntityMeta.Image.ID.code()), image.getString(EntityMeta.Image.BILL_CODE.code()),
                image.getString(EntityMeta.Image.BILL_ENTITY_CODE.code()));
            beanDispatcher.dispatch(tenantId, DefaultBillImageTicketServiceImpl.class).insertOrUpdateRecogInfo(bean);
        });
    }

    /**
     * image字段字段转为ticket的字段
     *
     * @param image 扫描image对象
     * @return 识别结果对象
     */
    private JSONObject imageToTicket(JSONObject image) {
        final JSONObject discern = new JSONObject();
        //  增加ext1到ext10扩展字段更新
        discern.put(EntityMeta.Ticket.IMAGE_ID.code(), image.getString(EntityMeta.Image.ID.code()));
        discern.put(EntityMeta.Ticket.IMAGE_O_T_M_TICKET_ID.code(), image.getString(EntityMeta.Image.ID.code()));
        discern.put(EntityMeta.Ticket.BILL_CODE.code(), image.getString(EntityMeta.Image.BILL_CODE.code()));
        discern.put(EntityMeta.Ticket.BILL_ENTITY_CODE.code(), image.getString(EntityMeta.Image.BILL_ENTITY_CODE.code()));
        discern.put(EntityMeta.Ticket.BILL_TYPE_CODE.code(), image.getString(EntityMeta.Image.BILL_ENTITY_CODE.code()));
        discern.put(EntityMeta.Ticket.SERIAL_NUMBER.code(), image.getString(EntityMeta.Image.SERIAL_NUMBER.code()));
        discern.put(EntityMeta.Ticket.SYSTEM_ORIG.code(), image.getString(EntityMeta.Image.IMAGE_SOURCE.code()));
        discern.put(EntityMeta.Ticket.SYSTEM_SOURCE.code(), image.getString(EntityMeta.Image.IMAGE_SOURCE.code()));
        discern.put(EntityMeta.Ticket.CREATE_USER_CODE.code(), image.getString(EntityMeta.Image.CREATE_USER_CODE.code()));
        //  增加ext1到ext10扩展字段更新
        discern.put(EntityMeta.Ticket.EXT1.code(), image.getString(EntityMeta.Image.EXT1.code()));
        discern.put(EntityMeta.Ticket.EXT2.code(), image.getString(EntityMeta.Image.EXT2.code()));
        discern.put(EntityMeta.Ticket.EXT3.code(), image.getString(EntityMeta.Image.EXT3.code()));
        discern.put(EntityMeta.Ticket.EXT4.code(), image.getString(EntityMeta.Image.EXT4.code()));
        discern.put(EntityMeta.Ticket.EXT5.code(), image.getString(EntityMeta.Image.EXT5.code()));
        discern.put(EntityMeta.Ticket.EXT6.code(), image.getString(EntityMeta.Image.EXT6.code()));
        discern.put(EntityMeta.Ticket.EXT7.code(), image.getString(EntityMeta.Image.EXT7.code()));
        discern.put(EntityMeta.Ticket.EXT8.code(), image.getString(EntityMeta.Image.EXT8.code()));
        discern.put(EntityMeta.Ticket.EXT9.code(), image.getString(EntityMeta.Image.EXT9.code()));
        discern.put(EntityMeta.Ticket.EXT10.code(), image.getString(EntityMeta.Image.EXT10.code()));
        return discern;
    }

    /**
     * 个人单证池扫描保存发票
     *
     * @param images  影像
     * @param batchNo 批次号
     * @param userId  用户id
     */
    public List<Long> personSave(List<JSONObject> images, String batchNo, Long userId) {
        final List<Long> imageIds = imageService.saveForMap(images);
        // 不需要识别的特殊处理
        final List<JSONObject> notRecogImages = images.stream().filter(image -> YesNo._0.getCode().equals(image.getString(EntityMeta.Image.REQUIRE_OCR_FLAG.code()))).collect(Collectors.toList());
        toAnnex(notRecogImages);
        if (CollectionUtils.isNotEmpty(notRecogImages)) {
            billImageTicketService.resetLevel(batchNo, userId);
        }
        return imageIds;
    }

}
