package com.xforceplus.elephant.image.config;

import com.alibaba.fastjson.JSONException;
import com.xforceplus.elephant.basecommon.exception.TokenException;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.general.alarm.service.AlarmService;
import com.xforceplus.general.starter.logger.TraceContext;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.ValidationException;
import org.apache.catalina.connector.ClientAbortException;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.http.converter.HttpMessageNotWritableException;
import org.springframework.validation.ObjectError;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

/**
 * @author xforceplus
 */
@ResponseBody
@ControllerAdvice
public class ExceptionAdvice {

    private static final Logger logger = LoggerFactory.getLogger(ExceptionAdvice.class);

    @Resource
    private HttpServletRequest request;
    @Autowired
    private AlarmService alarmService;

    @ExceptionHandler(value = JSONException.class)
    public ResponseEntity<CommonResponse> handleJSONException(final JSONException ex, final HttpServletRequest httpRequest) {
        logger.error("传参异常", ex);
        httpRequest.setAttribute("api.request.attribute.controller.advice.exception", ex);
        return ResponseEntity.ok(CommonResponse.failed(ex.getMessage()));
    }

    @ExceptionHandler(value = HttpRequestMethodNotSupportedException.class)
    public ResponseEntity<CommonResponse> handleHttpRequestMethodNotSupportedException(final HttpRequestMethodNotSupportedException ex, final HttpServletRequest httpRequest) {
        logger.error("方法类型调用错误", ex);
        httpRequest.setAttribute("api.request.attribute.controller.advice.exception", ex);
        return ResponseEntity.ok(CommonResponse.failed(ex.getMessage()));
    }

    @ExceptionHandler(value = HttpMessageNotReadableException.class)
    public ResponseEntity<CommonResponse> handleHttpMessageNotReadableException(final HttpMessageNotReadableException ex, final HttpServletRequest httpRequest) {
        logger.error("传参格式异常", ex);
        httpRequest.setAttribute("api.request.attribute.controller.advice.exception", ex);
        return ResponseEntity.ok(CommonResponse.failed("传参格式异常，请检查后再试！"));
    }

    @ExceptionHandler(value = HttpMessageNotWritableException.class)
    public void handleHttpMessageNotWritableException(final HttpMessageNotWritableException ex, final HttpServletRequest httpRequest, HttpServletResponse httpServletResponse) {

        final String contentType = httpServletResponse.getContentType();
        logger.error("handleHttpMessageNotWritableException，contentType:" + contentType, ex);
        if (!StringUtils.containsAny(contentType, "javascript", "text/event-stream")) {
            handleException(ex, httpRequest);
        }

    }

    @ExceptionHandler(value = {TokenException.class})
    public ResponseEntity<CommonResponse> handleTokenException(final Exception ex, final HttpServletRequest httpRequest) {
        logger.error(ex.getMessage());
        httpRequest.setAttribute("api.request.attribute.controller.advice.exception", ex);
        return ResponseEntity.status(HttpStatus.UNAUTHORIZED).body(CommonResponse.failed(ex.getMessage()));
    }

    @ExceptionHandler(value = {ValidationException.class, MethodArgumentNotValidException.class})
    public ResponseEntity<CommonResponse> handleValidationException(final Exception ex, final HttpServletRequest httpRequest) {
        logger.error("请求验证失败：" + ex.getMessage());
        if (ex instanceof MethodArgumentNotValidException) {
            final MethodArgumentNotValidException methodArgumentNotValidException = (MethodArgumentNotValidException) ex;
            return ResponseEntity.ok(CommonResponse.failed(methodArgumentNotValidException.getBindingResult()
                                                                                          .getAllErrors()
                                                                                          .stream()
                                                                                          .map(ObjectError::getDefaultMessage)
                                                                                          .collect(Collectors.joining("; "))));
        }
        //httpRequest.setAttribute("api.request.attribute.controller.advice.exception", ex);
        return ResponseEntity.ok(CommonResponse.failed(ex.getMessage()));
    }

    @ExceptionHandler(value = RuntimeException.class)
    public ResponseEntity<CommonResponse> handleRuntimeException(final RuntimeException ex, final HttpServletRequest httpRequest) {
        final Throwable e = ex.getCause();

        if (e == null) {
            return handleException(ex, httpRequest);
        }

        return handleException((Exception) e, httpRequest);
    }

    @ExceptionHandler(ClientAbortException.class)
    public ResponseEntity<CommonResponse> handleClientAbortException(ClientAbortException ex) {
        final ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        final String url = attributes.getRequest() != null ? attributes.getRequest().getRequestURI() : "";
        final Map<String, String> params = request != null
            ? request.getParameterMap()
                     .entrySet().stream()
                     .collect(Collectors.toMap(entry -> entry.getKey(),
                         entry -> StringUtils.join(entry.getValue(), ",")))
            : java.util.Collections.emptyMap();
        logger.error("[handleClientAbortException]系统异常,url:{}，params:{} ", url, params);
        logger.error("[handleClientAbortException]系统异常 " + url, ex);
        return ResponseEntity.ok(CommonResponse.failed("请求中断，请重试！"));
    }

    @ExceptionHandler(value = Exception.class)
    public ResponseEntity<CommonResponse> handleException(final Exception ex, final HttpServletRequest httpRequest) {
        final String traceId = MDC.get("traceId");
        if (ex.getClass().getPackage().getName().startsWith("com.xforceplus")) {
            MDC.remove("traceId");
            return ResponseEntity.ok(CommonResponse.failed(ex.getMessage()));
        }

        final String url = httpRequest != null ? httpRequest.getRequestURI() : StringUtils.EMPTY;
        final Map<String, String> params = request != null
            ? request.getParameterMap()
                     .entrySet().stream()
                     .collect(Collectors
                         .toMap(entry -> entry.getKey(),
                             entry -> StringUtils.join(entry.getValue(), ",")))
            : java.util.Collections.emptyMap();

        logger.error("服务内部异常，url:{}，params:{}", url, params);
        logger.error("服务内部异常" + url, ex);
        final Object[] args = new Object[params.size() * 2 + 2];
        args[0] = "url";
        args[1] = url;
        int i = 1;
        for (Entry<String, String> p : params.entrySet()) {
            args[i * 2] = p.getKey();
            args[i * 2 + 1] = p.getValue();
            i++;
        }
        alarmService.alarm(TraceContext.getPlusTraceId(), ex, args);
        // 以约定的方式，把异常通过request的属性传递给xlog。因为是约定方式，和xlog没有强耦合，即使未来xlog被移除也不会有问题。
        httpRequest.setAttribute("api.request.attribute.controller.advice.exception", ex);
        MDC.remove("traceId");
        return ResponseEntity.ok(CommonResponse.failed("系统故障，请联系管理员！"));
    }

}
