package com.xforceplus.elephant.image.controller.billdata.process;

import com.google.api.client.util.Lists;
import com.google.common.collect.Maps;
import com.xforceplus.elephant.basecommon.enums.EntityConstant;
import com.xforceplus.elephant.basecommon.enums.bill.BackTypeEnum;
import com.xforceplus.elephant.basecommon.enums.cqp.bill.ExtStatusEnum;
import com.xforceplus.elephant.basecommon.help.StringHelp;
import com.xforceplus.elephant.basecommon.process.AbstractProcess;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.client.model.BackBaseBillRequest;
import com.xforceplus.elephant.image.core.business.config.queue.MQUtils;
import com.xforceplus.elephant.image.core.domain.bill.BaseBillService;
import com.xforceplus.elephant.image.core.domain.collaboration.CooperationService;
import com.xforceplus.elephant.image.core.domain.image.ImageService;
import com.xforceplus.elephant.image.core.domain.ticket.TicketService;
import com.xforceplus.elephant.image.core.expand.BillImageTicketService;
import com.xforceplus.elephant.image.core.expand.OpenapiService;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.BackType;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.BillDataStatus;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.YesNo;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.BaseBill;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.Ticket;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import javax.validation.ValidationException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class PendingBackBaseBillProcess extends AbstractProcess<BackBaseBillRequest, Boolean> {

    @Autowired
    private BillImageTicketService billImageTicketService;
    @Autowired
    private ImageService imageService;
    @Autowired
    private MQUtils rabbitmqUtils;
    @Autowired
    private BaseBillService baseBillService;
    @Autowired
    private TicketService ticketService;
    @Autowired
    private OpenapiService openapiService;
    @Autowired
    private CooperationService cooperationService;

    @Override
    protected void check(BackBaseBillRequest request) throws ValidationException {
        super.check(request);
        checkEmpty(request.getBillCodes(), "billCodes不能为空");
    }

    @Override
    protected CommonResponse<Boolean> process(BackBaseBillRequest request) throws RuntimeException {
        final IAuthorizedUser authorizedUser = UserInfoHolder.get();// 获取登录用户上下文
        if (null == authorizedUser) {
            throw new ValidationException("获取用户信息为空，请重新登陆");
        }
        final List<BaseBill> baseBillEntities = baseBillService.selectBillDataByBillCodes(authorizedUser.getTenantId(), request.getBillCodes());
        final List<String> continueBillCodes = Lists.newArrayList();
        final List<String> checkFailedBillCodes = Lists.newArrayList();

        baseBillEntities.forEach(baseBill -> {
            if (BillDataStatus._3.getCode().equals(baseBill.getBillDataStatus())) {
                continueBillCodes.add(baseBill.getBillCode());
            } else {
                checkFailedBillCodes.add(baseBill.getBillCode());
            }
        });

        final boolean backStatus = billImageTicketService.backBillInfoAndLog(continueBillCodes, BackType._0.getCode(), request.getBackReason(),
            request.getBackRemark(), authorizedUser.getId(), authorizedUser.getUsername(), authorizedUser.getTenantId());
        if (backStatus) {
            continueBillCodes.forEach(billCode -> {
                final BaseBill baseBill = baseBillService.selectBaseBillByCode(authorizedUser.getTenantId(), billCode);
                final List<Ticket> ticketEntities = ticketService.selectBaseTicketByBillCode(baseBill.getTenantId(),
                    baseBill.getBillCode());
                final List<Long> imageIds = ticketEntities.stream().map(Ticket::getImageId).collect(Collectors.toList());

                final Map<String, Object> paramMap = Maps.newHashMap();
                paramMap.put("back_status", YesNo._1.getCode());
                paramMap.put("back_time", System.currentTimeMillis());
                paramMap.put("back_type", BackTypeEnum.DEFAULT.getCode());
                paramMap.put("ticket_status", ExtStatusEnum.BACK.getCode());
                //更新Ticket
                updateTicket(paramMap, ticketEntities);
                if (ValidatorUtil.isNotEmpty(request.getBackRemark())) {
                    paramMap.put("back_remark", request.getBackRemark());
                }
                //推送pubsub
                cooperationService.reCheckBillTicketAsync(authorizedUser.getTenantId(), authorizedUser.getTenantCode(), baseBill.getBillTypeCode(), baseBill.getBillCode(), "service-单据退回");
                //推送发票pubsub
                imageIds.forEach(imageId -> {
                    cooperationService.syncTicket(authorizedUser.getTenantId(), authorizedUser.getTenantCode(), imageId, "service-单据退回");
                });
            });
            if (ValidatorUtil.isNotEmpty(checkFailedBillCodes)) {
                final String message = StringHelp.safeToString(continueBillCodes.size()) + "单退回成功，"
                    + StringHelp.safeToString(checkFailedBillCodes.size()) + "单退回失败，失败原因：单据不是”已锁定“状态";
                return CommonResponse.ok(message);
            }
            return CommonResponse.ok("成功", true);
        } else {
            return CommonResponse.failed("退回失败");
        }
    }

    private boolean updateTicket(Map<String, Object> paramMap, List<Ticket> ticketEntities) {
        if (paramMap.size() > 0) {
            return ticketEntities.stream().map(ticket ->
                                     ticketService.updateTicketByParam(ticket.getId(), EntityConstant.TICKET, paramMap))
                                 .filter(count -> count > 0).count() == ticketEntities.size();
        }
        return false;
    }

}
