package com.xforceplus.elephant.image.controller.billdata.process;

import com.xforceplus.elephant.basecommon.process.AbstractProcess;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.client.model.ReTransformBillsRequest;
import com.xforceplus.elephant.image.core.business.application.collect.task.enums.TaskTypeEnum;
import com.xforceplus.elephant.image.core.business.application.collect.task.service.TaskQueueService;
import com.xforceplus.elephant.image.core.business.domain.ProcessContext;
import com.xforceplus.elephant.image.core.domain.imagefile.ImageFileService;
import com.xforceplus.elephant.image.core.expand.impl.billimageticket.DefaultBillImageTicketServiceImpl;
import com.xforceplus.elephant.image.core.facade.base.BaseFacade;
import com.xforceplus.elephant.image.core.util.RequestBuilder;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.Image;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.ImageFile;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.meta.EntityMeta;
import com.xforceplus.ultraman.metadata.domain.vo.dto.ConditionOp;
import com.xforceplus.ultraman.metadata.domain.vo.dto.ConditionQueryRequest;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import javax.validation.ValidationException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;

@Slf4j
@Service
@RequiredArgsConstructor
public class ReTransformBillsProcess extends AbstractProcess<ReTransformBillsRequest, Boolean> {

    private final ImageFileService imageFileService;
    private final BaseFacade baseFacade;
    private final TaskQueueService taskQueueService;
    private final DefaultBillImageTicketServiceImpl billImageTicketService;

    @Override
    protected void check(ReTransformBillsRequest request) throws ValidationException {
        super.check(request);
        checkEmpty(request.getBillCodes(), "billCodes不能为空");
    }

    @Override
    protected CommonResponse<Boolean> process(ReTransformBillsRequest request) throws RuntimeException {
        final ConditionQueryRequest imageQueryRequest = new RequestBuilder()
            .field(EntityMeta.Image.BILL_CODE.code(), ConditionOp.in, request.getBillCodes())
            .field(EntityMeta.Image.FILE_ID.code(), ConditionOp.gt, 0)
            .build();
        final String imageCode = request.isCompare() ? EntityMeta.CompareImage.code() : EntityMeta.Image.code();
        final List<Image> images = baseFacade.findByCondition(imageCode, imageQueryRequest).getRows()
            .stream().map(Image::fromOQSMap).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(images)) {
            return CommonResponse.ok(String.format("勾选的%s条单据没有转换失败的文件.", request.getBillCodes().size()));
        }
        final Map<Long, String> fileBillCodeMap = images.stream().collect(Collectors.toMap(Image::getFileId, Image::getBillCode, (o1, o2) -> o2));
        final List<Long> fileIds = images.stream().map(Image::getFileId).distinct().collect(Collectors.toList());
        final List<ImageFile> imageFiles = imageFileService.selectFailedFileByIds(fileIds)
            .stream().filter(imageFile -> ValidatorUtil.isNotEmpty(imageFile.getTaskType()))
            .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(imageFiles)) {
            return CommonResponse.ok(String.format("勾选的%s条单据没有转换失败的文件。", request.getBillCodes().size()));
        }
        final List<Long> failedFileIds = imageFiles.stream().map(ImageFile::getId).collect(Collectors.toList());
        //重新发起
        imageFiles.forEach(imageFile -> {
            //个人单证池的文件挂机后不会更新billCode
            if (ValidatorUtil.isEmpty(imageFile.getBillCode()) && ValidatorUtil.isNotEmpty(fileBillCodeMap.get(imageFile.getId()))) {
                imageFile.setBillCode(fileBillCodeMap.get(imageFile.getId()));
            }
            imageFile.setId(null);
            final ProcessContext context = ProcessContext.context().put(imageFile);
            taskQueueService.initiate(TaskTypeEnum.fromType(imageFile.getTaskType()), context);
        });
        final List<Long> failedImageIds = images.stream()
            .filter(image -> failedFileIds.contains(image.getFileId()))
            .map(Image::getId).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(failedImageIds)) {
            //删除影像和结构数据
            billImageTicketService.deleteImageAndTicketByImageIds(images.get(0).getTenantId(), failedImageIds);
            //删除影像文件
            imageFileService.deleteImageFileByIds(failedFileIds);
        }
        return CommonResponse.ok(String.format("勾选的%s条单据，转换成功%s个文件！", request.getBillCodes().size(), failedFileIds.size()));
    }

}
