package com.xforceplus.elephant.image.controller.billdata.process.delete;

import com.google.api.client.util.Lists;
import com.xforceplus.elephant.basecommon.annotation.Dispatch;
import com.xforceplus.elephant.basecommon.help.StringHelp;
import com.xforceplus.elephant.basecommon.process.AbstractProcess;
import com.xforceplus.elephant.basecommon.process.response.CommonResponse;
import com.xforceplus.elephant.basecommon.vaildate.ValidatorUtil;
import com.xforceplus.elephant.image.client.model.BaseBillDeleteRequest;
import com.xforceplus.elephant.image.core.domain.bill.BaseBillService;
import com.xforceplus.elephant.image.core.expand.BillImageTicketService;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.dict.BillDataStatus;
import com.xforceplus.ultraman.app.imageservicesaas.metadata.entity.BaseBill;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import javax.validation.ValidationException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Dispatch(isDefault = true)
@Service
public class DefaultDeleteBillDataProcess extends AbstractProcess<BaseBillDeleteRequest, Boolean> {

    @Autowired
    private BillImageTicketService billImageTicketService;
    @Autowired
    private BaseBillService baseBillService;

    @Override
    protected void check(BaseBillDeleteRequest request) throws ValidationException {
        super.check(request);
        checkEmpty(request.getBillCodes(), "billCodes不能为空");
    }

    @Override
    protected CommonResponse<Boolean> process(BaseBillDeleteRequest request) throws RuntimeException {
        final IAuthorizedUser authorizedUser = UserInfoHolder.get();// 获取登录用户上下文
        if (null == authorizedUser) {
            throw new ValidationException("获取用户信息为空，请重新登陆");
        }
        final List<BaseBill> billEntityList = baseBillService.selectBillDataByBillCodes(authorizedUser.getTenantId(), request.getBillCodes());
        if (ValidatorUtil.isEmpty(billEntityList)) {
            return CommonResponse.failed("无记录删除");
        }
        // 判断单据下影像是否识别完
        final List<BaseBill> updateBillList = Lists.newArrayList();
        final List<String> cantDeletedBillCodes = Lists.newArrayList();
        billEntityList.stream().forEach(baseBill -> {
            if (Arrays.asList(BillDataStatus._3.getCode(), BillDataStatus._1.getCode()).contains(baseBill.getBillDataStatus())) {
                cantDeletedBillCodes.add(baseBill.getBillCode());
                return;
            }
            updateBillList.add(baseBill);
        });
        String message = "删除成功";
        if (ValidatorUtil.isEmpty(updateBillList)) {
            message = String.format("单据%s已锁定或已完成，不可删除", StringHelp.join(cantDeletedBillCodes, "、"));
            return CommonResponse.failed(message);
        }

        // 已退回删除单据需要先下发，后删除
        billImageTicketService.billReturnDeleteSend(authorizedUser.getTenantId(), updateBillList);

        final List<String> updateBillCodes = updateBillList.stream().map(BaseBill::getBillCode).collect(Collectors.toList());
        final boolean result = billImageTicketService.deleteBillData(authorizedUser.getTenantId(), updateBillCodes, false);
        if (result) {
            if (ValidatorUtil.isNotEmpty(cantDeletedBillCodes)) {
                message = String.format("单据%s删除成功；单据%s未完成识别或已锁定，不可删除",
                    StringHelp.join(updateBillCodes, "、"),
                    StringHelp.join(cantDeletedBillCodes, "、"));
            }
            return CommonResponse.ok(message, true);
        }
        return CommonResponse.failed("删除失败");
    }

}